package org.mule.weave.v2.runtime.core.operator.selectors

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.core.exception
import org.mule.weave.v2.core.exception.InvalidSelectionException
import org.mule.weave.v2.core.exception.KeyNotFoundException
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.NameType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.parser.location.WeaveLocation

import scala.util.Try

object ObjectKeyValueSelector {
  def value(location: WeaveLocation): Array[BinaryFunctionValue] =
    Array(
      new ObjectStringKeyValueSelectorOperator(location),
      new ObjectNameKeyValueOperator(location),
      new ObjectIndexKeyOperator(location),
      new ArrayObjectNameFilterOperator(location),
      new NullNameValueSelectorOperator(location))
}

class ObjectStringKeyValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = ObjectType
  override val R = StringType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): ObjectValue = {
    new ObjectNameKeyValueOperator(location).doExecute(leftValue, NameType.coerce(rightValue))
  }
}

class ObjectNameKeyValueOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = ObjectType
  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): ObjectValue = {
    val objectSeq = leftValue.evaluate
    val selectedValue = objectSeq.allKeyValuesOf(rightValue) match {
      case Some(v) => ObjectValue(v, this)
      case _       => throw InvalidSelectionException(new KeyNotFoundException(this, rightValue.evaluate))
    }
    selectedValue
  }
}

class ObjectIndexKeyOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = ObjectType
  override val R = NumberType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): ObjectValue = {
    val objectSeq = leftValue.evaluate
    val index: Int = rightValue.evaluate.toInt
    if (index < 0) {
      val materializedObject = objectSeq.toSeq()
      val maybePair = Try(materializedObject.apply(materializedObject.size + index)).toOption
      maybePair match {
        case Some(value) => ObjectValue(ObjectSeq(value))
        case None        => throw InvalidSelectionException(new exception.IndexOutOfBoundsException(this.location, index, objectSeq.size))
      }
    } else {
      val maybeKeyValuePair: KeyValuePair = objectSeq.apply(index)
      if (maybeKeyValuePair != null) {
        ObjectValue(ObjectSeq(maybeKeyValuePair))
      } else {
        throw InvalidSelectionException(new exception.IndexOutOfBoundsException(this.location, index, objectSeq.size))
      }
    }
  }
}

class ArrayObjectNameFilterOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = ArrayType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val objects = leftValue.evaluate.toIterator().filter((item) => ObjectType.accepts(item))
    val result = objects.flatMap((v: Value[_]) => {
      v.evaluate.asInstanceOf[ObjectSeq].allKeyValuesOf(rightValue)
    })
    if (result.nonEmpty) {
      val objectValues = result.map(ObjectValue(_, this))
      ArrayValue(objectValues, this)
    } else {
      throw InvalidSelectionException(new KeyNotFoundException(this, rightValue.evaluate))
    }
  }
}
