package org.mule.weave.v2.interpreted.transform

import org.mule.weave.v2.core.exception.InvalidNativeNameException
import org.mule.weave.v2.grammar.literals.TypeLiteral
import org.mule.weave.v2.interpreted.exception.InvalidUriException
import org.mule.weave.v2.interpreted.node.DefaultFunctionCallNode
import org.mule.weave.v2.interpreted.node.ExistsSelectorNode
import org.mule.weave.v2.interpreted.node.NullUnSafeNode
import org.mule.weave.v2.interpreted.node.TypeReferenceNode
import org.mule.weave.v2.interpreted.node.TypeReferenceWithTypeParamNode
import org.mule.weave.v2.interpreted.node.TypeSelectorReferenceNode
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.executors.DefaultFunctionCallExecutor
import org.mule.weave.v2.interpreted.node.structure.ArrayNode
import org.mule.weave.v2.interpreted.node.structure.ArrayTypeNode
import org.mule.weave.v2.interpreted.node.structure.AttributesNode
import org.mule.weave.v2.interpreted.node.structure.BooleanNode
import org.mule.weave.v2.interpreted.node.structure.ConditionalNode
import org.mule.weave.v2.interpreted.node.structure.DateTimeNode
import org.mule.weave.v2.interpreted.node.structure.DynamicArrayNode
import org.mule.weave.v2.interpreted.node.structure.DynamicKeyNode
import org.mule.weave.v2.interpreted.node.structure.DynamicNameNode
import org.mule.weave.v2.interpreted.node.structure.FilteredOverloadedFunctionNode
import org.mule.weave.v2.interpreted.node.structure.HeadTailArrayNode
import org.mule.weave.v2.interpreted.node.structure.HeadTailObjectNode
import org.mule.weave.v2.interpreted.node.structure.KeyNode
import org.mule.weave.v2.interpreted.node.structure.KeyValuePairNode
import org.mule.weave.v2.interpreted.node.structure.LiteralArrayNode
import org.mule.weave.v2.interpreted.node.structure.LiteralObjectNode
import org.mule.weave.v2.interpreted.node.structure.LocalDateNode
import org.mule.weave.v2.interpreted.node.structure.LocalDateTimeNode
import org.mule.weave.v2.interpreted.node.structure.LocalTimeNode
import org.mule.weave.v2.interpreted.node.structure.NameValuePairNode
import org.mule.weave.v2.interpreted.node.structure.NamespaceNode
import org.mule.weave.v2.interpreted.node.structure.NoFilter
import org.mule.weave.v2.interpreted.node.structure.NullNode
import org.mule.weave.v2.interpreted.node.structure.NumberNode
import org.mule.weave.v2.interpreted.node.structure.ObjectNode
import org.mule.weave.v2.interpreted.node.structure.OverloadedFunctionNode
import org.mule.weave.v2.interpreted.node.structure.PeriodNode
import org.mule.weave.v2.interpreted.node.structure.RFunctionParamTypeNode
import org.mule.weave.v2.interpreted.node.structure.RFunctionTypeNode
import org.mule.weave.v2.interpreted.node.structure.RKeyTypeNode
import org.mule.weave.v2.interpreted.node.structure.RKeyValuePairTypeNode
import org.mule.weave.v2.interpreted.node.structure.RLiteralTypeNode
import org.mule.weave.v2.interpreted.node.structure.RNameTypeNode
import org.mule.weave.v2.interpreted.node.structure.RNameValuePairTypeNode
import org.mule.weave.v2.interpreted.node.structure.RObjectTypeNode
import org.mule.weave.v2.interpreted.node.structure.RegexNode
import org.mule.weave.v2.interpreted.node.structure.TimeNode
import org.mule.weave.v2.interpreted.node.structure.TimeZoneNode
import org.mule.weave.v2.interpreted.node.structure.TypeNode
import org.mule.weave.v2.interpreted.node.structure.UriNode
import org.mule.weave.v2.interpreted.node.structure.VersionFilter
import org.mule.weave.v2.interpreted.node.structure.function.DynamicFunctionNode
import org.mule.weave.v2.interpreted.node.structure.function.FunctionParameterNode
import org.mule.weave.v2.interpreted.node.structure.function.InterceptedCheckFunctionDecoratorNode
import org.mule.weave.v2.interpreted.node.structure.function.LiteralFunctionValueNode
import org.mule.weave.v2.interpreted.node.structure.function.StaticFunctionNode
import org.mule.weave.v2.interpreted.node.structure.header.directives
import org.mule.weave.v2.interpreted.node.structure.schema.{ SchemaNode => RSchemaNode }
import org.mule.weave.v2.interpreted.node.structure.{ IntersectionTypeNode => RIntersectionTypeNode }
import org.mule.weave.v2.interpreted.node.structure.{ NameNode => RNameNode }
import org.mule.weave.v2.interpreted.node.structure.{ StringInterpolationNode => RStringInterpolationNode }
import org.mule.weave.v2.interpreted.node.structure.{ StringNode => RStringNode }
import org.mule.weave.v2.interpreted.node.structure.{ TypeParameterNode => RTypeParameterNode }
import org.mule.weave.v2.interpreted.node.structure.{ UnionTypeNode => RUnionTypeNode }
import org.mule.weave.v2.interpreted.node.{ NullSafeNode => RNullSafeNode }
import org.mule.weave.v2.interpreted.node.{ VariableReferenceNode => RVariableReferenceNode }
import org.mule.weave.v2.model.capabilities.UpdateLocationCapable
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.NativeValueManager
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.parser
import org.mule.weave.v2.parser.annotation.SinceAstNodeAnnotation
import org.mule.weave.v2.parser.ast
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNodeHelper.arg
import org.mule.weave.v2.parser.ast.functions
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.functions.FunctionParameter
import org.mule.weave.v2.parser.ast.header.directives.AnnotationDirectiveNode
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.StringInterpolationNode
import org.mule.weave.v2.parser.ast.types
import org.mule.weave.v2.parser.ast.types.ObjectTypeNode
import org.mule.weave.v2.parser.ast.types.TypeParameterNode
import org.mule.weave.v2.parser.ast.types._
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.exception.InvalidDateLiteralException
import org.mule.weave.v2.parser.exception.ParseException
import org.mule.weave.v2.parser.exception.WeaveRuntimeException
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.runtime.exception.CompilationExecutionException
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.scope.Reference
import org.mule.weave.v2.scope.VariableScope
import org.mule.weave.v2.ts.ScopeGraphTypeReferenceResolver
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.ts.{ AnyType => TSAnyType }
import org.mule.weave.v2.utils.StringEscapeHelper

import java.time._
import java.time.format.DateTimeParseException
import scala.util.matching.Regex

trait EngineStructureTransformations extends AstTransformation with EngineVariableTransformations {
  def transformDateTimeNode(dateTime: String)(implicit location: Location): DateTimeNode = {
    try {
      new DateTimeNode(ZonedDateTime.parse(dateTime))
    } catch {
      case dtpe: DateTimeParseException => throw new InvalidDateLiteralException(location, dateTime)
    }
  }

  def transformStringNode(v: parser.ast.structure.StringNode): RStringNode = {
    val quotedChar: Option[Char] = v.quotedBy()
    if (quotedChar.isDefined) {
      val unescaped = StringEscapeHelper.unescapeString(v.value, quotedChar.get, v.location())
      RStringNode(unescaped)
    } else {
      RStringNode(v.value)
    }
  }

  def transformTimeNode(time: String)(implicit location: Location): TimeNode = {
    try {
      new TimeNode(OffsetTime.parse(time))
    } catch {
      case dtpe: DateTimeParseException => throw new InvalidDateLiteralException(location, time)
    }
  }

  def transformStringInterpolationNode(v: Seq[AstNode]) = RStringInterpolationNode(transformSeq(v))

  def transformLocalDateTimeNode(dateTime: String)(implicit location: Location): LocalDateTimeNode = {
    try {
      new LocalDateTimeNode(LocalDateTime.parse(dateTime))
    } catch {
      case dtpe: DateTimeParseException => throw new InvalidDateLiteralException(location, dateTime)
    }
  }

  def transformUriNode(uri: String, uriNode: AstNode): UriNode = {
    try {
      UriNode(uri)
    } catch {
      case e: IllegalArgumentException => throw new InvalidUriException(uri, e.getMessage, uriNode.location())
    }
  }

  def transformConditionalNode(value: AstNode, cond: AstNode) = new ConditionalNode(transform(value), transform(cond))

  def transformPeriodNode(period: String)(implicit location: Location): PeriodNode = {
    try {
      if (period.contains('T')) {
        new PeriodNode(Duration.parse(period))
      } else {
        new PeriodNode(Period.parse(period))
      }
    } catch {
      case e: DateTimeParseException => throw new InvalidDateLiteralException(location, period)
    }
  }

  def transformRegexNode(regex: String): RegexNode = new RegexNode(new Regex(regex))

  val nativeTypeMap: Map[String, Type] = Map(
    TypeLiteral.STRING_TYPE_NAME -> StringType,
    TypeLiteral.BOOLEAN_TYPE_NAME -> BooleanType,
    TypeLiteral.NUMBER_TYPE_NAME -> NumberType,
    TypeLiteral.RANGE_TYPE_NAME -> RangeType,
    TypeLiteral.URI_TYPE_NAME -> UriType,
    TypeLiteral.DATETIME_TYPE_NAME -> DateTimeType,
    TypeLiteral.LOCALDATETIME_TYPE_NAME -> LocalDateTimeType,
    TypeLiteral.DATE_TYPE_NAME -> LocalDateType,
    TypeLiteral.LOCALTIME_TYPE_NAME -> LocalTimeType,
    TypeLiteral.TIME_TYPE_NAME -> TimeType,
    TypeLiteral.TIMEZONE_TYPE_NAME -> TimeZoneType,
    TypeLiteral.PERIOD_TYPE_NAME -> PeriodType,
    TypeLiteral.BINARY_TYPE_NAME -> BinaryType,
    TypeLiteral.NULL_TYPE_NAME -> NullType,
    TypeLiteral.ANY_TYPE_NAME -> AnyType,
    TypeLiteral.REGEX_TYPE_NAME -> RegexType,
    TypeLiteral.NOTHING_TYPE_NAME -> NothingType,
    TypeLiteral.NAME_SPACE_TYPE_NAME -> NamespaceType,
    TypeLiteral.ARRAY_TYPE_NAME -> ArrayType,
    TypeLiteral.TYPE_TYPE_NAME -> TypeType,
    TypeLiteral.OBJECT_TYPE_NAME -> ObjectType,
    TypeLiteral.KEY_TYPE_NAME -> KeyType,
    TypeLiteral.KEY_VALUE_PAIR_TYPE_NAME -> KeyValuePairType,
    TypeLiteral.ATTRIBUTES_TYPE_NAME -> AttributesType,
    TypeLiteral.NAME_TYPE_NAME -> NameType,
    TypeLiteral.NAME_VALUE_PAIR_TYPE_NAME -> NameValuePairType,
    TypeLiteral.NAME_SPACE_TYPE_NAME -> NamespaceType,
    TypeLiteral.SCHEMA_TYPE_NAME -> SchemaType,
    TypeLiteral.SCHEMA_PROPERTY_TYPE_NAME -> SchemaPropertyType,
    TypeLiteral.FUNCTION_TYPE_NAME -> FunctionType)

  def transformTypeDirective(variable: NameIdentifier, typeParametersListNode: Option[TypeParametersListNode], t: WeaveTypeNode, codeAnnotations: Seq[AnnotationNode])(implicit location: Location): directives.Directive = {
    if (typeParametersListNode.isEmpty || typeParametersListNode.get.typeParameters.isEmpty || nativeTypeMap.contains(variable.name)) {
      new directives.TypeDirective(transform(variable), transformTypeNode(t, ignoreTypeParameter = false, codeAnnotations))
    } else {
      val typeNode = TypeReferenceNode(NameIdentifier(TypeLiteral.TYPE_TYPE_NAME))
      new directives.TypeFunctionDirective(
        transform(variable),
        transformTypeNode(t, ignoreTypeParameter = false, codeAnnotations),
        typeParametersListNode.get.typeParameters
          .map((typeParam) => {
            new FunctionParameterNode(transformNameSlot(typeParam.name), transform(typeNode), None, false, false)
          })
          .toArray)
    }
  }

  def transformTypeNode(weaveType: WeaveTypeNode, ignoreTypeParameter: Boolean = true, typeDirectiveAnnotation: Seq[AnnotationNode] = Seq())(implicit location: Location): ValueNode[Type] = {
    val result = weaveType match {
      case ObjectTypeNode(properties, asSchema, asTypeSchema, close, _, _) => {
        if (ignoreTypeParameter) {
          //We should filter properties with type parameters in the key
          val filteredProperties = properties.collect({
            case kvt: KeyValueTypeNode if !isTypeReferenceNode(kvt.key) => kvt
          })
          new RObjectTypeNode(transformSeq(filteredProperties), !close, transformOption(asSchema), transformOption(asTypeSchema))
        } else {
          new RObjectTypeNode(transformSeq(properties), !close, transformOption(asSchema), transformOption(asTypeSchema))
        }
      }
      case KeyValueTypeNode(k, v, repeated, opt)                 => new RKeyValuePairTypeNode(transform(k), transform(v), opt, repeated)
      case NameValueTypeNode(k, v, opt)                          => new RNameValuePairTypeNode(transform(k), transform(v), opt)
      case KeyTypeNode(name, attrs, _, _, _)                     => new RKeyTypeNode(transform(name), transformSeq(attrs))
      case NameTypeNode(Some(name), nsMaybe, maybeTypeSchema, _) => new RNameTypeNode(name, transformOption(nsMaybe), transformOption(maybeTypeSchema))
      case NameTypeNode(None, None, maybeTypeSchema, _)          => new TypeNode(NameType, None, transformOption(maybeTypeSchema))
      case FunctionTypeNode(params, returnWeaveType, asSchema, asTypeSchema, _) => {
        val paramTypeNodes: Seq[RFunctionParamTypeNode] = params.map((param) => {
          val valueType: ValueNode[Type] = transform(param.valueType)
          new RFunctionParamTypeNode(valueType, param.name.map(_.name), param.optional)
        })
        val returnTypeNode: ValueNode[Type] = transform(returnWeaveType)
        new RFunctionTypeNode(paramTypeNodes, returnTypeNode, transformOption(asSchema), transformOption(asTypeSchema))
      }
      case UnionTypeNode(elems, asSchema, asTypeSchema, _) => {
        val seq = elems.map(transformTypeNode(_))
        new RUnionTypeNode(seq, transformOption(asSchema), transformOption(asTypeSchema))
      }
      case IntersectionTypeNode(elems, asSchema, asTypeSchema, _) => {
        val nodes = elems.map(transformTypeNode(_))
        new RIntersectionTypeNode(nodes, transformOption(asSchema), transformOption(asTypeSchema))
      }
      case NativeTypeNode(typeId, asSchema) => new TypeNode(nativeTypeMap(typeId), transformOption(asSchema))
      case LiteralTypeNode(valueNode, asSchema, asTypeSchema, _) => {
        valueNode match {
          case sn @ ast.structure.StringNode(strValue, _) => new TypeNode(new StringType(Some(StringEscapeHelper.unescapeString(strValue, sn.quotedBy().getOrElse('"')))), transformOption(asSchema), transformOption(asTypeSchema))
          case ast.structure.NumberNode(numValue, _)      => new TypeNode(new NumberType(Some(Number(valueNode.location(), numValue))), transformOption(asSchema), transformOption(asTypeSchema))
          case ast.structure.BooleanNode(booValue, _)     => new TypeNode(new BooleanType(Some(booValue.toBoolean)), transformOption(asSchema), transformOption(asTypeSchema)) //TODO: Check toBoolean
        }
      }
      case TypeParameterNode(_, base) => {
        base match {
          case None        => new TypeNode(AnyType, None)
          case Some(value) => new RTypeParameterNode(transformTypeNode(value))
        }
      }
      case trn: types.TypeReferenceNode => {
        transformTypeReferenceNode(trn, ignoreTypeParameter)
      }
      case trns: TypeSelectorNode => {
        transformTypeSelectorReferenceNode(trns, ignoreTypeParameter)
      }
    }
    result._location = Some(weaveType.location())
    result
  }

  def isTypeReferenceNode(weaveTypeNode: WeaveTypeNode): Boolean = {
    weaveTypeNode.isInstanceOf[types.TypeReferenceNode] && asTypeParameterRef(weaveTypeNode.asInstanceOf[types.TypeReferenceNode]).nonEmpty
  }

  def transformTypeReferenceNode(trn: ast.types.TypeReferenceNode, ignoreTypeParameter: Boolean = true)(implicit location: Location): ValueNode[Type] = {
    if (nativeTypeMap.contains(trn.variable.name)) {
      if (trn.variable.name.equals("Array") && trn.typeArguments.isDefined && trn.typeArguments.get.size == 1) {
        val option: Option[RSchemaNode] = transformOption(trn.asSchema)
        new ArrayTypeNode(transform(trn.typeArguments.get.head), option, transformOption(trn.asTypeSchema))
      } else {
        new TypeNode(nativeTypeMap(trn.variable.name), transformOption(trn.asSchema), transformOption(trn.asTypeSchema))
      }
    } else {
      val typeParameterReference: Option[TypeParameterNode] = asTypeParameterRef(trn)
      if (ignoreTypeParameter && typeParameterReference.isDefined) {
        transformTypeNode(typeParameterReference.get)
      } else if (trn.typeArguments.isEmpty || trn.typeArguments.get.isEmpty) {
        val reference: Reference = scopeNavigator().resolveVariable(trn.variable).get
        new TypeReferenceNode(transformReference(reference), transformOption(trn.asSchema), transformOption(trn.asTypeSchema))
      } else {
        val reference: Reference = scopeNavigator().resolveVariable(trn.variable).get
        val functionExecutor: DefaultFunctionCallExecutor = new DefaultFunctionCallExecutor(RVariableReferenceNode(transformReference(reference)), false, trn.location())
        val node = new TypeReferenceWithTypeParamNode(
          trn.variable.name,
          new DefaultFunctionCallNode(functionExecutor, trn.typeArguments.get.map(transformTypeNode(_)).toArray),
          transformOption(trn.asSchema),
          transformOption(trn.asTypeSchema))
        node.asInstanceOf[ValueNode[Type]]
      }
    }
  }

  private def transformTypeSelectorReferenceNode(trns: TypeSelectorNode, ignoreTypeParameter: Boolean)(implicit location: Location): ValueNode[Type] = {
    trns.selector match {
      case nameNode @ NameNode(StringInterpolationNode(_), _, _) => throw new WeaveRuntimeException(s"Type selection using string interpolation is not allowed", nameNode.location());
      case _ => new TypeSelectorReferenceNode(transformTypeNode(trns.weaveTypeNode, ignoreTypeParameter), transform(trns.selector), transformOption(trns.asSchema), transformOption(trns.asTypeSchema))
    }
  }

  private def asTypeParameterRef(trn: types.TypeReferenceNode): Option[TypeParameterNode] = {
    val maybeReference = scopeNavigator().resolveVariable(trn.variable)
    maybeReference match {
      case Some(reference) => {
        if (reference.scope.astNode.isInstanceOf[FunctionNode]) {
          val maybeParent = reference.scope.astNavigator().parentOf(reference.referencedNode)
          val isTypeParameterNode = maybeParent.isDefined && maybeParent.get.isInstanceOf[TypeParameterNode]
          if (isTypeParameterNode) {
            maybeParent.asInstanceOf[Option[TypeParameterNode]]
          } else None
        } else {
          None
        }
      }
      case None => {
        throw new WeaveRuntimeException(s"Unable to resolve reference of Type Parameter ${trn.variable}", trn.location())
      }
    }

  }

  def transformTimeZoneNode(timeZone: String)(implicit location: Location): TimeZoneNode = {
    new TimeZoneNode(timeZone)
  }

  def transformLocalDateNode(value: String)(implicit location: Location): LocalDateNode = {
    try {
      val localDate: LocalDate = DateHelper.parseLocalDate(value)
      new LocalDateNode(localDate)
    } catch {
      case dtpe: DateTimeParseException => throw new InvalidDateLiteralException(location, value)
    }

  }

  def transformNullNode(): NullNode = new NullNode()

  def transformArrayNode(v: ast.structure.ArrayNode): ValueNode[ArraySeq] = {
    if (AstNodeHelper.isArrayLiteral(v, scopeNavigator())) {
      new LiteralArrayNode(transformSeq(v.elements).toArray)
    } else if (AstNodeHelper.containsConditionalElements(v)) {
      new DynamicArrayNode(transformSeq(v.elements))
    } else {
      new ArrayNode(transformSeq(v.elements).toArray)
    }
  }

  def transformHeadTailArrayNode(head: AstNode, tail: AstNode): HeadTailArrayNode = new HeadTailArrayNode(transform(head), transform(tail))

  def transformHeadTailObjectNode(headKey: AstNode, headValue: AstNode, tail: AstNode): HeadTailObjectNode = new HeadTailObjectNode(transform(headKey), transform(headValue), transform(tail))

  def transformAttributesNode(attrs: Seq[AstNode]): AttributesNode = new AttributesNode(transformSeq(attrs))

  def transformObjectNode(objectNode: ast.structure.ObjectNode): ValueNode[ObjectSeq] = {
    if (AstNodeHelper.isObjectLiteral(objectNode, scopeNavigator())) {
      val nodes: Array[KeyValuePairNode] = transformSeq(objectNode.elements).toArray
      new LiteralObjectNode(nodes)
    } else {
      ObjectNode(transformSeq(objectNode.elements))
    }
  }

  def transformFunctionNode(args: Seq[functions.FunctionParameter], body: AstNode, returnType: Option[WeaveTypeNode], typeParametersListNode: Option[TypeParametersListNode], name: Option[String] = None): ValueNode[_] = {
    if (AstNodeHelper.isNativeCall(body, scopeNavigator())) {
      val value = AstNodeHelper.getNativeIdentifierCall(body).getOrElse(throw new InvalidNativeNameException("Native name must be a literal string.", body.location()))
      val functionValue: FunctionValue = NativeValueManager.getFunctionValue(value)
      functionValue.defaultName = name
      functionValue match {
        case ulc: UpdateLocationCapable => {
          ulc.updateLocation(body.location())
        }
        case _ =>
      }
      new LiteralFunctionValueNode(functionValue)
    } else {
      val bodyValue: ValueNode[_] = transform(body)
      createFunctionNode(args, body, returnType, name, bodyValue)
    }
  }

  def createFunctionNode(args: Seq[FunctionParameter], body: AstNode, returnType: Option[WeaveTypeNode], name: Option[String], bodyValue: ValueNode[_]): ValueNode[_] = {
    val navigator = scopeNavigator().scopeOf(body).get.astNavigator()
    val requiresMaterialize = args.exists((arg) => arg.wtype.isDefined && TypeHelper.requiredMaterialize(WeaveType(arg.wtype.get, scopeNavigator().referenceResolver)))
    val returnTypeNode: Option[ValueNode[Type]] = if (returnType.isDefined && !returnType.get.isInstanceOf[DynamicReturnTypeNode]) Some(transform(returnType.get)) else None
    if (transformingModule && navigator.granGranParentOf(body).exists(_.isInstanceOf[ModuleNode])) {
      //Module Functions are always static
      new StaticFunctionNode(transformSeq(args).toArray, bodyValue, returnTypeNode, requiresMaterializedArguments = requiresMaterialize, name)
    } else {
      //We should be able to determine if a function is static or dynamic if it is self contained is static otherwise is dynamic
      //To determine if it self contain we should check that the body and params only reference things from inside the scope of the function
      val dynamicParameter = args.exists((fp) => {
        //Is Dynamic Parameter
        if (fp.defaultValue.isDefined) {
          !AstNodeHelper.isLiteral(fp.defaultValue.get)
        } else {
          false
        }
      })

      new DynamicFunctionNode(transformSeq(args).toArray, bodyValue, returnTypeNode, requiresMaterializedArguments = requiresMaterialize, name = name, literalParameters = !dynamicParameter)
    }
  }

  def applyInterceptor(functionValue: ValueNode[_], functionName: NameIdentifier, codeAnnotations: Seq[AnnotationNode]): ValueNode[_] = {
    val interceptors: Seq[(AnnotationNode, ValueNode[Any])] =
      codeAnnotations.flatMap((annotation) => {
        toInterceptor(annotation)
      })
    if (interceptors.nonEmpty) {
      val interceptorFunction: ValueNode[_] = interceptors.foldLeft[ValueNode[_]](functionValue)((functionValue, annotation) => {
        val annotationInterceptorFunction: ValueNode[Any] = annotation._2
        val args: Seq[KeyValuePairNode] = annotation._1.args
          .map((args) =>
            args.args.map((arg) => {
              val argValue: ValueNode[_] = transform(arg.value)
              new KeyValuePairNode(new KeyNode(RStringNode(arg.name.name), None, None), argValue, None, false)
            }))
          .getOrElse(Seq.empty)
        val decoratorNode = new InterceptedCheckFunctionDecoratorNode(functionValue, annotationInterceptorFunction, new ObjectNode(args.toArray), StringValue(functionName.name))
        decoratorNode._location = annotation._1._location
        decoratorNode
      })
      interceptorFunction
    } else {
      functionValue
    }

  }

  private def toInterceptor(annotation: AnnotationNode): Option[(AnnotationNode, ValueNode[Any])] = {
    val maybeAnnotationReference: Option[Reference] = scopeNavigator().resolveVariable(annotation.name)
    maybeAnnotationReference match {
      case Some(ref) => {
        val annotationRefScope: VariableScope = ref.scope
        val annotationAstNavigator: AstNavigator = annotationRefScope.astNavigator()
        val maybeAnnotationDirectiveNode = annotationAstNavigator.parentWithType(ref.referencedNode, classOf[AnnotationDirectiveNode])
        maybeAnnotationDirectiveNode match {
          case Some(interceptorAnnotation) => {
            //We detected that the annotation is an InterceptorAnnotation
            val mayBeInterceptorAnnotation: Option[AnnotationNode] = interceptorAnnotation.codeAnnotations.find((ann) => {
              val resolvedAnnotationRef: Option[Reference] = annotationRefScope.resolveVariable(ann.name)
              resolvedAnnotationRef.exists((ref) => ref.fqnReferenceName == NameIdentifier.INTERCEPTOR_ANNOTATION)
            })
            if (mayBeInterceptorAnnotation.isDefined) {
              val interceptorAnnotation = mayBeInterceptorAnnotation.get
              val interceptorFunctionRef: AstNode = arg("interceptorFunction", mayBeInterceptorAnnotation.get)
                .getOrElse(throw new CompilationExecutionException(interceptorAnnotation.location(), "Missing required argument `interceptorFunction`"))

              val interceptorFunction: ValueNode[Any] = interceptorFunctionRef match {
                case functionName: LiteralValueAstNode => {
                  val nativeInterceptorName: NameIdentifier = NameIdentifier(functionName.literalValue.substring("@native".length).trim)
                  val nativeFunctionInterceptor: FunctionValue = NativeValueManager.getFunctionValue(nativeInterceptorName)
                  nativeFunctionInterceptor.defaultName = Some(nativeInterceptorName.name)
                  nativeFunctionInterceptor match {
                    case ulc: UpdateLocationCapable => {
                      ulc.updateLocation(functionName.location())
                    }
                    case _ =>
                  }
                  val functionValueNode: LiteralFunctionValueNode = new LiteralFunctionValueNode(nativeFunctionInterceptor)
                  functionValueNode._location = functionName._location
                  functionValueNode
                }
                case vrn: ast.variables.VariableReferenceNode => {
                  val maybeReference: Option[Reference] = annotationRefScope.resolveVariable(vrn.variable)
                  val functionReference: Reference = maybeReference.getOrElse(throw new CompilationExecutionException(interceptorAnnotation.location(), s"Unable to resolve reference to `${vrn.variable}`"))
                  val isLocalReference: Boolean = functionReference.fqnReferenceName.parent().contains(parsingContext().nameIdentifier)

                  val ref = if (isLocalReference) {
                    functionReference
                  } else {
                    Reference(functionReference.referencedNode, functionReference.scope, functionReference.fqnReferenceName.parent())
                  }
                  val referenceNode = RVariableReferenceNode(transformReference(ref))
                  referenceNode._location = vrn._location
                  referenceNode
                }
                case astNode: AstNode => {
                  transform(astNode)
                }
              }

              Some((annotation, interceptorFunction))
            } else {
              None
            }
          }
          case None => None
        }
      }
      case None => None
    }
  }

  def transformOverloadedFunctionNode(functionDirectives: Seq[ast.header.directives.FunctionDirectiveNode], name: Option[String] = None): ValueNode[_] = {
    val functionNodes: Seq[ValueNode[_]] =
      functionDirectives.map((fdn) => {
        applyInterceptor(transform[ValueNode[_]](fdn.literal), fdn.variable, fdn.codeAnnotations)
      })

    val navigator = scopeNavigator()
    val functionParamTypes: Seq[(Seq[WeaveType], Int)] =
      functionDirectives
        .map(_.literal)
        .collect({
          case fn: functions.FunctionNode => fn
        })
        .map((fn) => {
          fn.params.paramList.map(param => {
            param.wtype.map((pt) => WeaveType(pt, new ScopeGraphTypeReferenceResolver(navigator))).getOrElse(TSAnyType())
          })
        })
        .zipWithIndex

    val notCacheable = functionParamTypes.exists((paramTypesWithIndex) => {
      functionParamTypes.exists((otherTypes) => {
        val actualParams = paramTypesWithIndex._1
        val otherParams = otherTypes._1
        if (otherTypes._2 != paramTypesWithIndex._2 && otherParams.size == actualParams.size) {
          actualParams.zipWithIndex.forall((param) => {
            val paramIndex = param._2
            val paramType = param._1
            TypeHelper.canBeSubstitutedWithErasure(paramType, otherParams(paramIndex), new WeaveTypeResolutionContext(null))
          })
        } else {
          false
        }
      })
    })
    val hasASince = functionDirectives.exists((fd) => {
      fd.isAnnotatedWith(classOf[SinceAstNodeAnnotation])
    })
    if (hasASince) {
      val functionFilters = functionDirectives.map((fd) => {
        val maybeSinceAnnotation = fd.annotation(classOf[SinceAstNodeAnnotation])
        if (maybeSinceAnnotation.isDefined) {
          new VersionFilter(maybeSinceAnnotation.get.version)
        } else {
          NoFilter
        }
      })
      new FilteredOverloadedFunctionNode(functionNodes.toArray, functionFilters.toArray, name, !notCacheable)
    } else {
      new OverloadedFunctionNode(functionNodes.toArray, name, !notCacheable)
    }
  }

  def transformNameValuePairNode(key: AstNode, value: AstNode, cond: Option[AstNode]): NameValuePairNode =
    new NameValuePairNode(transform(key), transform(value), transformOption(cond))

  def transformNamespaceNode(prefix: NameIdentifier): NamespaceNode = {
    val resolveVariable = scopeNavigator().resolveVariable(prefix)
    if (resolveVariable.isEmpty) {
      throw new ParseException(s"Unable to resolve namespace ${prefix.name}", prefix.location())
    }
    val reference = resolveVariable.get
    NamespaceNode(transformReference(reference))
  }

  def transformFunctionParameter(nameIdentifier: NameIdentifier, defaultValue: Option[AstNode], maybeWtype: Option[WeaveTypeNode], codeAnnotations: Seq[AnnotationNode]): FunctionParameterNode = {
    val typeNode = maybeWtype.getOrElse(TypeReferenceNode(NameIdentifier(TypeLiteral.ANY_TYPE_NAME)))
    val materialize: Boolean = needsMaterialization(nameIdentifier)
    val isDesignOnly = codeAnnotations.exists((annotation) => annotation.name.name == "DesignOnlyType")
    val parameterType: ValueNode[Type] = if (isDesignOnly) {
      //As is design only we just put ANY
      transform(TypeReferenceNode(NameIdentifier(TypeLiteral.ANY_TYPE_NAME)))
    } else {
      transform(typeNode)
    }
    val paramTypeRequiresMaterialize = if (isDesignOnly) false else TypeHelper.requiredMaterialize(WeaveType(typeNode, scopeNavigator().referenceResolver))
    val node = new RLiteralTypeNode(parameterType)
    new FunctionParameterNode(transform(nameIdentifier), node, transformOption(defaultValue), materialize, paramTypeRequiresMaterialize)
  }

  def transformLocalTimeNode(time: String)(implicit location: Location): LocalTimeNode = {
    try {
      new LocalTimeNode(LocalTime.parse(time))
    } catch {
      case dtpe: DateTimeParseException => throw new InvalidDateLiteralException(location, time)
    }
  }

  def transformKeyValuePairNode(key: AstNode, value: AstNode, cond: Option[AstNode]): KeyValuePairNode = {
    val isLiteralKeyValueNode = AstNodeHelper.isLiteral(value) && AstNodeHelper.isLiteralKey(key, scopeNavigator())
    new KeyValuePairNode(transform(key), transform(value), transformOption(cond), isLiteralKeyValueNode)
  }

  def transformNameNode(keyName: AstNode, cond: Option[AstNode]): ValueNode[QualifiedName] = RNameNode(transform(keyName), transformOption(cond))

  def transformDynamicNameNode(keyName: AstNode): DynamicNameNode = new DynamicNameNode(transform(keyName))

  def transformNumberNode(location: Location, v: String): NumberNode = {
    new NumberNode(Number(location, v))
  }

  def transformBooleanNode(value: String): BooleanNode =
    new BooleanNode(value.toBoolean)

  def transformKeyNode(key: parser.ast.structure.KeyNode): ValueNode[QualifiedName] = {
    val literalKey = AstNodeHelper.isLiteralKey(key, scopeNavigator())
    if (literalKey) {
      KeyNode.literalKeyNode(transform(key.keyName), transformOption(key.ns), transformOption(key.attr))
    } else {
      if (AstNodeHelper.isLiteralNamespace(key.ns, scopeNavigator()) && AstNodeHelper.isLiteral(key.keyName)) {
        KeyNode.literalNameKeyNode(transform(key.keyName), transformOption(key.ns), transformOption(key.attr))
      } else {
        KeyNode(transform(key.keyName), transformOption(key.ns), transformOption(key.attr))
      }
    }
  }

  def transformDynamicKeyNode(keyName: AstNode, attr: Option[AstNode]): DynamicKeyNode = {
    KeyNode(transform(keyName), transformOption(attr))
  }

  def transformNullSafeNode(selector: AstNode): ValueNode[_] = {
    selector match {
      case variableReferenceNode: ast.variables.VariableReferenceNode => {
        val value: ValueNode[_] = transform(variableReferenceNode)
        value
      }
      case _ => RNullSafeNode(transform(selector))
    }
  }

  def transformNullUnSafeNode(selector: AstNode): NullUnSafeNode =
    NullUnSafeNode(transform(selector))

  def transformExistsSelectorNode(selectable: AstNode): ExistsSelectorNode =
    new ExistsSelectorNode(transform(selectable))
}
