package org.mule.weave.v2.runtime.core.functions.collections

import org.mule.weave.v2.api.tooling.message.Message
import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.runtime.core.exception.InvalidComparisonException

object ArrayOrderByFunctionValue extends BinaryFunctionValue {
  val L: ArrayType = ArrayType

  val R: FunctionType = FunctionType

  override def doExecute(lhs: L.V, rhs: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    var i = 0
    val elementToCompare: Array[ValueComparison] =
      lhs.evaluate
        .toIterator()
        .map(v => {
          val theValue = v.materialize
          i = i + 1
          ValueComparison(theValue, i, rhs)
        })
        .toArray

    java.util.Arrays.sort(
      elementToCompare,
      (o1: ValueComparison, o2: ValueComparison) => {
        val value1 = o1.comparator
        val value2 = o2.comparator
        ValueTypeChecker.checkTypes(value1.valueType.baseType, value2.valueType.baseType, location())
        value1.compareTo(value2).toInt
      })

    val finalValues = new Array[Value[_]](elementToCompare.length)
    var e = 0
    while (e < elementToCompare.length) {
      finalValues.update(e, elementToCompare(e).value)
      e = e + 1
    }
    ArrayValue(ArraySeq(finalValues), this)
  }

  private case class ValueComparison(value: Value[_], index: Int, fn: FunctionType.V) {
    def comparator(implicit ctx: EvaluationContext): Value[_] = {
      val result = fn.call(value, NumberValue(index))
      result
    }
  }
}

object ObjectOrderByFunctionValue extends BinaryFunctionValue {

  val L: ObjectType = ObjectType

  val R: FunctionType = FunctionType

  override def doExecute(lhs: L.V, rhs: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val elementToCompare: Array[KVComparison] =
      lhs.evaluate
        .toIterator()
        .map(keyValuePair => {
          KVComparison(keyValuePair.materialize, rhs)
        })
        .toArray

    java.util.Arrays.sort(
      elementToCompare,
      (o1: KVComparison, o2: KVComparison) => {
        val value1 = o1.comparator
        val value2 = o2.comparator
        ValueTypeChecker.checkTypes(value1.valueType.baseType, value2.valueType.baseType, location())
        value1.compareTo(value2).toInt
      })

    val finalValues = new Array[KeyValuePair](elementToCompare.length)
    var e = 0
    while (e < elementToCompare.length) {
      finalValues.update(e, elementToCompare(e).keyValuePair)
      e = e + 1
    }
    ObjectValue(ObjectSeq(finalValues), this)
  }

  private case class KVComparison(keyValuePair: KeyValuePair, fn: FunctionType.V) {
    def comparator(implicit ctx: EvaluationContext): Value[_] = {
      val result = fn.call(AttributeDelegateValue(keyValuePair._2, keyValuePair._1), keyValuePair._1)
      result
    }
  }
}

object ValueTypeChecker {
  def checkTypes(baseType1: Type, baseType2: Type, location: Location)(implicit ctx: EvaluationContext): Unit = {
    if (!baseType1.eq(baseType2)) {
      if (ctx.serviceManager.settingsService.execution().stableOrderBy) {
        throw new InvalidComparisonException(baseType1, baseType2, location)
      } else {
        ctx.serviceManager.messageLoggingService.logWarnDebounce(MixedTypesComparison(baseType1, baseType2))
      }
    }
  }
}

case class MixedTypesComparison(type1: Type, type2: Type) extends Message {

  override def getKind: String = "MixedTypesComparison"

  override def getMessage: String = {
    s"Sorting using values of different types may produce unexpected results (type1: ${type1.name}, type2: ${type2.name})."
  }
}
