package org.mule.weave.v2.runtime.core.functions.stringops

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.core.functions.UnaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.types.RegexType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.NullValue
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.location.LocationCapable

import java.util.regex.Matcher
import scala.collection.mutable.ArrayBuffer
import scala.util.matching.Regex

object ReplaceStringRegexFunctionValue extends BinaryFunctionValue {

  override val L = StringType

  override val R = RegexType

  override def doExecute(first: L.V, second: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val value: String = first.evaluate.toString
    val pattern = second.evaluate.pattern
    val matcher: Matcher = ctx.serviceManager.patternService.matcher(pattern, value, this)
    new DoReplaceFunctionValue(value, matcher)
  }
}

class DoReplaceFunctionValue(value: String, matcher: Matcher) extends UnaryFunctionValue {
  override val R: Type = FunctionType

  override def doExecute(third: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val result: StringBuilder = new StringBuilder()
    var index: Int = 0
    var begin: Int = 0
    while (matcher.find()) {
      result.++=(value.substring(begin, matcher.start()))
      val lambdaResult: Value[_] = third.asInstanceOf[FunctionValue].call(buildMatcherResult(matcher, UnknownLocationCapable), NumberValue(index))
      result.++=(StringType.coerce(lambdaResult).evaluate(ctx).toString)
      begin = matcher.end()
      index += 1
    }
    result.++=(value.substring(begin))
    StringValue(result.toString(), UnknownLocationCapable)
  }

  def buildMatcherResult(matcher: Matcher, location: LocationCapable): ArrayValue = {
    val count: Int = matcher.groupCount()
    var index = 0
    val matchBuilder = new ArrayBuffer[Value[_]]
    while (index <= count) {
      val group: String = matcher.group(index)
      matchBuilder.+=(if (group == null) NullValue else StringValue(group, this))
      index = index + 1
    }
    ArrayValue(matchBuilder.result(), location)
  }
}

object ReplaceStringStringFunctionValue extends BinaryFunctionValue {

  override val L = StringType

  override val R = StringType

  override def doExecute(first: L.V, second: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val value: String = first.evaluate.toString
    val matcher = new Regex(Regex.quote(second.evaluate.toString)).pattern.matcher(value)
    new DoReplaceFunctionValue(value, matcher)
  }
}

object ReplaceFunctionValue {
  val value = Seq(
    ReplaceStringRegexFunctionValue,
    ReplaceStringStringFunctionValue)
}
