package org.mule.weave.v2.interpreted.debugger.server

import org.mule.weave.v2.debugger.DebuggerFrame
import org.mule.weave.v2.debugger.DebuggerValue
import org.mule.weave.v2.debugger.FieldDebuggerValue
import org.mule.weave.v2.debugger.KeyDebuggerValue
import org.mule.weave.v2.debugger.ObjectDebuggerValue
import org.mule.weave.v2.debugger.SimpleDebuggerValue
import org.mule.weave.v2.debugger.event.ExecutionStopReason
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.InterpretedMappingExecutableWeave
import org.mule.weave.v2.interpreted.debugger.server.DebuggerConverters.toDebuggerLocation
import org.mule.weave.v2.interpreted.debugger.server.DebuggerConverters.toDebuggerPosition
import org.mule.weave.v2.interpreted.debugger.server.DebuggerConverters.unknownLocation
import org.mule.weave.v2.interpreted.listener.WeaveExecutionListener
import org.mule.weave.v2.interpreted.node.FunctionCallNode
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.structure.DocumentNode
import org.mule.weave.v2.interpreted.node.structure.header.VariableTable
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.runtime.exception.InvalidScriptException
import org.mule.weave.v2.sdk.ParsingContextFactory
import org.mule.weave.v2.sdk.WeaveResourceFactory
import org.mule.weave.v2.utils.IdentityHashMap

/**
  * Handles the execution of a weave script
  *
  * @param session The debugging session
  */
class WeaveDebuggerExecutor(val session: WeaveDebuggingSession) extends WeaveExecutionListener {

  @volatile
  var status: Int = WeaveDebuggerExecutor.RESUMED
  val sessionLock = new Object()
  var debuggingStackIndex: Int = -1
  var lastLine: Int = -1
  var lastFrame: Frame = _
  //The context where the execution was stopped
  var stoppedContext: ExecutionContext = _

  var maxValueDepth = 30
  var maxValueElements = 100

  override def preExecution(node: ValueNode[_])(implicit ctx: ExecutionContext): Unit = {
    //
    if (session.started()) {
      val location: WeaveLocation = node.location()
      val threadStack = ctx.executionStack()
      val activeFrame: Frame = threadStack.activeFrame()
      val locationPosition: Position = location.startPosition
      var reason = ExecutionStopReason.STEP

      if (validateNewPositionOrFrame(activeFrame, locationPosition)) {
        lastLine = locationPosition.line
        lastFrame = activeFrame
        val currentFrames: Array[Frame] = threadStack.frames()
        //If we are waiting for a message
        val stackSize: Int = currentFrames.length
        if ((status == WeaveDebuggerExecutor.RESUMED || debuggingStackIndex != stackSize) && session.getWeaveBreakpointManager().hasBreakpointOn(node)) {
          debuggingStackIndex = stackSize
          reason = ExecutionStopReason.BREAKPOINT
          status = WeaveDebuggerExecutor.NEXT_STEP
        }

        //If we are debugging a message let them know that a new node has been hit
        if (status == WeaveDebuggerExecutor.STEP_IN
          || (status == WeaveDebuggerExecutor.NEXT_STEP && debuggingStackIndex >= stackSize)) {
          debuggingStackIndex = stackSize
          val values = new IdentityHashMap[Value[_], DebuggerValue]()
          val frames: Array[DebuggerFrame] = currentFrames.zipWithIndex.map {
            case (frame, id) => {
              var frameEntries: Seq[(String, DebuggerValue)] = toFrameValueMap(frame, values)
              frameEntries = node match {
                case functionCallNode: FunctionCallNode if (frame eq activeFrame) => {
                  frameEntries :+ (WeaveDebuggerExecutor.FUNCTION_CALL_FIELD_NAME -> SimpleDebuggerValue(getFunctionName(functionCallNode), "FunctionName", toDebuggerLocation(node.location())))
                }
                case _ => frameEntries
              }
              val weaveLocation: WeaveLocation = frame.currentCallSite().getOrElse(frame).location().asInstanceOf[WeaveLocation]
              val startPosition: Position = weaveLocation.startPosition
              val endPosition: Position = weaveLocation.endPosition
              val resourceName = weaveLocation.resourceName
              DebuggerFrame(id, frameEntries.toArray, toDebuggerPosition(resourceName, startPosition), toDebuggerPosition(resourceName, endPosition), frame.name)
            }
          }
          stopExecution(frames, location, reason)
        }
      }
    }
  }

  def validateNewPositionOrFrame(activeFrame: Frame, locationPosition: Position): Boolean = {
    locationPosition != null && (locationPosition.line != lastLine || activeFrame != lastFrame)
  }

  override def postExecution(node: ValueNode[_], result: Value[_])(implicit ctx: ExecutionContext): Unit = {

    if (session.started()) {
      //If we are debugging a message let them node that a new node has been hit
      node match {
        case functionCallNode: FunctionCallNode =>
          //Makes sure we consume the result of every function call so that
          val resultDebugValue = DebuggerValueFactory.create(result, maxValueElements, maxValueDepth)
          val threadStack = ctx.executionStack()
          val activeFrame: Frame = threadStack.activeFrame()
          val currentIndex: Int = threadStack.frames().length
          val functionCallName = getFunctionName(functionCallNode)
          if (status == WeaveDebuggerExecutor.STEP_IN
            || (status == WeaveDebuggerExecutor.NEXT_STEP && debuggingStackIndex >= currentIndex)
            || (status == WeaveDebuggerExecutor.STEP_OUT && debuggingStackIndex > currentIndex)) {
            //We need this cache of already processed values
            val values = new IdentityHashMap[Value[_], DebuggerValue]()
            val frames = threadStack.frames().zipWithIndex.map {
              case (frame, id) => {
                var frameEntries: Seq[(String, DebuggerValue)] = toFrameValueMap(frame, values)
                if (frame eq activeFrame) {
                  frameEntries = frameEntries.:+((s"$functionCallName Result", resultDebugValue))
                }
                val weaveLocation: WeaveLocation = frame.currentCallSite().getOrElse(frame).location().asInstanceOf[WeaveLocation]
                val startPosition: Position = weaveLocation.startPosition
                val endPosition: Position = weaveLocation.endPosition
                val resourceName = weaveLocation.resourceName
                DebuggerFrame(id, frameEntries.toArray, toDebuggerPosition(resourceName, startPosition), toDebuggerPosition(resourceName, endPosition), frame.name)
              }
            }
            stopExecution(frames, node.location(), ExecutionStopReason.STEP)
          }
        case _ =>
      }
    }

  }

  private def getFunctionName(functionCallNode: FunctionCallNode)(implicit ctx: ExecutionContext) = {
    functionCallNode.functionName()
  }

  override def postExecution(node: ValueNode[_], e: Exception)(implicit ctx: ExecutionContext): Unit = {
    if (session.started()) {
      node match {
        case functionCallNode: FunctionCallNode =>
          val threadStack = ctx.executionStack()
          val activeFrame: Frame = threadStack.activeFrame()
          //If we are debugging a message let them node that a new node has been hit
          val currentIndex: Int = threadStack.frames().length
          if (status == WeaveDebuggerExecutor.STEP_IN
            || (status == WeaveDebuggerExecutor.NEXT_STEP && debuggingStackIndex >= currentIndex)
            || session.getWeaveBreakpointManager().hasExceptionBreakpoint(e)) {
            val values = new IdentityHashMap[Value[_], DebuggerValue]()
            val frames = threadStack.frames().zipWithIndex.map {
              case (frame, id) => {
                var frameEntries: Seq[(String, DebuggerValue)] = toFrameValueMap(frame, values)
                if (frame == activeFrame) {
                  frameEntries = frameEntries.:+((s"${getFunctionName(functionCallNode)} Exception", SimpleDebuggerValue(e.getMessage, e.getClass.getName, toDebuggerLocation(node.location()))))
                }
                val weaveLocation: WeaveLocation = frame.currentCallSite().getOrElse(frame).location().asInstanceOf[WeaveLocation]
                val startPosition: Position = weaveLocation.startPosition
                val endPosition: Position = weaveLocation.endPosition
                val resourceName: NameIdentifier = weaveLocation.resourceName
                DebuggerFrame(id, frameEntries.toArray, toDebuggerPosition(resourceName, startPosition), toDebuggerPosition(resourceName, endPosition), frame.name)
              }
            }
            stopExecution(frames, node.location(), ExecutionStopReason.EXCEPTION)
          }
        case _ => {
          if (session.getWeaveBreakpointManager().hasExceptionBreakpoint(e)) {
            val threadStack = ctx.executionStack()
            val values = new IdentityHashMap[Value[_], DebuggerValue]()
            val frames = threadStack.frames().zipWithIndex.map {
              case (frame, id) => {
                val frameEntries: Seq[(String, DebuggerValue)] = toFrameValueMap(frame, values)
                val weaveLocation: WeaveLocation = frame.currentCallSite().getOrElse(frame).location().asInstanceOf[WeaveLocation]
                val startPosition: Position = weaveLocation.startPosition
                val endPosition: Position = weaveLocation.endPosition
                val resourceName: NameIdentifier = weaveLocation.resourceName
                DebuggerFrame(id, frameEntries.toArray, toDebuggerPosition(resourceName, startPosition), toDebuggerPosition(resourceName, endPosition), frame.name)
              }
            }
            stopExecution(frames, node.location(), ExecutionStopReason.EXCEPTION)
          }

        }
      }
    }
  }

  def resume(): Unit = {
    sessionLock.synchronized {
      status = WeaveDebuggerExecutor.RESUMED
      sessionLock.notifyAll()
    }
  }

  def nextStep(): Unit = {
    sessionLock.synchronized {
      status = WeaveDebuggerExecutor.NEXT_STEP
      sessionLock.notifyAll()
    }
  }

  def stepOut(): Unit = {
    sessionLock.synchronized {
      status = WeaveDebuggerExecutor.STEP_OUT
      sessionLock.notifyAll()
    }
  }

  def stepInto(): Unit = {
    sessionLock.synchronized {
      status = WeaveDebuggerExecutor.STEP_IN
      sessionLock.notifyAll()
    }
  }

  def toFrameValueMap(frame: Frame, values: IdentityHashMap[Value[_], DebuggerValue])(implicit ctx: ExecutionContext): Seq[(String, DebuggerValue)] = {
    val names = frame.moduleContext.variableTable.variableNames()
    frame.content.zipWithIndex.flatMap((valueWithIndex) => {
      if (valueWithIndex._1 != null) {
        val variableIndex = valueWithIndex._2
        if (names.size > variableIndex) {
          val debuggerValue = values.getOrElseUpdate(valueWithIndex._1, DebuggerValueFactory.create(valueWithIndex._1, maxValueElements, maxValueDepth))
          Some((names(variableIndex), debuggerValue))
        } else {
          //What to do here
          None
        }
      } else {
        None
      }
    })
  }

  def stopExecution(frames: Seq[DebuggerFrame], weaveLocation: WeaveLocation, reason: Int)(implicit ctx: ExecutionContext): Unit = {
    status = WeaveDebuggerExecutor.WAITING
    stoppedContext = ctx
    session.onExecutionPaused(frames.toArray, weaveLocation, reason)
    sessionLock.synchronized {
      if (status == WeaveDebuggerExecutor.WAITING) {
        ctx.writer.foreach(_.flush())
        sessionLock.wait()
      }
    }
  }

  def evalScript(script: String, frameIndex: Int = -1): DebuggerValue = {
    try {
      val (context: ExecutionContext, materialize: Value[_]) = evaluate(script, frameIndex)
      DebuggerValueFactory.create(materialize, maxValueElements, maxValueDepth)(context)
    } catch {
      case e: Exception => {
        ObjectDebuggerValue(
          Array(
            FieldDebuggerValue(
              KeyDebuggerValue("Exception", Array(), unknownLocation()),
              SimpleDebuggerValue(e.getMessage, e.getClass.getSimpleName, unknownLocation()),
              unknownLocation())),
          ":exception",
          unknownLocation())
      }
    }
  }

  def evaluate(script: String, frameIndex: Int = -1, ctx: ExecutionContext = stoppedContext): (ExecutionContext, Value[_]) = {
    val activeFrame: Frame = if (frameIndex == -1) {
      ctx.executionStack().activeFrame()
    } else {
      ctx.executionStack().frames()(frameIndex)
    }

    val parsingContext: ParsingContext = ParsingContextFactory.createParsingContext(NameIdentifier.anonymous)
    activeFrame.moduleContext.variableTable.variables.foreach((variable) => {
      parsingContext.addImplicitInput(variable.name, None)
    })
    val parser = WeaveCompiler.compileWithNoCheck(WeaveResourceFactory.fromContent(script), parsingContext)
    if (parser.hasErrors()) {
      val error: (WeaveLocation, Message) = parser.errorMessages().head
      new InvalidScriptException(error._2.message, error._1)
    }
    val parse: DocumentNode = parser.getResult().executable.asInstanceOf[InterpretedMappingExecutableWeave].executableDocument
    val table: VariableTable = parse.header.variableTable

    //We need to map variables from the script with the ones in the specified frame
    val mappedVariables = table.variables.map((variable) => {
      val slot: Int = activeFrame.moduleContext.getVariableSlot(variable.name)
      if (slot == -1) {
        throw new RuntimeException(s"Invalid variable name ${variable.name}.")
      }
      (variable.slot, activeFrame.variableAt(slot))
    })
    //We create a new frame with the new variables
    val context: ExecutionContext = ExecutionContext(activeFrame.child(mappedVariables.toArray, UnknownLocationCapable, None), ctx)
    val execute: Value[Any] = parse.execute(context)
    val materialize: Value[_] = execute.materialize(context)
    (context, materialize)
  }
}

object WeaveDebuggerExecutor {
  //Execute normal
  val RESUMED = 0
  //Execute next step
  val NEXT_STEP = 1
  //Execute step in
  val STEP_IN = 3
  //Waiting for the client to continue
  val WAITING = 4
  //Execute step in
  val STEP_OUT = 5

  val FUNCTION_CALL_FIELD_NAME = "Function Call"
}
