package org.mule.weave.v2.interpreted.node

import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.node.executors.BinaryExecutor
import org.mule.weave.v2.interpreted.node.executors.EmptyExecutor
import org.mule.weave.v2.interpreted.node.executors.FunctionExecutor
import org.mule.weave.v2.interpreted.node.executors.TernaryExecutor
import org.mule.weave.v2.interpreted.node.executors.UnaryExecutor
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.location.WeaveLocation

trait FunctionCallNode extends ValueNode[Any] {

  def functionExecutor: FunctionExecutor

  def args: Array[ValueNode[_]]

  def functionName()(implicit ctx: ExecutionContext): String

}

class DefaultFunctionCallNode(val functionExecutor: FunctionExecutor, val args: Array[ValueNode[_]]) extends FunctionCallNode with Product2[FunctionExecutor, Array[ValueNode[_]]] {

  override def doExecute(implicit ctx: ExecutionContext): Value[_] = {
    val executedArgs = executeArgs()
    functionExecutor.execute(executedArgs)
  }

  private def executeArgs()(implicit ctx: ExecutionContext): Array[Value[_]] = {
    val argsValues = new Array[Value[_]](args.length)
    var i = 0
    while (i < args.length) {
      argsValues.update(i, args(i).execute)
      i = i + 1
    }
    argsValues
  }

  def functionName()(implicit ctx: ExecutionContext): String = functionExecutor.name()

  override def _1: FunctionExecutor = functionExecutor

  override def _2: Array[ValueNode[_]] = args

}

class EmptyFunctionCallNode(val functionExecutor: EmptyExecutor) extends FunctionCallNode with Product2[FunctionExecutor, Array[ValueNode[_]]] {

  override def doExecute(implicit ctx: ExecutionContext): Value[_] = {
    functionExecutor.executeEmpty()
  }

  def functionName()(implicit ctx: ExecutionContext): String = functionExecutor.name()

  override def _1: FunctionExecutor = functionExecutor

  override def _2: Array[ValueNode[_]] = args

  override def args: Array[ValueNode[_]] = Array.empty
}

class UnaryFunctionCallNode(val functionExecutor: UnaryExecutor, val firstArg: ValueNode[_]) extends FunctionCallNode with Product2[FunctionExecutor, ValueNode[_]] {

  override def doExecute(implicit ctx: ExecutionContext): Value[_] = {
    functionExecutor.executeUnary(firstArg.execute)
  }

  def functionName()(implicit ctx: ExecutionContext): String = functionExecutor.name()

  override def _1: FunctionExecutor = functionExecutor

  override def _2: ValueNode[_] = firstArg

  override def args: Array[ValueNode[_]] = Array(firstArg)
}

class BinaryFunctionCallNode(val functionExecutor: BinaryExecutor, val firstArg: ValueNode[_], val secondArg: ValueNode[_]) extends FunctionCallNode with Product3[FunctionExecutor, ValueNode[_], ValueNode[_]] {

  override def doExecute(implicit ctx: ExecutionContext): Value[_] = {
    functionExecutor.executeBinary(firstArg.execute, secondArg.execute)
  }

  def functionName()(implicit ctx: ExecutionContext): String = functionExecutor.name()

  override def _1: FunctionExecutor = functionExecutor

  override def _2: ValueNode[_] = firstArg

  override def _3: ValueNode[_] = secondArg

  override def args: Array[ValueNode[_]] = Array(firstArg, secondArg)
}

class TernaryFunctionCallNode(
  val functionExecutor: TernaryExecutor,
  val firstArg: ValueNode[_],
  val secondArg: ValueNode[_],
  val thirdArg: ValueNode[_]) extends FunctionCallNode with Product4[FunctionExecutor, ValueNode[_], ValueNode[_], ValueNode[_]] {

  override def doExecute(implicit ctx: ExecutionContext): Value[_] = {
    functionExecutor.executeTernary(firstArg.execute, secondArg.execute, thirdArg.execute)
  }

  def functionName()(implicit ctx: ExecutionContext): String = functionExecutor.name()

  override def _1: FunctionExecutor = functionExecutor

  override def _2: ValueNode[_] = firstArg

  override def _3: ValueNode[_] = secondArg

  override def _4: ValueNode[_] = thirdArg

  override def args: Array[ValueNode[_]] = Array(firstArg, secondArg, thirdArg)
}

object FunctionCallNode {

  def addCallToStacktrace(ex: ExecutionException, loc: WeaveLocation, name: String)(implicit ctx: ExecutionContext): Unit = {
    ex.weaveStacktrace.addFunctionCall(name, loc)
  }
}