package org.mule.weave.v2.runtime.core.functions.types

import org.mule.weave.v2.core.functions.UnaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.types.KeyType
import org.mule.weave.v2.model.types.NameType
import org.mule.weave.v2.model.types.NameValuePairType
import org.mule.weave.v2.model.types.NamespaceType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.TypeType
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.BooleanValue
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.NullValue
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.TypeValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.runtime.core.functions.types.exceptions.InvalidTypeException

object ObjectTypeFieldsFunction extends UnaryFunctionValue {
  override val R = TypeType

  def attributes(key: KeyType): ArrayValue = {
    val attrs = key.attrsType.map((nvp) => {
      ObjectValue(
        Array(
          KeyValuePair(KeyValue("name"), attrQName(nvp)),
          KeyValuePair(KeyValue("value"), TypeValue(nvp.value)),
          KeyValuePair(KeyValue("required"), BooleanValue(!nvp.optional))))
    })
    ArrayValue(attrs)
  }

  def attrQName(nvp: NameValuePairType): ObjectValue = {
    nvp.nameType match {
      case nameType @ NameType(Some(name), _) => {
        ObjectValue(
          Array(
            KeyValuePair(KeyValue("localName"), StringValue(name)),
            KeyValuePair(KeyValue("namespace"), namespace(nameType))))
      }
      case _ => {
        ObjectValue(
          Array(
            KeyValuePair(KeyValue("localName"), StringValue("_")),
            KeyValuePair(KeyValue("namespace"), NullValue)))
      }
    }
  }

  def asKeyObject(key: KeyType): Value[_] = {
    ObjectValue(
      Array(
        KeyValuePair(KeyValue("name"), asName(key)),
        KeyValuePair(KeyValue("attributes"), attributes(key))))
  }

  private def asName(key: KeyType): ObjectValue = {
    key.keyName match {
      case Some(nameType @ NameType(Some(name), _)) => {
        ObjectValue(
          Array(
            KeyValuePair(KeyValue("localName"), StringValue(name)),
            KeyValuePair(KeyValue("namespace"), namespace(nameType))))
      }
      case _ => {
        ObjectValue(
          Array(
            KeyValuePair(KeyValue("localName"), StringValue("_")),
            KeyValuePair(KeyValue("namespace"), NullValue)))
      }
    }
  }

  private def namespace(key: NameType): Value[_] = {
    key.ns match {
      case Some(NamespaceType(Some(ns))) => StringValue(ns)
      case _                             => NullValue
    }

  }

  override def doExecute(v: R.V)(implicit executionContext: EvaluationContext): Value[_] = {
    v.evaluate.baseType match {
      case ot: ObjectType => {
        val fields = ot.keyValuePairs.map((kvp) => {
          ObjectValue(
            Array(
              KeyValuePair(KeyValue("key"), asKeyObject(kvp.key.baseType.asInstanceOf[KeyType])),
              KeyValuePair(KeyValue("required"), BooleanValue(!kvp.optional)),
              KeyValuePair(KeyValue("repeated"), BooleanValue(kvp.repeated)),
              KeyValuePair(KeyValue("value"), TypeValue(kvp.value))))
        })
        ArrayValue(fields)
      }
      case t => throw new InvalidTypeException(location, "Object", t.name)
    }
  }
}
