package org.mule.weave.v2.interpreted.node.executors

import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.core.exception.NotEnoughArgumentsException
import org.mule.weave.v2.core.exception.TooManyArgumentsException
import org.mule.weave.v2.core.exception.UnexpectedFunctionCallTypesException
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.VariableReferenceNode
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.types.Types
import org.mule.weave.v2.model.values.FunctionParameter
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.coercion.FunctionValueCoercer
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.location.WeaveLocation

import java.util.concurrent.atomic.AtomicInteger
import java.util.concurrent.atomic.AtomicReference

/**
  * This is the default function it can handle dynamic and static calls of any arity. This is the slow path as it needs to do lots of checks.
  *
  * @param functionNode     The node that produces the function
  * @param showInStacktrace If this function should add an entry on the stacktrace when it fails
  */
class DefaultFunctionCallExecutor(val functionNode: ValueNode[_], val showInStacktrace: Boolean = true, override val location: WeaveLocation) extends DefaultExecutor with Product2[ValueNode[_], Boolean] {

  // We cached the index of the overloaded function that was previously dispatched
  private val cachedFunction: AtomicInteger = new AtomicInteger(-1)

  private val cachedCoercedFunction: AtomicReference[FunctionDispatchInformation] = new AtomicReference()

  override def execute(arguments: Array[Value[_]])(implicit ctx: ExecutionContext): Value[Any] = {
    val activeFrame: Frame = ctx.executionStack().activeFrame()
    try {
      activeFrame.updateCallSite(functionNode)
      val functionValue: FunctionValue = FunctionValueCoercer.coerce(functionNode.execute, None, functionNode)
      //Validate arity first
      if (arguments.length > functionValue.maxParams) {
        val parameterValues: Array[FunctionParameter] = functionValue.parameters
        throw new TooManyArgumentsException(location, arguments.length, parameterValues)
      } else if (arguments.length < functionValue.minParams) {
        val parameterValues: Array[FunctionParameter] = functionValue.parameters
        throw new NotEnoughArgumentsException(location, arguments.length, parameterValues)
      }

      if (!functionValue.isOverloaded) {
        //For simple functions we just try to call it
        val maybeMatArguments =
          if (functionValue.paramsTypesRequiresMaterialize) {
            FunctionDispatchingHelper.materializeArgs(functionValue.parameters, arguments)
          } else {
            arguments
          }

        val expandedArguments =
          if (arguments.length != functionValue.parameters.length) {
            FunctionDispatchingHelper.expandArguments(maybeMatArguments, functionValue)
          } else {
            maybeMatArguments
          }

        if (FunctionDispatchingHelper.matchesFunctionTypes(functionValue, maybeMatArguments)) {
          doCall(functionValue, expandedArguments)
        } else {
          val mayBeValues = FunctionDispatchingHelper.tryToCoerce(expandedArguments, functionValue)
          mayBeValues match {
            case Some(coercedValues) => doCall(functionValue, coercedValues)
            case None                => throw new UnexpectedFunctionCallTypesException(location, functionValue.label, arguments, Seq(functionValue.parameters.map(_.wtype)))
          }
        }
      } else {
        dispatchOverloadedFunction(functionValue, arguments, functionValue.overloads)
      }
    } finally {
      activeFrame.cleanCallSite()
    }
  }

  private def doCall(functionValue: FunctionValue, coercedValues: Array[Value[_]])(implicit ctx: ExecutionContext) = {
    try {
      functionValue.call(coercedValues)
    } catch {
      case ex: ExecutionException =>
        if (showInStacktrace) {
          ex.addCallToStacktrace(location, name())
        }
        throw ex
    }
  }

  /**
    * This method return the arguments materialized if the corresponding parameters require that to check if the types match.
    * This is for Objects with defined fields, for example.
    */

  private def dispatchOverloadedFunction(function: FunctionValue, arguments: Array[Value[_]], functionOverloads: Array[_ <: FunctionValue])(implicit ctx: ExecutionContext): Value[_] = {
    val maybeMatArguments =
      if (function.paramsTypesRequiresMaterialize) {
        FunctionDispatchingHelper.materializeOverloadedFunctionArgs(functionOverloads, arguments)
      } else {
        arguments
      }

    //Try cached data and if it doesn't work find new dispatching
    if (cachedFunction.get() != -1) {
      val dispatchIndex = cachedFunction.get
      if (dispatchIndex < functionOverloads.length && FunctionDispatchingHelper.matchesFunctionTypes(functionOverloads(dispatchIndex), maybeMatArguments)) {
        val dispatchFunction = functionOverloads(dispatchIndex)
        val expandedArguments = FunctionDispatchingHelper.expandArguments(maybeMatArguments, dispatchFunction)
        return doCall(dispatchFunction, expandedArguments)
      }
    } else if (cachedCoercedFunction.get != null) {
      val cf = cachedCoercedFunction.get
      if (cf.dispatchIndex < functionOverloads.length && Types.validate(cf.expectedTypes, maybeMatArguments)) {
        val dispatchFunction = functionOverloads(cf.dispatchIndex)
        val expandedArguments = FunctionDispatchingHelper.expandArguments(maybeMatArguments, dispatchFunction)
        val coercedValues = FunctionDispatchingHelper.tryToCoerceOnly(expandedArguments, dispatchFunction, cf.indexOfParamsToCoerce)
        if (coercedValues != null) {
          return doCall(dispatchFunction, coercedValues)
        }
      }
    }

    //Cached function didn't work so we try first to find a matching function else we
    //Try auto coercion: This is we sort the function based on types weights and distance
    //Then we try to coerce the values
    val functionToCall = FunctionDispatchingHelper.findMatchingFunction(maybeMatArguments, functionOverloads)
    functionToCall match {
      case Some((functionIndex, dispatchFunction)) =>
        if (function.dispatchCanBeCached)
          cachedFunction.set(functionIndex)
        val expandedArguments = FunctionDispatchingHelper.expandArguments(maybeMatArguments, dispatchFunction)
        doCall(dispatchFunction, expandedArguments)
      case None => {
        val materializedValues: Array[Value[Any]] = maybeMatArguments.map(_.materialize)
        val argTypes: Array[Type] = materializedValues.map(_.valueType)
        val sortedOperators: Array[FunctionValue] = FunctionDispatchingHelper.sortByParameterTypeWeight(functionOverloads, argTypes)
        val functionToCallWithCoercion = FunctionDispatchingHelper.findMatchingFunctionWithCoercion(materializedValues, sortedOperators, this)
        functionToCallWithCoercion match {
          case Some((dispatchFunctionIndex, coercedArguments, indexToCoerce)) => {
            //We cache the target method for the specified values
            //We only cached the base types as types can have schemas and schemas have values and values can not be cached
            //As they may leak resources
            val dispatchFunction = sortedOperators(dispatchFunctionIndex)
            if (function.dispatchCanBeCached) {
              cachedCoercedFunction.set(FunctionDispatchInformation(functionOverloads.indexOf(dispatchFunction), argTypes.map(_.baseType), indexToCoerce))
            }

            doCall(dispatchFunction, FunctionDispatchingHelper.expandArguments(coercedArguments, dispatchFunction))
          }
          case None =>
            throw new UnexpectedFunctionCallTypesException(location, function.label, maybeMatArguments, sortedOperators.map(_.parameterTypes))
        }
      }
    }
  }

  override def name()(implicit ctx: ExecutionContext): String = {
    val maybeString = (functionNode match {
      case vrn: VariableReferenceNode => Some(vrn.variable.name)
      case _                          => None
    }).orElse({
      FunctionType.coerce(functionNode.execute, functionNode).name
    })
    maybeString.getOrElse(AstNodeHelper.ANONYMOUS_FUNCTION)
  }

  override def node(): ValueNode[_] = functionNode

  override def _1: ValueNode[_] = functionNode

  override def _2: Boolean = showInStacktrace
}

case class FunctionDispatchInformation(dispatchIndex: Int, expectedTypes: Array[Type], indexOfParamsToCoerce: Seq[Int])