package org.mule.weave.v2.runtime.core.functions.runtime

import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.BooleanValue
import org.mule.weave.v2.model.values.NullValue
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.ObjectValueBuilder
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.option.ModuleOption

import scala.annotation.tailrec

object DataFormatDescriptorConverter {

  private val NAME_KEY = "name"
  private val BINARY_KEY = "binary"
  private val DEFAULT_ENCODING_KEY = "defaultEncoding"
  private val EXTENSIONS_KEY = "extensions"
  private val DEFAULT_MIME_TYPE_KEY = "defaultMimeType"
  private val ACCEPTED_MIME_TYPES_KEY = "acceptedMimeTypes"
  private val READER_PROPERTIES_KEY = "readerProperties"
  private val WRITER_PROPERTIES_KEY = "writerProperties"
  private val OPTIONAL_KEY = "optional"
  private val DEFAULT_VALUE_KEY = "defaultValue"
  private val DESCRIPTION_KEY = "description"
  private val POSSIBLE_VALUES_KEY = "possibleValues"

  def toObjectValue(df: DataFormat[_, _]): ObjectValue = {
    val dataFormatDescriptor = new ObjectValueBuilder()
    dataFormatDescriptor.addPair(NAME_KEY, df.name())
    dataFormatDescriptor.addPair(BINARY_KEY, df.binaryFormat)
    if (df.defaultCharset.isDefined) {
      dataFormatDescriptor.addPair(DEFAULT_ENCODING_KEY, df.defaultCharset.get.name())
    }
    dataFormatDescriptor.addPair(EXTENSIONS_KEY, df.fileExtensions)
    dataFormatDescriptor.addPair(DEFAULT_MIME_TYPE_KEY, df.defaultMimeType.toString)
    dataFormatDescriptor.addPair(ACCEPTED_MIME_TYPES_KEY, df.acceptedMimeTypes.map(_.toString))
    dataFormatDescriptor.addPair(READER_PROPERTIES_KEY, buildDataFormatProperties(df.readerOptions()))
    dataFormatDescriptor.addPair(WRITER_PROPERTIES_KEY, buildDataFormatProperties(df.writerOptions()))
    dataFormatDescriptor.build
  }

  private def buildDataFormatProperties(options: Map[String, ModuleOption]): ArrayValue = {
    val optionsValue = options
      .filterNot(_._2.internal())
      .map(option => {
        val optionValue = option._2
        val dataFormatDescriptor = new ObjectValueBuilder()
        dataFormatDescriptor.addPair(NAME_KEY, option._1)
        dataFormatDescriptor.addPair(OPTIONAL_KEY, !optionValue.required)
        dataFormatDescriptor.addPair(DEFAULT_VALUE_KEY, asValue(optionValue.defaultValue))
        dataFormatDescriptor.addPair(DESCRIPTION_KEY, optionValue.description)
        dataFormatDescriptor.addPair(POSSIBLE_VALUES_KEY, ArrayValue(optionValue.possibleValues.map(asValue).toSeq))
        dataFormatDescriptor.build
      })
      .toSeq

    ArrayValue(optionsValue)
  }

  @tailrec
  private def asValue(v: Any): Value[_] = {
    v match {
      case s: String  => StringValue(s)
      case s: Boolean => BooleanValue(s)
      case s: Int     => NumberValue(s)
      case s: Long    => NumberValue(s)
      case s: Double  => NumberValue(s)
      case null       => NullValue
      case None       => StringValue("None")
      case Some(s)    => asValue(s)
      case _          => StringValue(String.valueOf(v))
    }
  }
}
