package org.mule.weave.v2.helper

import org.mule.weave.compiler.WeaveCompilerConfigProperties.DISABLED_COMMON_SUBEXPRESSION_REDUCTION_PROPERTY

import java.io.File
import org.mule.weave.v2.parser.MappingParser
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.{ BaseParsingContextSettings, CompilationPhase, ModuleParsingPhasesManager, ParsingContentInput, ParsingContext, ParsingResult, PhaseResult, ResourceBasedModuleParser }
import org.mule.weave.v2.parser.phase.metrics.ParsingMetricsCollectorListener
import org.mule.weave.v2.sdk.ParsingContextFactory.createDefaultModuleParserManager
import org.mule.weave.v2.sdk.{ ParsingContextFactory, WeaveResource, WeaveResourceFactory, WeaveResourceResolver }

trait ParsingContextTestAware extends BaseDataWeaveTest {

  val dependencyParser: ModuleParsingPhasesManager = ParsingContextFactory.createDefaultModuleParserManager()

  def createTestParsingContext(weaveFile: File, strictMode: Boolean = true): ParsingContext = {
    val parsingContext = ParsingContext(createNameIdentifier(weaveFile), new MessageCollector(), dependencyParser, strictMode = strictMode)
    registerProfilerListener(parsingContext)
  }

  def createTestParsingContext(strictMode: Boolean): ParsingContext = {
    val context = ParsingContext(NameIdentifier.anonymous, new MessageCollector(), dependencyParser, strictMode = strictMode)
    registerProfilerListener(context)
  }

  def createTestParsingContext(): ParsingContext = {
    val context = ParsingContext(NameIdentifier.anonymous, new MessageCollector(), dependencyParser)
    registerProfilerListener(context)
  }

  def createTestParsingContext(weaveFile: File, parsingContextSettings: BaseParsingContextSettings, weaveResourceResolver: WeaveResourceResolver): ParsingContext = {
    val identifier: NameIdentifier = createNameIdentifier(weaveFile)
    val context = new ParsingContext(identifier, new MessageCollector,
      createDefaultModuleParserManager(new ResourceBasedModuleParser(weaveResourceResolver, Option.apply("dw"))), settings = parsingContextSettings)
    registerProfilerListener(context)
  }

  def createTestParsingContext(weaveFile: File, parsingContextSettings: BaseParsingContextSettings): ParsingContext = {
    val identifier: NameIdentifier = createNameIdentifier(weaveFile)
    val context = ParsingContextFactory.createParsingContext(identifier, parsingContextSettings)
    registerProfilerListener(context)
  }

  def createParsingContext(weaveFile: File): ParsingContext = {
    val identifier: NameIdentifier = createNameIdentifier(weaveFile)
    val context = ParsingContextFactory.createParsingContext(identifier)
    registerProfilerListener(context)
  }

  private def registerProfilerListener(context: ParsingContext): ParsingContext = {
    if (java.lang.Boolean.getBoolean("com.dw.parser.profile")) {
      context.notificationManager.addListener(new ParsingMetricsCollectorListener())
    }
    if (java.lang.Boolean.getBoolean(DISABLED_COMMON_SUBEXPRESSION_REDUCTION_PROPERTY)) {
      context.disableCommonSubExpressionElimination()
    }
    context
  }

  private def createNameIdentifier(weaveFile: File) = {
    val replace = getClass.getPackage.getName.replace(".", "::") + "::" + baseName(weaveFile)
    val identifier = NameIdentifier(replace)
    identifier
  }

  def parseWeaveFile(weaveFile: File, phase: CompilationPhase[ParsingContentInput, ParsingResult[DocumentNode]] = MappingParser.canonicalAstPhase()): PhaseResult[ParsingResult[DocumentNode]] = {
    val resource: WeaveResource = WeaveResourceFactory.fromFile(weaveFile)
    parseResource(weaveFile, resource, phase)
  }

  def parseResource(weaveFile: File, resource: WeaveResource, phase: CompilationPhase[ParsingContentInput, ParsingResult[DocumentNode]] = MappingParser.canonicalAstPhase()): PhaseResult[ParsingResult[DocumentNode]] = {
    MappingParser.parse(phase, resource, createTestParsingContext(weaveFile))
  }

  private def baseName(x: File): String = {
    x.getName.substring(0, x.getName.lastIndexOf("."))
  }

  def toJson(location: WeaveLocation, builder: JsonBuilder): JsonBuilder = {
    builder
      .key("location")
      .startObject()
      .key("startColumn")
      .value(location.startPosition.column.toString)
      .key("startLine")
      .value(location.startPosition.line.toString)
      .key("endColumn")
      .value((location.endPosition.column.toString))
      .key("endLine")
      .value((location.endPosition.line.toString))
      .endObject()
  }

}
