package org.mule.weave.v2.helper

import java.io.File
import java.net.URL

import scala.collection.JavaConverters._

trait FolderBasedTest extends BaseDataWeaveTest {
  def getTestFoldersFor(testClass: Class[_]): Seq[File] = {
    val rootTestsUrl: Seq[URL] = testClass.getClassLoader.getResources(testClass.getPackage.getName.replaceAll("\\.", "/")).asScala.toSeq
    val folders = for {
      url <- rootTestsUrl if url.getProtocol.startsWith("file")
    } yield new File(url.toURI)
    folders
  }

  def ignoredFor(clazz: Class[_]): Seq[String] = {
    val ignored = (for {
      rootTestsFolder <- getTestFoldersFor(clazz)
      testFolder <- rootTestsFolder.listFiles if isWIP(testFolder) && isTestToRun(testFolder) && !isTestToSkip(testFolder)
    } yield testFolder.getName).toList
    ignored
  }

  def scenarios(clazz: Class[_] = this.getClass): Seq[Scenario] = {
    val unsortedScenarios = for {
      rootTestsFolder <- getTestFoldersFor(clazz)
      testFolder <- rootTestsFolder.listFiles if acceptScenario(testFolder)
      outputFile <- outputFiles(testFolder)
    } yield Scenario(scenarioName(testFolder, outputFile), inputFiles(testFolder), new File(testFolder, mainTestFile), outputFile, configProperty(testFolder))

    unsortedScenarios.sortBy(_.name)
  }

  def mainTestFile: String = {
    "transform.dwl"
  }

  def scenariosWithoutOutputFor(clazz: Class[_] = this.getClass): Seq[(String, Array[File], File)] = {
    val scenarioTuples = for {
      rootTestsFolder <- getTestFoldersFor(clazz)
      testFolder <- rootTestsFolder.listFiles if acceptScenario(testFolder)
    } yield (testFolder.getName, inputFiles(testFolder), new File(testFolder, mainTestFile))

    scenarioTuples.sortBy((scenarioEntry) => scenarioEntry._1)
  }

  def outputFiles(testFolder: File): Array[File] = {
    testFolder.listFiles.filter(isOutput)
  }

  def configProperty(testFolder: File): Option[File] = {
    testFolder.listFiles.find(f => f.getName.equals("config.properties"))
  }

  def isOutput(file: File): Boolean = {
    file.getName.matches("out\\.[a-zA-Z]+")
  }

  def isTransform(file: File): Boolean = {
    file.getName.equals(mainTestFile)
  }

  def isInput(file: File): Boolean = {
    file.getName.matches("in[0-9]+\\.[a-zA-Z]+")
  }

  def inputFiles(testFolder: File): Array[File] = {
    testFolder.listFiles.filter(isInput).sortBy(_.getName)
  }

  def scenarioName(testFolder: File, outputFile: File): String = {
    testFolder.getName + "-" + outputFile.getName
  }

  def isWIP(file: File): Boolean = {
    file.getName.endsWith("_wip")
  }

  def isTestToRun(file: File) = {
    val fileName: String = file.getName
    val testsToRun = Option(System.getProperty("testToRun")) match {
      case Some(x) => x.split(",")
      case None    => Array[String]()
    }
    testsToRun.isEmpty || testsToRun.exists(fileName.matches)
  }

  def isTestToSkip(file: File) = {
    val fileName = file.getName
    val testsToSkip = Option(System.getProperty("testToSkip")) match {
      case Some(x) => x.split(",")
      case None    => Array[String]()
    }
    testsToSkip.exists(fileName.equals(_))
  }

  def acceptScenario(file: File): Boolean = {
    if (ignoreTests().contains(file.getName)) {
      false
    } else {
      val accepted: Boolean = isTestToRun(file) && !isTestToSkip(file)
      file.isDirectory && !isWIP(file) && accepted
    }
  }

  def ignoreTests(): Array[String] = Array()

  def baseName(x: File): String = {
    val dotIndex = x.getName.lastIndexOf(".")
    if (dotIndex > 0)
      x.getName.substring(0, dotIndex)
    else
      x.getName
  }
}

case class Scenario(name: String, inputs: Array[File], transform: File, output: File, configProperty: Option[File])