package org.mule.weave.v2.matchers

import org.w3c.dom.Node
import org.xmlunit.builder.Input
import org.xmlunit.diff.Diff
import org.xmlunit.util.Predicate

import java.io.File
import java.io.FileInputStream
import java.io.StringReader
import java.util.Properties
import org.scalactic.Equality
import org.scalatest.matchers.MatchResult
import org.scalatest.matchers.Matcher
import org.scalatest.matchers.dsl.MatcherFactory1
import org.xmlunit.builder.DiffBuilder
import spray.json.JsonParser
import spray.json.JsonParserSettings

import scala.io.Source

object WeaveMatchers {

  def matchBin(expected: Array[Byte]) = new BinaryMatcherFactory(expected)

  def matchBin(expected: File): BinaryMatcherFactory = {
    val fin: FileInputStream = new FileInputStream(expected)
    val fileContent: Array[Byte] = new Array[Byte](expected.length.asInstanceOf[Int])
    fin.read(fileContent)
    fin.close()
    new BinaryMatcherFactory(fileContent)
  }

  def matchXml(expectedXml: String) = new XmlMatcher(expectedXml)

  def matchProperties(expectedXml: String) = new PropertiesMatcher(expectedXml)

  def matchJson(expectedJson: String) = new JsonMatcher(expectedJson)

  def matchJson(expectedJson: File): JsonMatcher = {
    val source = Source.fromFile(expectedJson)
    try {
      new JsonMatcher(source.mkString)
    } finally {
      source.close()
    }
  }

  def matchString(expected: Any) = new StringMatcherFactory2(expected, None, None)

  def matchString(expected: Any, actualToShow: String, expectedToShow: String) = new StringMatcherFactory2(expected, Some(actualToShow), Some(expectedToShow))

  class BinaryMatcherFactory(expected: Array[Byte]) extends Matcher[Array[Byte]] {
    def apply(actual: Array[Byte]): MatchResult = {
      val A = arrToString(actual)
      val B = arrToString(expected)
      MatchResult(
        A == B,
        s"$A did not equal $B",
        "Binaries were identical")
    }

    def arrToString(b: Array[Byte]): String = if (b == null) "null" else java.util.Arrays.asList(b: _*).toString

    override def toString: String = "matchBinary (" + expected + ")"
  }

  class StringMatcherFactory(expected: Any) extends MatcherFactory1[Any, Equality] {
    def matcher[T <: Any: Equality]: Matcher[T] = {
      val equality = implicitly[Equality[T]]
      (actual: T) => {
        MatchResult(
          equality.areEqual(actual, expected),
          s"$actual did not equal $expected",
          "Strings were identical")
      }
    }

    override def toString: String = "matchString (" + expected + ")"
  }

  class StringMatcherFactory2(expected: Any, actualToShow: Option[String], expectedToShow: Option[String]) extends MatcherFactory1[Any, Equality] {
    def matcher[T <: Any: Equality]: Matcher[T] = {
      val equality = implicitly[Equality[T]]
      (actual: T) => {
        MatchResult(
          equality.areEqual(actual, expected),
          s"${actualToShow.getOrElse(actual)} did not equal ${expectedToShow.getOrElse(expected)}",
          "Strings were identical")
      }
    }

    override def toString: String = "matchString (" + expected + ")"
  }

  class JsonMatcher(expected: String) extends Matcher[String] {
    def apply(actual: String): MatchResult = {
      val settings = JsonParserSettings.default.withMaxNumberCharacters(1000)

      val expectedAst = JsonParser(expected, settings)
      val actualAst = JsonParser(actual, settings)
      MatchResult(
        actualAst == expectedAst,
        actual + " did not equal " + expected,
        "JSONs were identical")
    }
  }

  class XmlMatcher(expected: String) extends Matcher[String] {
    def apply(actual: String): MatchResult = {
      val diff: Diff = DiffBuilder.compare(Input.fromString(actual))
        .withTest(Input.fromString(expected))
        .ignoreWhitespace()
        .withNodeFilter(new Predicate[Node]() {
          override def test(n: Node): Boolean = {
            true
          }
        })
        .build()

      MatchResult(
        !diff.hasDifferences,
        actual + " did not equal " + expected,
        "XMLs were identical")
    }
  }

  class PropertiesMatcher(expected: String) extends Matcher[String] {
    def apply(actual: String): MatchResult = {
      val expectedProp = new Properties()
      expectedProp.load(new StringReader(expected))
      val actualProp = new Properties()
      actualProp.load(new StringReader(actual))

      MatchResult(expectedProp.equals(actualProp), actual + " did not equal " + expected, "Properties were identical")
    }
  }

}
