package org.mule.weave.v2.helper

import org.mule.weave.v2.codegen.StringCodeWriter
import spray.json.JsNull
import spray.json.JsString
import spray.json.PrettyPrinter

import scala.collection.mutable.ListBuffer

class JsonBuilder {

  private val stringCodeWriter = new StringCodeWriter()

  val containers: ListBuffer[Container] = ListBuffer()

  def onNewChild(childType: String): Unit = {
    if (containers.nonEmpty) {
      containers.last.onNewChild(childType)
    }
  }

  def startArray(): JsonBuilder = {
    onNewChild("array")
    stringCodeWriter.println("[")
    stringCodeWriter.indent()
    containers.+=(new ArrayContainer)
    this
  }

  def endArray(): JsonBuilder = {
    stringCodeWriter.dedent()
    stringCodeWriter.println()
    stringCodeWriter.printIndent()
    stringCodeWriter.print("]")
    containers.remove(containers.length - 1)
    this
  }

  def startObject(): JsonBuilder = {
    onNewChild("object")
    stringCodeWriter.print("{\n")
    stringCodeWriter.indent()
    containers.+=(new ObjectContainer)
    this
  }

  def endObject(): JsonBuilder = {
    stringCodeWriter.dedent()
    stringCodeWriter.println()
    stringCodeWriter.printIndent()
    stringCodeWriter.print("}")
    containers.remove(containers.length - 1)
    this
  }

  def writeComma(): JsonBuilder = {
    stringCodeWriter.print(",")
    this
  }

  def value(value: String): JsonBuilder = {
    onNewChild("string")
    if (value != null) {
      stringCodeWriter.print(PrettyPrinter(new JsString(value.replaceAll("\\u000a", ""))))
    } else {
      stringCodeWriter.println(PrettyPrinter(JsNull))
    }
    this
  }

  def value(value: Map[String, String]): JsonBuilder = {
    startObject()
    value.foreach((entry) => {
      key(entry._1).value(entry._2)
    })
    endObject()
    this
  }

  def key(key: String): JsonBuilder = {
    onNewChild("key")
    stringCodeWriter.print(PrettyPrinter(new JsString(key.replaceAll("\\u000a", "")))).print(": ")
    this
  }

  def value(value: Int): JsonBuilder = {
    onNewChild("number")
    stringCodeWriter.print(value.toString)
    this
  }

  def value(value: Boolean): JsonBuilder = {
    writeBoolean(value)
  }

  def writeBoolean(value: Boolean): JsonBuilder = {
    onNewChild("boolean")
    stringCodeWriter.print(value.toString)
    this
  }

  override def toString(): String = {
    stringCodeWriter.toString
  }

  trait Container {
    def onNewChild(childType: String): Unit
  }

  class ArrayContainer extends Container {

    var count = 0

    def onNewChild(childType: String) = {
      if (count > 0) {
        stringCodeWriter.println(",")
      }
      count = count + 1
    }
  }

  class ObjectContainer extends Container {

    var count = 0

    def onNewChild(childType: String): Unit = {
      if (count > 0 && childType == "key") {
        stringCodeWriter.println(",")
      }
      count = count + 1
    }
  }

}

