package org.mule.weave.v2.api.tooling.message

import org.mule.weave.v2.api.tooling.location.Location

/**
  * Represents a message. See [[MessageBuilder]] builder
  */
trait Message {

  /**
    * @return The kind of the message.
    */
  def getKind: String

  /**
    * @return The description of the message.
    */
  def getMessage: String
}

/**
  * Builder to create [[Message]] instances.
  */
trait MessageBuilder {

  /**
    * Set the kind for the [[Message]] instance to create (Required!).
    *
    * @param kind The kind to set.
    * @return The builder instance.
    */
  def withKind(kind: String): MessageBuilder

  /**
    * Set the message for the [[Message]] instance to create (Required!).
    *
    * @param message The message to set.
    * @return The builder instance.
    */
  def withMessage(message: String): MessageBuilder

  /**
    * @return The desired [[Message]] instance.
    */
  def build(): Message
}

/**
  * Represents a message referring to a performed validation, with a [[Message]], and a [[Location]]
  */
class ValidationMessage(message: Message, location: Location) {

  /**
    * @return The validation [[Message]].
    */
  def getMessage: Message = {
    this.message
  }

  /**
    * @return The [[Location]] in the script where the validation message happened.
    */
  def getLocation: Location = {
    this.location
  }
}

/**
  * A utility class to capture or retrieve validation messages.
  */
trait MessageCollector {

  /**
    * @return The list of the current [[ValidationMessage]] errors.
    */
  def getErrorValidationMessages: Array[ValidationMessage]

  /**
    * @return The list of the current [[ValidationMessage]] warnings.
    */
  def getWarningValidationMessages: Array[ValidationMessage]

  /**
    * Add an error message in the current context.
    *
    * @param message The error [[ValidationMessage]] to add.
    * @return this message collector instance.
    */
  def addErrorValidationMessage(message: ValidationMessage): MessageCollector

  /**
    * Add a warning message in the current context.
    *
    * @param message The warning [[ValidationMessage]] to add.
    * @return this message collector instance.
    */
  def addWarningValidationMessage(message: ValidationMessage): MessageCollector
}