/**
 * The encodeURI() function encodes a Uniform Resource Identifier (URI) by replacing each instance of certain characters by
 * one, two, three, or four escape sequences representing the UTF-8 encoding of the character
 * (will only be four escape sequences for characters composed of two "surrogate" characters).
 *
 * Assumes that the URI is a complete URI, so does not encode reserved characters that have special meaning in the URI.
 *
 * encodeURI replaces all characters except the following with the appropriate UTF-8 escape sequences:

 * [%header%autowidth.spread]
 * |===
 * | Type                 | Includes
 * | Reserved characters  | ; , / ? : @ & = $
 * | Unescaped characters | alphabetic, decimal digits, - _ . ! ~ * ' ( )
 * | Number sign          | #
 * |===
 */
fun encodeURI(rhs: String): String = native("system::StringUrlEncodeFunctionValue")

/**
 * The decodeURI() function decodes a Uniform Resource Identifier (URI) previously created by encodeURI or by a similar routine.
 * Replaces each escape sequence in the encoded URI with the character that it represents,
 * but does not decode escape sequences that could not have been introduced by encodeURI.
 * The character `#` is not decoded from escape sequences.
 */
fun decodeURI(rhs: String): String = native("system::StringUrlDecodeFunctionValue")

/**
 * The encodeURIComponent() function encodes a Uniform Resource Identifier (URI) component by replacing each instance of certain characters by
 * one, two, three, or four escape sequences representing the UTF-8 encoding of the character
 * (will only be four escape sequences for characters composed of two "surrogate" characters).
 *
 * encodeURIComponent escapes all characters except the following: alphabetic, decimal digits, - _ . ! ~ * ' ( )
 * encodeURIComponent differs from encodeURI in that it encodes reserved characters and the Number sign # of encodeURI:
 *
 * [%header%autowidth.spread]
 * |===
 * | Type                 | Includes
 * | Reserved characters  |
 * | Unescaped characters | alphabetic, decimal digits, - _ . ! ~ * ' ( )
 * | Number sign          |
 * |===
 */
fun encodeURIComponent(rhs: String): String = native("system::StringUrlEncodeComponentFunctionValue")

/**
 * The decodeURIComponent() function decodes a Uniform Resource Identifier (URI) component previously created by
 * encodeURIComponent or by a similar routine.
 */
fun decodeURIComponent(rhs: String): String = native("system::StringUrlDecodeComponentFunctionValue")

type URI = {
  isValid: Boolean,
  host?: String,
  authority?: String,
  fragment?: String,
  path?: String,
  port?: Number,
  query?: String,
  scheme?: String,
  user?: String,
  isAbsolute?: Boolean,
  isOpaque?: Boolean
}

/**
 * Parses an URL and returns an URI object.
 * The `isValid: Boolean` property dennotes if the parse was succeed.
 * Every field in the URI object is optional, and it will be present only if it was present in the original URL
 */
fun parseURI(uri: String): URI = native("system::StringParseUrlFunctionValue")


/**
 * Compose is a custom interpolator used to replace URL components by the encodeURIComponent result of it.
 * Example:
 * [source, dataweave]
 * ----
 *  compose `http://asd/$(' texto a encodear ')/texto a encodear`
 *   ==
 *  'http://asd/%20texto%20a%20encodear%20/texto a encodear'
 * ----
 */
fun compose(parts: Array<String>, interpolation: Array<String>): String =
  parts[0] ++ (interpolation map (encodeURIComponent($) ++ parts[($$ + 1)]) joinBy '')
