/**
* This module contains helper functions for working with binaries.
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::core::Binaries` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
 * Transforms a binary value into the hexadecimal string.
 *
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `content` | The `Binary` value to transform.
 * |===
 *
 * == Example
 *
 * This example transforms a binary version of "Mule" (defined in the variable,
 * `myBinary`) to hexadecimal.
 *
 * === Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Binaries
 * output application/json
 * var myBinary = "Mule" as Binary
 * var testType = typeOf(myBinary)
 * ---
 * {
 *    "binaryToHex" : toHex(myBinary)
 * }
 * ----
 *
 * === Output
 *
 * [source,JSON,linenums]
 * ----
 * { "binaryToHex": "4D756C65" }
 * ----
 */
fun toHex(content: Binary): String = native("system::BinaryToHexFunctionValue")

/**
 * Transforms an hexadecimal string into a binary.
 *
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `hexString` | A hexadecimal string to transform.
 * |===
 *
 * == Example
 *
 * This example transforms a hexadecimal string to "Mule".
 *
 * === Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Binaries
 * output application/json
 * ---
 * { "hexToBinary": fromHex("4D756C65") }
 * ----
 *
 * === Output
 *
 * [source,JSON,linenums]
 * ----
 * { "hexToBinary": "Mule" }
 * ----
 */
fun fromHex(hexString: String): Binary = native("system::HexToBinaryFunctionValue")

/**
 * Transforms a binary value a Base64 string.
 *
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `content` | The binary value to transform.
 * |===
 *
 * == Example
 *
 * This example transforms a binary to Base64.
 *
 * === Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Binaries
 * var myBinary = "Mule" as Binary
 * output application/json
 * ---
 * { "BinaryToBase64" : toBase64(myBinary) }
 * ----
 *
 * === Output
 * [source,JSON,linenums]
 * ----
 *  { "BinaryToBase64": "TXVsZQ==" }
 * ----
 */
fun toBase64(content: Binary): String = native("system::BinaryToBase64FunctionValue")

/**
 * Transforms a Base64 string to binary.
 *
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `base64String` | The Base64 string to transform.
 * |===
 *
 * == Example
 *
 * This example takes a Base64 string and transforms it to binary.
 *
 * === Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Binaries
 * output application/json
 * ---
 * { "BinaryFromBase64" : fromBase64("TXVsZQ==") }
 * ----
 *
 * === Output
 *
 * [source,JSON,linenums]
 * ----
 * { "BinaryFromBase64": "Mule" }
 * ----
 */
fun fromBase64(base64String: String): Binary = native("system::Base64ToBinaryFunctionValue")
