/**
* This module contains functions that allow you to interact with the DataWeave
* engine.
*
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::Runtime` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
 * Throws an exception with the specified message.
 *
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `message` | An error message (`String`).
 * |===
 *
 * === Example
 *
 * This example returns a failure message `Data was empty` because the expression
 * `(sizeOf(myVar) &lt;= 0)` is `true`. A shortened version of the error message
 * is shown in the Output section below.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::Runtime
 * var result = []
 * output application/json
 * ---
 * if(sizeOf(result) <= 0) fail('Data was empty') else result
 * ----
 *
 * ==== Output
 *
 * [source,TXT,linenums]
 * ----
 * ERROR 2018-07-29 11:47:44,983 ...
 * *********************************
 * Message               : "Data was empty
 * ...
 * ----
 */
fun fail (message: String = 'Error'): Nothing = native("system::fail")

/**
 * Produces an error with the specified message if the expression in
 * the evaluator returns `true`, otherwise returns the value.
 *
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `value` | The value to return only if the `evaluator` expression is `false`.
 * | `evaluator` | Expression that returns `true` or `false`.
 * |===
 *
 * === Example
 *
 * This example produces a runtime error (instead of a SUCCESS message) because
 * the expression `isEmpty(result)` is `true`. It is `true` because an empty
 * object is passed through variable `result`.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import failIf from dw::Runtime
 * var result = {}
 * output application/json
 * ---
 * { "result" : "SUCCESS" failIf (isEmpty(result)) }
 * ----
 *
 * ==== Output
 *
 * [source,TXT,linenums]
 * ----
 * ERROR 2018-07-29 11:56:39,988 ...
 * **********************************
 * Message               : "Failed
 * ----
 */
fun failIf <T>(value: T, evaluator: (value: T) -> Boolean, message: String = 'Failed'): T =
    if(evaluator(value)) fail(message) else value

/**
 * Stops the execution for the specified timeout (in milliseconds).
 *
 *
 * WARNING: Stopping the execution will block the thread used, potentially
 * causing slowness, low performance and potentially freezes of the entire
 * runtime. This operation is intented for limited functional testing purposes
 * and should not be used in production application, performance testing or
 * with multiple applications deployed.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `value` | Input of any type.
 * | `timeout` | The number of milliseconds to wait.
 * |===
 *
 * === Example
 *
 * This example waits 2000 milliseconds (2 seconds) to execute.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::Runtime
 * output application/json
 * ---
 * { "user" : 1 } wait 2000
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * { "user": 1 }
 * ----
 */
fun wait <T>(value: T, timeout: Number): T = native("system::wait")


// A type: `TryResult`.
/**
 * Object with a result or error message. If `success` is `false`, it contains
 * the `error`. If `true`, it provides the `result`.
 *
 */
type TryResult<T> = {
  success: Boolean,
  result?: T,
  error?: {
    kind: String,
    message: String,
    stack?: Array<String>,
    location?: String
  }
}

/**
 * Evaluates the delegate function and returns an object with
 * the result or an error message.
 *
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `delegate` | The function to evaluate.
 * |===
 *
 * === Example
 *
 * This example passes the `fail` function as an argument to `try`.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import try, fail from dw::Runtime
 * output application/json
 * ---
 * try(fail)
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *     "success": false,
 *     "error": {
 *     "kind": "UserException",
 *     "message": "Error",
 *     "location": "Unknown location",
 *     "stack": [
 *       "main (anonymous:0:0)"
 *     ]
 *   }
 * }
 * ----
 */
fun try<T>(delegate: () -> T): TryResult<T> = native("system::try")

/**
* Function to be use with try in order to chain multiple `try`
*
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `previous` | Previous try result
* | `orElse` | The next option to try if the previous fails
* |===
*
* === Example
*
* This example waits shows how to chain different try
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* var user = {}
* var otherUser = {}
* output application/json
* ---
* {
*     a: try(() -> user.name!) orElseTry otherUser.name!,
*     b: try(() -> user.name!) orElseTry "No User Name"
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": {
*     "success": false,
*     "error": {
*       "kind": "KeyNotFoundException",
*       "message": "There is no key named 'name'",
*       "location": "\n9|     a: try(() -> user.name!) orElseTry otherUser.name!,\n                                          ^^^^^^^^^^^^^^",
*       "stack": [
*         "main (org::mule::weave::v2::engine::transform:9:40)"
*       ]
*     }
*   },
*   "b": {
*     "success": true,
*     "result": "No User Name"
*   }
* }
* ----
*/
fun orElseTry<T, R>(previous: TryResult<T>, orElse: () -> R): TryResult<T | R> = do {
    if(previous.success)
       previous
    else
       try(orElse)
}

/**
* Returns the result of the `orElse` if the `previous` try result failed if not returns the result of the `previous`
*
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `previous` | Previous try result
* | `orElse` | The next option to try if the previous fails
* |===
*
* === Example
*
* This example waits shows how to chain different try
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* var user = {}
* var otherUser = {name: "DW"}
* output application/json
* ---
* {
*     a: try(() -> user.name!) orElse "No User Name",
*     b: try(() -> otherUser.name) orElse "No User Name"
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": "No User Name",
*   "b": "DW"
* }
* ----
*/
fun orElse<T, R>(previous: TryResult<T>, orElse: () -> R): T | R = do {
    if(previous.success)
       previous.result!
    else
       orElse()
}

/**
* Returns the location string of a given value.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `value` | A value of any type.
* |===
*
* === Example
*
* This example returns the contents of the line (the location) that defines
* variable `a` in the header of the DataWeave script.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* var a = 123
* output application/json
* ---
* locationString(a)
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* "var a = 123"
* ----
*/
fun locationString(value:Any): String = native("system::locationString")


/**
* Returns all the properties configured for Mule runtime.
*
*
* === Example
*
* This example returns all properties from the `java.util.Properties` class.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* output application/dw
* ---
* { "props" : props() }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*  props: {
*    "java.vendor": "Oracle Corporation" as String {class: "java.lang.String"},
*    "sun.java.launcher": "SUN_STANDARD" as String {class: "java.lang.String"},
*    "sun.management.compiler": "HotSpot 64-Bit Tiered Compilers" as String ..., *    "os.name": "Mac OS X" as String {class: "java.lang.String"},
*    "sun.boot.class.path": "/Library/Java/JavaVirtualMachines/ ...,
*    "org.glassfish.grizzly.nio.transport.TCPNIOTransport...": "1048576" ...,
*    "java.vm.specification.vendor": "Oracle Corporation" as String ...,
*    "java.runtime.version": "1.8.0_111-b14" as String {class: "java.lang.String"},
*    "wrapper.native_library": "wrapper" as String {class: "java.lang.String"},
*    "wrapper.key": "XlIl4YartmfEU3oKu7o81kNQbwhveXi-" as String ...,
*    "user.name": "me" as String {class: "java.lang.String"},
*    "mvel2.disable.jit": "TRUE" as String {class: "java.lang.String"},
*    "user.language": "en" as String {class: "java.lang.String"} ...,
*    "sun.boot.library.path": "/Library/Java/JavaVirtualMachines ...
*    "xpath.provider": "com.mulesoft.licm.DefaultXPathProvider" ...,
*    "wrapper.backend": "pipe" as String {class: "java.lang.String"},
*    "java.version": "1.8.0_111" as String {class: "java.lang.String"},
*    "user.timezone": "America/Los_Angeles" as String {class: "java.lang.String"},
*    "java.net.preferIPv4Stack": "TRUE" as String {class: "java.lang.String"},
*    "sun.arch.data.model": "64" as String {class: "java.lang.String"},
*    "java.endorsed.dirs": "/Library/Java/JavaVirtualMachines/...,
*    "sun.cpu.isalist": "" as String {class: "java.lang.String"},
*    "sun.jnu.encoding": "UTF-8" as String {class: "java.lang.String"},
*    "mule.testingMode": "" as String {class: "java.lang.String"},
*    "file.encoding.pkg": "sun.io" as String {class: "java.lang.String"},
*    "file.separator": "/" as String {class: "java.lang.String"},
*    "java.specification.name": "Java Platform API Specification" ...,
*    "java.class.version": "52.0" as String {class: "java.lang.String"},
*    "jetty.git.hash": "82b8fb23f757335bb3329d540ce37a2a2615f0a8" ...,
*    "user.country": "US" as String {class: "java.lang.String"},
*    "mule.agent.configuration.folder": "/Applications/AnypointStudio.app/ ...,
*    "log4j.configurationFactory": "org.apache.logging.log4j.core...",
*    "java.home": "/Library/Java/JavaVirtualMachines/...,
*    "java.vm.info": "mixed mode" as String {class: "java.lang.String"},
*    "wrapper.version": "3.5.34-st" as String {class: "java.lang.String"},
*    "os.version": "10.13.4" as String {class: "java.lang.String"},
*    "org.eclipse.jetty.LEVEL": "WARN" as String {class: "java.lang.String"},
*    "path.separator": ":" as String {class: "java.lang.String"},
*    "java.vm.version": "25.111-b14" as String {class: "java.lang.String"},
*    "wrapper.pid": "5212" as String {class: "java.lang.String"},
*    "java.util.prefs.PreferencesFactory": "com.mulesoft.licm..."},
*    "wrapper.java.pid": "5213" as String {class: "java.lang.String"},
*    "mule.home": "/Applications/AnypointStudio.app/...,
*    "java.awt.printerjob": "sun.lwawt.macosx.CPrinterJob" ...,
*    "sun.io.unicode.encoding": "UnicodeBig" as String {class: "java.lang.String"},
*    "awt.toolkit": "sun.lwawt.macosx.LWCToolkit" ...,
*    "org.glassfish.grizzly.nio.transport...": "1048576" ...,
*    "user.home": "/Users/me" as String {class: "java.lang.String"},
*    "java.specification.vendor": "Oracle Corporation" ...,
*    "java.library.path": "/Applications/AnypointStudio.app/...,
*    "java.vendor.url": "http://java.oracle.com/" as String ...,
*    "java.vm.vendor": "Oracle Corporation" as String {class: "java.lang.String"},
*    gopherProxySet: "false" as String {class: "java.lang.String"},
*    "wrapper.jvmid": "1" as String {class: "java.lang.String"},
*    "java.runtime.name": "Java(TM) SE Runtime Environment" ...,
*    "mule.encoding": "UTF-8" as String {class: "java.lang.String"},
*    "sun.java.command": "org.mule.runtime.module.reboot....",
*    "java.class.path": "%MULE_LIB%:/Applications/AnypointStudio.app...",
*    "log4j2.loggerContextFactory": "org.mule.runtime.module.launcher...,
*    "java.vm.specification.name": "Java Virtual Machine Specification" ,
*    "java.vm.specification.version": "1.8" as String {class: "java.lang.String"},
*    "sun.cpu.endian": "little" as String {class: "java.lang.String"},
*    "sun.os.patch.level": "unknown" as String {class: "java.lang.String"},
*    "com.ning.http.client.AsyncHttpClientConfig.useProxyProperties": "true" ...,
*    "wrapper.cpu.timeout": "10" as String {class: "java.lang.String"},
*    "java.io.tmpdir": "/var/folders/42/dd73l3rx7qz0n625hr29kty80000gn/T/" ...,
*    "anypoint.platform.analytics_base_uri": ...,
*    "java.vendor.url.bug": "http://bugreport.sun.com/bugreport/" ...,
*    "os.arch": "x86_64" as String {class: "java.lang.String"},
*    "java.awt.graphicsenv": "sun.awt.CGraphicsEnvironment" ...,
*    "mule.base": "/Applications/AnypointStudio.app...",
*    "java.ext.dirs": "/Users/staceyduke/Library/Java/Extensions: ..."},
*    "user.dir": "/Applications/AnypointStudio.app/..."},
*    "line.separator": "\n" as String {class: "java.lang.String"},
*    "java.vm.name": "Java HotSpot(TM) 64-Bit Server VM" ...,
*    "org.quartz.scheduler.skipUpdateCheck": "true" ...,
*    "file.encoding": "UTF-8" as String {class: "java.lang.String"},
*    "mule.forceConsoleLog": "" as String {class: "java.lang.String"},
*    "java.specification.version": "1.8" as String {class: "java.lang.String"},
*    "wrapper.arch": "universal" as String {class: "java.lang.String"}
*  } as Object {class: "java.util.Properties"}
* ----
*/
fun props(): Dictionary<String> = native("system::props")

/**
* Returns the value of the property with the specified name or `null` if the
* property is not defined.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `propertyName` | The property to retrieve.
* |===
*
* === Example
*
* This example gets the `user.timezone` property.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* output application/dw
* ---
* { "props" : prop("user.timezone") }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* { props: "America/Los_Angeles" as String {class: "java.lang.String"} }
* ----
*/
fun prop(propertyName: String): String | Null = props()[propertyName]
