/**
* This module contains helper functions to work with Strings.
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::core::Strings` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
* Returns the Unicode for the first character in an input string.
*
*
* For an empty string, the function fails and returns `Unexpected empty string`.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* |===
*
* === Example
*
* This example returns Unicode for the "M" in "Mule".
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "charCode" : charCode("Mule")
* }
* ----
*
* ==== Output
*
* [source,XML,linenums]
* ----
* { "charCode" : 77 }
* ----
*/
fun charCode(text: String): Number = native("system::CharCodeFunctionValue")

/**
* Returns the Unicode for a character at the specified index.
*
*
* This function fails if the index is invalid.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `content` | The input string.
* | `position` | The index (a `Number` type) of a character in the string (as a string array). Note that the index of the first character is `0`.
* |===
*
* === Example
*
* This example returns Unicode for the "u" at index `1` in "MuleSoft".
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "charCodeAt" : charCodeAt("MuleSoft", 1)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* { "charCodeAt": 117 }
* ----
*/
fun charCodeAt(content: String, position: Number): Number =
  charCode(content[position]!)

/**
* Returns a character that matches the specified Unicode.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `charCode` | The input Unicode (a `Number`).
* |===
*
* === Example
*
* This example inputs the Unicode number `117` to return the character "u".
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "fromCharCode" : fromCharCode(117)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* { "fromCharCode": "u" }
* ----
*/
fun fromCharCode(charCode: Number): String = native("system::FromCharCodeFunctionValue")

/**
* Pluralizes a singular string.
*
*
* If the input is already plural (for example, "boxes"), the output will match
* the input.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to pluralize.
* |===
*
* === Example
*
* This example pluralizes the input string "box" to return "boxes".
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
*  { "pluralize" : pluralize("box") }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* { "pluralize" : "boxes" }
* ----
**/
fun pluralize(text: String): String = native("system::StringPluralizeOperator")

// Helper function. Do not publish doc.
/**
* Helper function that allows *pluralize* to work with null values.
*/
fun pluralize(text: Null): Null = null

/**
* Converts a plural string to its singular form.
*
*
* If the input is already singular (for example, "box"), the output will match
* the input.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to convert to singular form.
* |===
*
* === Example
*
* This example converts the input string "boxes" to return "box".
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* { "singularize" : singularize("boxes") }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* { "singularize" : "box" }
* ----
**/
fun singularize(text: String): String = native("system::StringSingularizeFunctionValue")

// Helper function. Do not publish doc.
/**
* Helper function that allows *singularize* to work with null values.
*/
fun singularize(text: Null): Null = null

/**
* Returns a string in camel case based on underscores in the string.
*
*
* All underscores are deleted, including any underscores at the beginning of the string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to convert to camel case.
* |===
*
* === Example
*
* This example converts a string that contains underscores to camel case.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a" : camelize("customer_first_name"),
*   "b" : camelize("_name_starts_with_underscore")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*    "a": "customerFirstName",
*    "b": "nameStartsWithUnderscore"
*  }
* ----
**/
fun camelize(text: String): String = native("system::StringCamelizeFunctionValue")

// Helper function. Do not doc.
/**
* Helper function that allows `camelize` to work with null values.
*/
fun camelize(text: Null): Null = null

/**
* Capitalizes the first letter of each word in a string.
*
*
* It also removes underscores between words and puts a space before each
* capitalized word.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to capitalize.
* |===
*
* === Example
*
* This example capitalizes a set of strings.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a" : capitalize("customer"),
*   "b" : capitalize("customer_first_name"),
*   "c" : capitalize("customer NAME"),
*   "d" : capitalize("customerName")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": "Customer",
*   "b": "Customer First Name",
*   "c": "Customer Name",
*   "d": "Customer Name"
* }
* ----
**/
fun capitalize(text: String): String = native("system::StringCapitalizeFunctionValue")

// Helper function. Do not publish doc.
/**
* Helper function that allows *capitalize* to work with null values.
*/
fun capitalize(text: Null): Null = null

/**
* Returns a number as an ordinal, such as `1st` or `2nd`.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `num` | An input number to return as an ordinal.
* |===
*
* === Example
*
* This example returns a variety of input numbers as ordinals.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a" : ordinalize(1),
*   "b": ordinalize(2),
*   "c": ordinalize(5),
*   "d": ordinalize(103)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*    "a": "1st",
*    "b": "2nd",
*    "c": "5th",
*    "d": "103rd"
* }
* ----
**/
fun ordinalize(num: Number): String = native("system::NumberOrdinalizeFunctionValue")

// Helper function. Do not publish doc.
/**
* Helper function that allows `ordinalize` to work with null values.
*/
fun ordinalize(num: Null): Null = null

/**
* Replaces hyphens, spaces, and camel-casing in a string with underscores.
*
*
* If no hyphens, spaces, and camel-casing are present, the output will match
* the input.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* |===
*
* === Example
*
* This example replaces the hyphens and spaces in the input. Notice that
* the input "customer" is not modified in the output.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*    "a" : underscore("customer"),
*    "b" : underscore("customer-first-name"),
*    "c" : underscore("customer NAME"),
*    "d" : underscore("customerName")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*    "a": "customer",
*    "b": "customer_first_name",
*    "c": "customer_name",
*    "d": "customer_name"
* }
* ----
**/
fun underscore(text: String): String = native("system::StringUnderscoreFunctionValue")

// Helper function. Do not publish doc.
/**
* Helper function that allows *underscore* to work with null values.
*/
fun underscore(text: Null): Null = null

/**
* Replaces spaces, underscores, and camel-casing in a string with dashes
* (hyphens).
*
*
* If no spaces, underscores, and camel-casing are present, the output will
* match the input.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* |===
*
* === Example
*
* This example replaces the spaces, underscores, and camel-casing in the input.
* Notice that the input "customer" is not modified in the output.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a" : dasherize("customer"),
*   "b" : dasherize("customer_first_name"),
*   "c" : dasherize("customer NAME"),
*   "d" : dasherize("customerName")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": "customer",
*   "b": "customer-first-name",
*   "c": "customer-name",
*   "d": "customer-name"
* }
* ----
**/
fun dasherize(text: String): String = native("system::StringDasherizeFunctionValue")

// Helper function. Do not publish doc.
/**
* Helper function that allows *dasherize* to work with null values.
*/
fun dasherize(text: Null): Null = null

/**
* The specified `text` is _left_-padded to the `size` using the `padText`.
* By default `padText` is `" "`.
*
*
* Returns left-padded `String` or original `String` if no padding is necessary.
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `size` | The size to pad to.
* | `padText` | The text to pad with. It defaults to one space if not specified.
* |===
* === Example
*
* This example shows how `leftPad` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*    "a": leftPad(null, 3),
*    "b": leftPad("", 3),
*    "c": leftPad("bat", 5),
*    "d": leftPad("bat", 3),
*    "e": leftPad("bat", -1)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": null,
*   "b": "   ",
*   "c": "  bat",
*   "d": "bat",
*   "e": "bat"
* }
* ----
*/
fun leftPad(text: String, size: Number, padText: String = " "): String = do {
    var actualSize = sizeOf(text)
    ---
    if (actualSize >= size)
        text
    else
        padText repeat (size - actualSize) ++ text
}

/**
* Helper function to make `leftPad` work with `null` value.
*/
fun leftPad(text: Null, size: Number, padText: String = " "): Null = null

/**
* The specified `text` is _right_-padded to the `size` using the `padText`.
* By default `padText` is `" "`.
*
*
* Returns right padded `String` or original `String` if no padding is necessary.
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `size` | The size to pad to.
* | `padText` | The text to pad with. It defaults to one space if not specified.
* |===
* === Example
*
* This example shows how `rightPad` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": rightPad(null, 3),
*   "b": rightPad("", 3),
*   "c": rightPad("bat", 5),
*   "d": rightPad("bat", 3),
*   "e": rightPad("bat", -1)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": null,
*   "b": "   ",
*   "c": "bat  ",
*   "d": "bat",
*   "e": "bat"
* }
* ----
*/
fun rightPad(text: String, size: Number, padChar: String = " "): String =  do {
    var actualSize = sizeOf(text)
    ---
    if (actualSize >= size)
       text
    else
       text ++ (padChar repeat (size - actualSize))
}

/**
* Helper function to make `rightPad` work with `null` value.
*/
fun rightPad(text: Null, size: Number, padText: String = " "): Null = null

/**
* Wraps the specified `text` with the given `wrapper`.
*
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `wrapper` | The content used to wrap.
* |===
*
* === Example
*
* This example shows how `wrapWith` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": wrapWith(null, "'"),
*   "b": wrapWith("", "'"),
*   "c": wrapWith("ab", "x"),
*   "d": wrapWith("'ab'", "'"),
*   "e": wrapWith("ab", "'")
* }
* ----
* ==== Output
* [source,JSON,linenums]
* ----
* {
*   "e": "'ab'",
*   "a": null,
*   "b": "''",
*   "c": "xabx",
*   "d": "''ab''"
* }
* ----
*/
fun wrapWith(text: String, wrapper: String): String =
    "$(wrapper)$(text)$(wrapper)"

/**
* Helper function for `wrapWith` to work with null value.
*/
fun wrapWith(text: Null, wrapper: String): Null = null

/**
*
* Wraps `text` with `wrapper` if that `wrapper` is missing from the start or
* end of the given string.
*
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `wrapper` | The content used to wrap.
* |===
*
* === Example
*
* This example shows how `wrapIfMissing` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
*  {
*    "a": wrapIfMissing(null, "'"),
*    "b": wrapIfMissing("", "'"),
*    "c": wrapIfMissing("ab", "x"),
*    "d": wrapIfMissing("'ab'", "'"),
*    "e": wrapIfMissing("/", '/'),
*    "f": wrapIfMissing("a/b/c", '/'),
*    "g": wrapIfMissing("/a/b/c", '/'),
*    "h": wrapIfMissing("a/b/c/", '/')
*  }
* ----
* ==== Output
* [source,JSON,linenums]
* ----
* {
*    "a": null,
*    "b": "'",
*    "c": "xabx",
*    "d": "'ab'",
*    "e": "/",
*    "f": "/a/b/c/",
*    "g": "/a/b/c/",
*    "h": "/a/b/c/"
*  }
* ----
*/
fun wrapIfMissing(text: String, wrapper: String): String = do {
    var withHead = if(!startsWith(text, wrapper)) "$(wrapper)$(text)" else text
    ---
    if (!endsWith(withHead, wrapper))
        "$(withHead)$(wrapper)"
    else
        withHead
}

/**
* Helper function for `wrapIfMissing` to work with `null` value.
*/
fun wrapIfMissing(text: Null, wrapper: String): Null = text

/**
* Gets the substring before the first occurrence of a separator. The separator
* is not returned.
*
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `separator` | String to search for.
* |===
*
* === Example
*
* This example shows how `substringBefore` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": substringBefore(null, "'"),
*   "b": substringBefore("", "-"),
*   "c": substringBefore("abc", "b"),
*   "d": substringBefore("abc", "c"),
*   "e": substringBefore("abc", "d"),
*   "f": substringBefore("abc", "")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": null,
*   "b": "",
*   "c": "a",
*   "d": "ab",
*   "e": "",
*   "f": ""
* }
* ----
*/
fun substringBefore(text: String, separator: String): String = do {
    var index = (text find separator)[0]
    ---
    index match {
        case is Null -> ""
        case index is Number ->
            text[0 to index - 1]
    }
}

/**
* Helper function for `substringBefore` to work with null value.
*/
fun substringBefore(text: Null, separator: String): Null = null

/**
* Gets the substring before the last occurrence of a separator. The separator
* is not returned.
*
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `separator` | String to search for.
* |===
*
* === Example
*
* This example shows how `substringBeforeLast` behaves with different inputs
* and sizes.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": substringBeforeLast(null, "'"),
*   "b": substringBeforeLast("", "-"),
*   "c": substringBeforeLast("abc", "b"),
*   "d": substringBeforeLast("abcba", "b"),
*   "e": substringBeforeLast("abc", "d"),
*   "f": substringBeforeLast("abc", "")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": null,
*   "b": "",
*   "c": "ab",
*   "d": "abc",
*   "e": "",
*   "f": "a"
* }
* ----
*/
fun substringBeforeLast(text: String, separator: String): String = do {
    var index = (text find separator)[-1]
    ---
    index match {
        case is Null -> ""
        case index is Number ->
            text[0 to index - 1]
    }
}

/**
* Helper function for `substringBeforeLast` to work with null value.
*/
fun substringBeforeLast(text: Null, separator: String): Null = null


/**
* Gets the substring after the first occurrence of a separator. The separator
* is not returned.
*
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `separator` | String to search for.
* |===
*
* === Example
*
* This example shows how `substringAfter` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": substringAfter(null, "'"),
*   "b": substringAfter("", "-"),
*   "c": substringAfter("abc", "b"),
*   "d": substringAfter("abcba", "b"),
*   "e": substringAfter("abc", "d"),
*   "f": substringAfter("abc", "")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*
*   "a": null,
*   "b": "",
*   "c": "c",
*   "d": "cba",
*   "e": "",
*   "f": "bc"
* }
* ----
*/
fun substringAfter(text: String, separator: String): String = do {
  var index = (text find separator)[0]
  ---
  index match {
      case is Null -> ""
      case index is Number ->
          text[index + sizeOf(separator) to -1]
  }
}

/**
* Helper function for `substringAfter` work with null value.
*/
fun substringAfter(text: Null, separator: String): Null = null

/**
* Gets the substring after the last occurrence of a separator. The separator
* is not returned.
*
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `separator` | String to search for.
* |===
*
* === Example
*
* This example shows how `substringAfterLast` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": substringAfterLast(null, "'"),
*   "b": substringAfterLast("", "-"),
*   "c": substringAfterLast("abc", "b"),
*   "d": substringAfterLast("abcba", "b"),
*   "e": substringAfterLast("abc", "d"),
*   "f": substringAfterLast("abc", "")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*  "a": null,
*  "b": "",
*  "c": "c",
*  "d": "a",
*  "e": "",
*  "f": null
* }
* ----
*/
fun substringAfterLast(text: String, separator: String): String = do {
    var index = (text find separator)[-1]
    ---
    if(isEmpty(separator))
        ""
    else
        index match {
            case is Null -> ""
            case index is Number ->
                text[index + sizeOf(separator) to -1]
        }
}

/**
* Helper function for `substringAfterLast` to work with `null` value.
*/
fun substringAfterLast(text: Null, separator: String): Null = null


 /**
 * Repeats a `text` the number of specified `times`.
 *
 *
 * _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * | `times` | Number of times to repeat char. Negative is treated as zero.
 * |===
 *
 * === Example
 *
 * This example shows how `repeat` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": repeat("e", 0),
 *   "b": repeat("e", 3),
 *   "c": repeat("e", -2)
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": "",
 *   "b": "eee",
 *   "c": ""
 * }
 * ----
 */
fun repeat(text: String, times: Number): String =
    if(times <= 0) "" else (1 to times) map text joinBy ""


/**
 * Unwraps a given `text` from a `wrapper` text.
 *
 *
 * _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * | `wrapper` | The text used to unwrap.
 * |===
 *
 * === Example
 *
 * This example shows how `unwrap` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import unwrap from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": unwrap(null, ""),
 *   "b": unwrap(null, '\0'),
 *   "c": unwrap("'abc'", "'"),
 *   "d": unwrap("AABabcBAA", 'A'),
 *   "e": unwrap("A", '#'),
 *   "f": unwrap("#A", '#'),
 *   "g": unwrap("A#", '#')
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": null,
 *   "b": null,
 *   "c": "abc",
 *   "d": "ABabcBA",
 *   "e": "A",
 *   "f": "A#",
 *   "g": "#A"
 * }
 * ----
 */
fun unwrap(text: String, wrapper: String): String = do {
    if(!startsWith(text, wrapper) and !endsWith(text, wrapper))
        text
    else
        text[1 to -2] default text
}

/**
* Helper function to make `unwrap` work with `null` value.
*/
fun unwrap(text: Null, wrapper: String): Null = text

/**
 * Prepends the `prefix` to the beginning of the string if the `text` does not
 * already start with that prefix.
 *
 *
 * _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * | `prefix` | The text to use as prefix.
 * |===
 *
 * === Example
 *
 * This example shows how `prependIfMissing` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import prependIfMissing from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": prependIfMissing(null, ""),
 *   "b": prependIfMissing("abc", ""),
 *   "c": prependIfMissing("", "xyz"),
 *   "d": prependIfMissing("abc", "xyz"),
 *   "e": prependIfMissing("xyzabc", "xyz")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": null,
 *   "b": "abc",
 *   "c": "xyz",
 *   "d": "xyzabc",
 *   "e": "xyzabc"
 * }
 * ----
 */
fun prependIfMissing(text: String, prefix: String): String = do {
    if(!startsWith(text, prefix))
        "$(prefix)$(text)"
    else
        "$(text)"
}

/**
* Helper function to make `prependIfMissing` work with `null` value.
*/
fun prependIfMissing(text: Null, prefix: String): Null = text

/**
 * Appends the `suffix` to the end of the `text` if the `text` does not already
 * ends with the `suffix`.
 *
 *
 * _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * | `suffix` | The text used as the suffix.
 * |===
 *
 * === Example
 *
 * This example shows how `appendIfMissing` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import appendIfMissing from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": appendIfMissing(null, ""),
 *   "b": appendIfMissing("abc", ""),
 *   "c": appendIfMissing("", "xyz") ,
 *   "d": appendIfMissing("abc", "xyz") ,
 *   "e": appendIfMissing("abcxyz", "xyz")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": null,
 *   "b": "abc",
 *   "c": "xyz",
 *   "d": "abcxyz",
 *   "e": "abcxyz"
 * }
 * ----
 */
fun appendIfMissing(text:String, suffix: String): String = do {
    if(!endsWith(text, suffix))
      "$(text)$(suffix)"
    else
      text
}

/**
* Helper function to make `appendIfMissing` work with `null` value.
*/
fun appendIfMissing(text: Null, suffix: String): Null = null

/**
 * Checks if the `text` contains only Unicode digits.
 *
 *
 * A decimal point is not a Unicode digit and returns false.
 * Note that the method does not allow for a leading sign, either positive or
 * negative.
 *
 * _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isNumeric` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import isNumeric from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isNumeric(null),
 *   "b": isNumeric(""),
 *   "c": isNumeric("  "),
 *   "d": isNumeric("123"),
 *   "e": isNumeric("१२३"),
 *   "f": isNumeric("12 3"),
 *   "g": isNumeric("ab2c"),
 *   "h": isNumeric("12-3"),
 *   "i": isNumeric("12.3"),
 *   "j": isNumeric("-123"),
 *   "k": isNumeric("+123")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": false,
 *   "c": false,
 *   "d": true,
 *   "e": true,
 *   "f": false,
 *   "g": false,
 *   "h": false,
 *   "i": false,
 *   "j": false,
 *   "k": false
 * }
 * ----
 */
fun isNumeric(text: String): Boolean = native("system::IsNumericStringFunctionValue")

/**
* Helper function for `isNumeric` so it works with `null` value.
*/
fun isNumeric(text: Null): Boolean = false

/**
 * Checks if the `text` contains only whitespace.
 *
 *
 * _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isWhitespace` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import isWhitespace from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isWhitespace(null),
 *   "b": isWhitespace(""),
 *   "c": isWhitespace("  "),
 *   "d": isWhitespace("abc"),
 *   "e": isWhitespace("ab2c"),
 *   "f": isWhitespace("ab-c")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": true,
 *   "c": true,
 *   "d": false,
 *   "e": false,
 *   "f": false
 * }
 * ----
 */
fun isWhitespace(text: String): Boolean = native("system::IsWhitespaceStringFunctionValue")

/**
* Helper function for `isWhitespace` so it works with `null` value.
*/
fun isWhitespace(text: Null): Boolean = false

/**
 * Checks if the `text` contains only Unicode digits. A decimal point is not a Unicode digit and returns `false`.
 *
 *
 * Note that the method does not allow for a leading sign, either positive or negative.
 *
 * _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isNumeric` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import isAlpha from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isAlpha(null),
 *   "b": isAlpha(""),
 *   "c": isAlpha("  "),
 *   "d": isAlpha("abc"),
 *   "e": isAlpha("ab2c"),
 *   "f": isAlpha("ab-c")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": false,
 *   "c": false,
 *   "d": true,
 *   "e": false,
 *   "f": false
 * }
 * ----
 */
fun isAlpha(text: String): Boolean  = native("system::IsAlphaStringFunctionValue")

/**
* Helper function for `isAlpha` so it works with `null` value.
*/
fun isAlpha(text: Null): Boolean = false

/**
 * Checks if the `text` contains only Unicode letters or digits.
 *
 *
 * _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isAlphanumeric` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import isAlphanumeric from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isAlphanumeric(null),
 *   "b": isAlphanumeric(""),
 *   "c": isAlphanumeric("  "),
 *   "d": isAlphanumeric("abc"),
 *   "e": isAlphanumeric("ab c"),
 *   "f": isAlphanumeric("ab2c"),
 *   "g": isAlphanumeric("ab-c")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": false,
 *   "c": false,
 *   "d": true,
 *   "e": false,
 *   "f": true,
 *   "g": false
 * }
 * ----
 */
fun isAlphanumeric(text: String): Boolean  = native("system::IsAlphaNumericStringFunctionValue")

/**
* Helper function for `isAlphanumeric` so it works with `null` value.
*/
fun isAlphanumeric(text: Null): Boolean  = false

/**
 * Checks if the `text` contains only uppercase characters.
 *
 *
 * _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isNumeric` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import isUpperCase from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isUpperCase(null),
 *   "b": isUpperCase(""),
 *   "c": isUpperCase("  "),
 *   "d": isUpperCase("ABC"),
 *   "e": isUpperCase("aBC"),
 *   "f": isUpperCase("A C"),
 *   "g": isUpperCase("A1C"),
 *   "h": isUpperCase("A/C")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": false,
 *   "c": false,
 *   "d": true,
 *   "e": false,
 *   "f": false,
 *   "g": false,
 *   "h": false
 * }
 * ----
 */
fun isUpperCase(text: String): Boolean  = native("system::IsUpperCaseStringFunctionValue")

/**
* Helper function for `isLowerCase` so that it works with `null` value.
*/
fun isUpperCase(text: Null): Boolean  = false
/**
 * Checks if the `text` contains only lowercase characters.
 *
 *
 * _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isNumeric` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import isLowerCase from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isLowerCase(null),
 *   "b": isLowerCase(""),
 *   "c": isLowerCase("  "),
 *   "d": isLowerCase("abc"),
 *   "e": isLowerCase("aBC"),
 *   "f": isLowerCase("a c"),
 *   "g": isLowerCase("a1c"),
 *   "h": isLowerCase("a/c")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": false,
 *   "c": false,
 *   "d": true,
 *   "e": false,
 *   "f": false,
 *   "g": false,
 *   "h": false
 * }
 * ----
 */
fun isLowerCase(text: String): Boolean  = native("system::IsLowerCaseStringFunctionValue")

/**
* Helper function for `isLowerCase` so it works with `null` value.
*/
fun isLowerCase(text: Null): Boolean  = false
