/**
* This utility module provides functions that enable you to handle values
* as though they are tree data structures.
*
* The module is included with Mule runtime. To use it, you must import it into
* your DataWeave code, for example, by adding the line
* `import * from dw::util::Tree` to the header of your script.
*
* _Introduced in DataWeave 2.2.2. Supported by Mule 4.2.2 and later._
*
*/
%dw 2.0

@Since(version = "2.2.2")
var OBJECT_TYPE = "Object"

@Since(version = "2.2.2")
var ATTRIBUTE_TYPE = "Attribute"

@Since(version = "2.2.2")
var ARRAY_TYPE = "Array"

/**
* Represents an array of `PathElement` types that identify the location of a node in a tree.
*/
@Since(version = "2.2.2")
type Path = Array<PathElement>

/**
* Represents a specific selection of a node in a path.
*/
@Since(version = "2.2.2")
type PathElement = {|
        kind: "Object" | "Attribute" | "Array" ,
        selector: String | Number,
        namespace: Namespace | Null
    |}


/**
* Transforms a path to a string representation.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | path | The path to transform to a string.
* |===
*
* === Example
*
* This example transforms a path to a string representation.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Tree
* output application/json
* ---
* asExpressionString([{kind: OBJECT_TYPE, selector: "user", namespace: null}, {kind: ATTRIBUTE_TYPE, selector: "name", namespace: null}])
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* ".user.@name"
* ----
**/
@Since(version = "2.2.2")
fun asExpressionString(path: Path): String =
    path reduce ((item, accumulator = "") -> do {
        var selectorExp = item.kind match {
            case "Attribute" -> ".@$(item.selector as String)"
            case "Array" -> "[$(item.selector as String)]"
            case "Object" -> ".$(item.selector as String)"
        }
        ---
        if(isEmpty(accumulator))
            selectorExp
        else
           accumulator ++ selectorExp
     })

/**
* Returns true if the provided path is an Object Type expression
*
* _Introduced in DataWeave 2.3.1. Supported by Mule 4.3.1 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | path | The path to be validated
* |===
*
* === Example
*
* This example shows how the `isAttributeType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Tree
* output application/json
* ---
* {
*   a: isAttributeType([{kind: OBJECT_TYPE, selector: "user", namespace: null}, {kind: ATTRIBUTE_TYPE, selector: "name", namespace: null}]),
*   b: isAttributeType([{kind: OBJECT_TYPE, selector: "user", namespace: null}, {kind: OBJECT_TYPE, selector: "name", namespace: null}])
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   a: true,
*   b: false
* }
* ----
**/
@Since(version = "2.3.1")
fun isAttributeType(path: Path): Boolean = do {
    path[-1].kind == ATTRIBUTE_TYPE
}

/**
* Returns true if the provided path is an Object Type expression
*
* _Introduced in DataWeave 2.3.1. Supported by Mule 4.3.1 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | path | The path to be validated
* |===
*
* === Example
*
* This example shows how the `isAttributeType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::utils::Tree
* output application/json
* ---
* {
*   a: isObjectType([{kind: OBJECT_TYPE, selector: "user", namespace: null}, {kind: ATTRIBUTE_TYPE, selector: "name", namespace: null}]),
*   b: isObjectType([{kind: OBJECT_TYPE, selector: "user", namespace: null}, {kind: OBJECT_TYPE, selector: "name", namespace: null}])
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   a: false,
*   b: true
* }
* ----
**/
@Since(version = "2.3.1")
fun isObjectType(path: Path): Boolean = do {
    path[-1].kind == OBJECT_TYPE
}

/**
* Returns true if the provided path is an Array Type expression
*
* _Introduced in DataWeave 2.3.1. Supported by Mule 4.3.1 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | path | The path to be validated
* |===
*
* === Example
*
* This example shows how the `isAttributeType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::utils::Tree
* output application/json
* ---
* {
*   a: isArrayType([{kind: OBJECT_TYPE, selector: "user", namespace: null}, {kind: ATTRIBUTE_TYPE, selector: "name", namespace: null}]),
*   b: isArrayType([{kind: OBJECT_TYPE, selector: "user", namespace: null}, {kind: ARRAY_TYPE, selector: 0, namespace: null}])
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   a: false,
*   b: true
* }
* ----
**/
@Since(version = "2.3.1")
fun isArrayType(path: Path): Boolean = do {
    path[-1].kind == ARRAY_TYPE
}


/**
* Maps the terminal (leaf) nodes in the tree.
*
*
* Leafs nodes cannot have an object or an array as a value.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `value` | The value to map.
* | `callback` | The mapper function.
* |===
*
* === Example
*
* This example transforms all the string values to upper case.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* %dw 2.0
*  import * from dw::util::Tree
*   output application/json
*   ---
*  {
*      user: [{
*          name: "mariano",
*          lastName: "achaval"
*      }],
*      group: "data-weave"
*  } mapLeafValues (value, path) -> upper(value)
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "user": [
*      {
*        "name": "MARIANO",
*        "lastName": "ACHAVAL"
*      }
*    ],
*    "group": "DATA-WEAVE"
*  }
* ----
**/
@Since(version = "2.2.2")
fun mapLeafValues(value: Any, callback: (value: Any, path: Path) -> Any): Any = do {

    fun doMapAttributes(key: Key, path: Path, callback: (value: Any, path: Path) -> Any) =
        key.@ mapObject (value, key) -> {
            (key) : doMapChildValues(value, path << {kind: ATTRIBUTE_TYPE, selector: key as String, namespace: key.#}, callback)
        }

    fun doMapChildValues(value: Any, path: Path, callback: (value: Any, path: Path) -> Any) = do {
        value match {
            case obj is  Object -> obj mapObject ((value, key, index) -> {
                (key)
                    @((doMapAttributes(key,path,callback))):
                        doMapChildValues(value, path << {kind: OBJECT_TYPE, selector: key as String, namespace: key.#}, callback)
            })
            case arr is Array -> arr map ((item, index) -> doMapChildValues(item, path << {kind: ARRAY_TYPE, selector: index, namespace: null}, callback))
            else -> callback(value, path)
        }
    }
    ---
    doMapChildValues(value, [], callback)
}


/**
* Go through the nodes of the value and allows to filter them.
* The `criteria` is going to be called with each value and its path.
* If it returns true the node will remain if false it will be filtered.
*
* _Introduced in DataWeave 2.3.1. Supported by Mule 4.3.1 and later._
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value | The value to be filtered
* | criteria | The expression that will determine if the node is filtered or not.
* |===
*
* === Example
*
* This example shows how the `filterTree` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
*  %dw 2.0
*   import * from dw::util::Tree
*   output application/json
*   ---
*   {
*     a: {name : "", lastName @(foo: ""): "Achaval", friends @(id: 123): [{id: "", test: true}, {age: 123}, ""]} filterTree ((value, path) ->
*       value match  {
*         case s is String -> !isEmpty(s)
*         else -> true
*       }
*     ),
*     b: null filterTree ((value, path) -> value is String),
*     c: [{name: "Mariano", friends: []}, {test: [1,2,3]}, {dw: ""}] filterTree ((value, path) -> value match  {
*       case a is Array ->  !isEmpty(a as Array)
*       else -> true
*     }),
*
*   }
*
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
* {
*    a: {
*      lastName: "Achaval",
*      friends @(id: 123): [
*        {
*          test: true
*        },
*        {
*          age: 123
*        }
*      ]
*    },
*    b: null,
*    c: [
*      {
*        name: "Mariano"
*      },
*      {
*        test: [
*          1,
*          2,
*          3
*        ]
*      },
*      {
*        dw: ""
*      }
*    ]
*  }
* ----
**/
@Since(version = "2.3.1")
fun filterTree(value: Any, criteria: (value: Any, path: Path) -> Boolean): Any = do {

    fun doFilterAttributes(key: Key, path: Path, callback: (value: Any, path: Path) -> Boolean): Object =
        (key.@ default {}) filterObject  ((value, key) ->
            callback(value, path << {kind: ATTRIBUTE_TYPE, selector: key as String, namespace: key.#})
        )

    fun doFilter(value: Any, path: Path, callback: (value: Any, path: Path) -> Boolean): Any = do {
        value match {
            case obj is Object -> do {
                obj
                    filterObject ((value, key, index) ->
                        callback(value, path << {kind: OBJECT_TYPE, selector: key as String, namespace: key.#})
                    )
                    mapObject ((value, key, index) -> {
                        (key) @((doFilterAttributes(key,path << {kind: OBJECT_TYPE, selector: key as String, namespace: key.#},callback))):
                                doFilter(value, path << {kind: OBJECT_TYPE, selector: key as String, namespace: key.#}, callback)
                               }
                    )

            }
            case arr is Array -> do {
                arr
                    filter ((item, index) -> callback(item, path << {kind: ARRAY_TYPE, selector: index, namespace: null}))
                    map ((item, index) ->
                        doFilter(item, path << {kind: ARRAY_TYPE, selector: index, namespace: null}, callback)
                    )
            }
            else -> value
        }
    }
    ---
    doFilter(value, [], criteria)
}


/**
* This function filter only the Object Leaf values.
* So the criteria is only going to be applied to Object values and Attributes which values are
* either `SimpleType` or `Null`
*
*
* _Introduced in DataWeave 2.3.1. Supported by Mule 4.3.1 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value | The value to be used
* | criteria | The criteria to be used to filter the Object values.
* |===
*
* === Example
*
* This example shows how the `filterObjectLeafs` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* %dw 2.0
*  import * from dw::util::Tree
*  output application/json
*  ---
*  {
*    a: {
*        name: "Mariano",
*        lastName: null,
*        age: 123,
*        friends: [{name @(mail: "mariano.achaval@gmail.com", test:123 ): "", id:"test"}, {name: "Mariano", id:null}]
*       }  filterObjectLeafs ((value, path) -> !(value is Null or value is String)),
*    b: null filterObjectLeafs ((value, path) -> !(value is Null or value is String))
*  }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   a: {
*     age: 123,
*     friends: [
*       {},
*       {}
*     ]
*   },
*   b: null
* }
* ----
**/
@Since(version = "2.3.1")
fun filterObjectLeafs(value: Any, criteria: (value: Any, path: Path) -> Boolean): Any = do {
    (value filterTree ((value, path) -> do {
        if((isObjectType(path) or isAttributeType(path)) and value is (SimpleType | Null))
            criteria(value, path)
        else
            true
    }))
}

/**
*
* This function filter only the Array Leaf values.
* So the criteria is only going to be applied to Arrays values which values are
* either `SimpleType` or `Null`
*
*
* _Introduced in DataWeave 2.3.1. Supported by Mule 4.3.1 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value | The value to be filtered
* | criteria | The criteria to be used to filter the arrays
* |===
*
* === Example
*
* This example shows how the `filterArrayLeafs` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Tree
* output application/json
* ---
* {
*    a: [1,2, {name: ["", true], test: 213}, "123"] filterArrayLeafs ((value, path) -> !(value is Null or value is String)),
*    b: null filterArrayLeafs ((value, path) -> !(value is Null or value is String))
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   a: [
*     1,
*     2,
*     {
*       name: [
*         true
*       ],
*       test: 213
*     }
*   ],
*   b: null
* }
* ----
**/
@Since(version = "2.3.1")
fun filterArrayLeafs(value: Any, criteria: (value: Any, path: Path) -> Boolean): Any = do {
    (value filterTree ((value, path) -> do {
        if((isArrayType(path)) and value is (SimpleType | Null))
            criteria(value, path)
        else
            true
    }))
}


/**
* Returns `true` if any node in the tree validates against the specified criteria.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `value` | The value to search.
* | `callback` | The criteria.
* |===
*
* === Example
*
* This example checks for any user with the name `peter`.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
** %dw 2.0
*  import * from dw::util::Tree
*   output application/json
*   ---
*  {
*      user: [{
*          name: "mariano",
*          lastName: "achaval",
*          friends: [
*              {
*                  name: "julian"
*              },
*              {
*                  name: "tom"
*              }
*          ]
*      },
*      {
*          name: "leandro",
*          lastName: "shokida",
*          friends: [
*              {
*                  name: "peter"
*              },
*              {
*                  name: "robert"
*              }
*          ]
*
*      }
*      ],
*
*  } nodeExists ((value, path) -> path[-1].selector == "name" and value == "peter")
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* true
* ----
**/
@Since(version = "2.2.2")
fun nodeExists(value: Any, callback: (value: Any, path: Path) -> Boolean):Boolean = do {

    fun existsInObject(value:Object,path: Path, callback: (value: Any, path: Path) -> Boolean):Boolean = do {
        value match {
            case {k:v ~ xs} -> do{
                var currentPath = path << {kind: OBJECT_TYPE, selector: k as String, namespace: k.#}
                var exists = callback(v, currentPath) or treeExists(v, currentPath, callback) or existsInAttribute(k.@ default {}, currentPath, callback)
                ---
                if(exists)
                    true
                else
                   treeExists(xs, path, callback)
            }
            case {} -> false
        }
    }

    fun existsInAttribute(value:Object, path: Path, callback: (value: Any, path: Path) -> Boolean):Boolean = do {
        value match {
            case {k:v ~ xs} -> do{
                var currentPath = path << {kind: ATTRIBUTE_TYPE, selector: k as String, namespace: k.#}
                var exists = callback(v, currentPath)
                ---
                if(exists)
                    true
                else
                   treeExists(xs, path, callback)
            }
            case {} -> false
        }
    }

    fun existsInArray(value:Array,path: Path,callback: (value: Any, path: Path) -> Boolean, index: Number):Boolean = do {
        value match {
            case [v ~ xs] -> do{
                var currentPath = path << {kind: ARRAY_TYPE, selector: index, namespace: null}
                var exists = callback(v, currentPath) or treeExists(v, currentPath, callback)
                ---
                if(exists)
                    true
                else
                   existsInArray(xs, path, callback, index + 1)
            }
            case [] -> false
        }
    }

    fun treeExists(value: Any, path: Path, callback: (value: Any, path: Path) -> Boolean):Boolean = do {
        value match {
            case obj is  Object -> existsInObject(obj,path,callback)
            case arr is Array -> existsInArray(arr,path,callback, 0)
            else -> callback(value, path) as Boolean
        }
    }
    ---
    treeExists(value, [], callback)
}
