
/**
* This module contains type conversions that allows to format, parse and transform values.
* 
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*/
@Since(version = "2.4.0")
%dw 2.0

type RoundingMode = "UP"| "DOWN" | "CEILING" | "FLOOR" | "HALF_UP" | "HALF_DOWN" | "HALF_EVEN"

type MillisOrSecs = "milliseconds" | "seconds"

type PeriodUnits = "hours" |  "minutes" | "seconds" | "milliseconds" | "nanos"


/**
* Formats a Number and return a String with the result.
* A Pattern can be used in order to specify how the number needs to be formatted.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | number | The number to be formatted
* | format | The format to be used
* | locale | The locale
* | roundMode | "UP" or "DOWN" or "CEILING" or "FLOOR" or "HALF_UP" or "HALF_DOWN" or "HALF_EVEN"
* |===
*
* === Example
*
* This example shows how the `toString` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toString from dw::util::Coercions
* output application/json
* ---
*  {
*     a: toString(1.0),
*     b: toString(0.005,".00"),
*     c: toString(0.035,"#.##","ES"),
*     d: toString(0.005,"#.##","ES","HALF_EVEN"),
*     e: toString(0.035,"#.00",null,"HALF_EVEN"),
*  }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "1",
*   "b": ".01",
*   "c": "0,04",
*   "d": "0",
*   "e": ".04"
* }
* ----
**/
@Labels(labels = ["format"])
@Since(version = "2.4.0")
fun toString(number: Number, format: String | Null = null, locale: String | Null = null, roundMode: RoundingMode | Null = null): String =
    number as String {
                        (format: format) if (format != null),
                        (locale: locale) if (locale != null),
                        (roundMode: roundMode) if (roundMode != null)
                    }

/**
* Formats a Date DateTime Time into a String.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | date   | The date to be formatted
* | format | The format to be used to format the date. If null no format will be used
* | locale | The locale to be used. If null no locale will be used
* |===
*
* === Example
*
* This example shows how the `toString` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toString from dw::util::Coercions
* output application/json
* ---
* {
*    a: toString(|2015-10-01T23:57:59|),
*    b: toString(|2003-10-01T23:57:59|, "yyyy-MM-dd HH:mm:ss a"),
*    c: toString(|2003-10-01|, "yyyy/MM/dd"),
*    d: toString(|23:57:59|, "HH-mm-ss"),
*    e: toString(|2003-10-01T23:57:59|, "yyyy-MM-dd HH:mm:ss a", "ES"),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "2015-10-01T23:57:59",
*   "b": "2003-10-01 23:57:59 PM",
*   "c": "2003/10/01",
*   "d": "23-57-59",
*   "e": "2003-10-01 23:57:59 PM"
* }
* ----
**/
@Since(version = "2.4.0")
fun toString(date: Date | DateTime | LocalDateTime | LocalTime | Time, format: String | Null = null, locale: String | Null = null): String =
      date as String {
                        (format: format) if (format != null),
                        (locale: locale) if (locale != null)
                     }
/**
* Transforms a Binary data into a String using the given encoding
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | binary | The binary data
* | encoding | The encoding to be used
* |===
*
* === Example
*
* This example shows how the `toString` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* var binaryData= "DW Test" as Binary {encoding: "UTF-32"}
* ---
* {
*   a: toString(binaryData, "UTF-32"),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "DW Test"
* }
* ----
**/
@Since(version = "2.4.0")
fun toString(binary: Binary, encoding: String): String =
    binary as String {
                        encoding: encoding
                     }

/**
* Transform the value (TimeZone | Uri | Boolean| Period| Regex| Key) into the String representation
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | data | The data to be transformed
* |===
*
* === Example
*
* This example shows how the `toString` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*  a: toString(|Z|),
*  b: toString(true),
*  c: toString(|P1D|),
*  d: toString(/a-Z/),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "Z",
*   "b": "true",
*   "c": "P1D",
*   "d": "a-Z"
* }
* ----
**/
@Since(version = "2.4.0")
fun toString(data: TimeZone | Uri | Boolean| Period| Regex| Key): String =
      data as String

/**
* Transform an array of characters into a String, by joining them.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | arr | The array to transform
* |===
*
* === Example
*
* This example shows how `toString` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toString from dw::util::Coercions
* output application/json
* ---
* {
*  a: toString([]),
*  b: toString(["h", "o", "l", "a"]),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "",
*   "b": "hola",
* }
* ----
**/
@Since(version = "2.4.0")
fun toString(arr: Array<String>): String = arr joinBy ""

/**
* Transform a string into an array of characteres, splitting every character of the string.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | text | The string to transform
* |===
*
* === Example
*
* This example shows how `toArray` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toString from dw::util::Coercions
* output application/json
* ---
* {
*  a: toArray(""),
*  b: toArray("hola"),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": [],
*   "b": ["h", "o", "l", "a"],
* }
* ----
**/
@Since(version = "2.4.0")
fun toArray(@StreamCapable text: String): Array<String> = text reduce ((item, acc=[]) -> acc ++ [item])

/**
* Transforms a DateTime into a the Number of seconds or milliseconds depending in the specified unit.
* 
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | dateTime | The datetime to be used
* | unit | The unit ("milliseconds" | "seconds") if null "seconds" will be used
* |===
*
* === Example
*
* This example shows how the `toNumber` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toNumber from dw::util::Coercions
* output application/json
* ---
* {
*     epoch: toNumber(|2015-10-01T23:57:59Z|),
*     millis: toNumber(|2015-10-01T23:57:59Z|, "milliseconds"),
* }
*
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
*  {
*     epoch: 1443743879,
*     millis: 1443743879000
*  }
* ----
**/
@Labels(labels = ["epoch", "currentTimeMillis"])
@Since(version = "2.4.0")
fun toNumber(dateTime: DateTime, unit: MillisOrSecs | Null = null): Number =
     dateTime as Number {
                           (unit: unit) if(unit != null)
                        }
/**
* Transforms the given Period into a Number using the specified unit : "hours",  "minutes" , "seconds" , "milliseconds" or "nanos"
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | period | The period to be converted
* | unit | The unit to be used
* |===
*
* === Example
*
* This example shows how the `toNumber` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toNumber from dw::util::Coercions
* output application/json
* ---
* {
*     toSeconds: toNumber(|PT1H10M|, "seconds")
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*  {
*   "toSeconds": 4200
*  }
* ----
**/
@Since(version = "2.4.0")
fun toNumber(period: Period, unit: PeriodUnits | Null = null): Number =
     period as Number {
                         (unit: unit) if(unit != null)
                      }


/**
* Transforms the specified String into a Number.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value |
* | format |
* | locale |
* |===
*
* === Example
*
* This example shows how the `toNumber` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toNumber from dw::util::Coercions
* output application/json
* ---
*  {
*      "default": toNumber("1.0"),
*      withFormat: toNumber("0.005",".00"),
*      withLocale: toNumber("0,035","#.##","ES"),
*  }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "default": 1.0,
*    "withFormat": 0.005,
*    "withLocale": 0.035
* }
* ----
**/
@Since(version = "2.4.0")
fun toNumber(value: String | Key, format: String | Null = null, locale: String | Null = null  ): Number =
    value as Number {
                      (format: format) if (format != null),
                      (locale: locale) if (locale != null)
                   }

/**
* Transform the given number into a DateTime using the given unit :"milliseconds" or "seconds"
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | number | The number of millis or secs
* | unit | The unit if null "seconds" is the default
* |===
*
* === Example
*
* This example shows how the `toDateTime` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/dw
*
* import toDateTime from dw::util::Coercions
* ---
* {
*     fromEpoch: toDateTime(1443743879),
*     fromMillis: toDateTime(1443743879000, "milliseconds"),
* }
*
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
*  {
*      fromEpoch: |2015-10-01T23:57:59Z|,
*      fromMillis: |2015-10-01T23:57:59Z|
*  }
* ----
**/
@Since(version = "2.4.0")
fun toDateTime(number: Number, unit: MillisOrSecs | Null = null): DateTime =
    number as DateTime {
                          (unit: unit) if(unit != null)
                       }

/**
* Transforms a String into a DateTime using the given format
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str | The string to transform
* | format | The format to be used
* | locale | The locale of the format
* |===
*
* === Example
*
* This example shows how the `toDateTime` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toDateTime from dw::util::Coercions
* output application/dw
* ---
* {
*    a: toDateTime("2015-10-01T23:57:59Z"),
*    b: toDateTime("2003-10-01 23:57:59Z","yyyy-MM-dd HH:mm:ssz"),
* }
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
* {
*    a: |2015-10-01T23:57:59Z|,
*    b: |2003-10-01T23:57:59Z| as DateTime {format: "yyyy-MM-dd HH:mm:ssz"},
*  }
* ----
**/
@Labels(labels = ["parseDateTime"])
@Since(version = "2.4.0")
fun toDateTime(str: String, format: String | Null = null, locale: String | Null = null ): DateTime =
    str as DateTime {
                       (format: format) if (format != null),
                       (locale: locale) if (locale != null)
                    }

/**
* Transforms a String into a LocalDateTime using the given format
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str | The string to transform
* | format | The format to be used
* | locale | The locale of the format
* |===
*
* === Example
*
* This example shows how the `toLocalDateTime` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toLocalDateTime from dw::util::Coercions
* output application/dw
* ---
* {
*
*   a: toLocalDateTime("2015-10-01T23:57:59"),
*   b: toLocalDateTime("2003-10-01 23:57:59","yyyy-MM-dd HH:mm:ss")
* }
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
* {
*    a: |2015-10-01T23:57:59|,
*    b: |2003-10-01T23:57:59| as LocalDateTime {format: "yyyy-MM-dd HH:mm:ss"}
* }
* ----
**/
@Labels(labels = ["parseLocalDateTime"])
@Since(version = "2.4.0")
fun toLocalDateTime(str: String, format: String | Null = null, locale: String | Null = null ): LocalDateTime =
    str as LocalDateTime {
                            (format: format) if (format != null),
                            (locale: locale) if (locale != null)
                         }

/**
* Transforms a String into a Date using the given format
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str | The string to transform
* | format | The format to be used
* | locale | The locale of the format
* |===
*
* === Example
*
* This example shows how the `toDate` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toDate from dw::util::Coercions
* output application/dw
* ---
* {
*
*   a: toDate("2015-10-01"),
*   b: toDate("2003/10/01","yyyy/MM/dd")
* }
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
* {
*    a: |2015-10-01|,
*    b: |2003-10-01| as Date {format: "yyyy/MM/dd"}
*  }
* ----
**/
@Since(version = "2.4.0")
fun toDate(str: String, format: String | Null = null, locale: String | Null = null ): Date =
    str as Date {
                   (format: format) if (format != null),
                   (locale: locale) if (locale != null)
                }



/**
* Transforms a String into a Time using the given format
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str | The string to transform
* | format | The format to be used
* | locale | The locale of the format
* |===
*
* === Example
*
* This example shows how the `toTime` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toTime from dw::util::Coercions
* output application/dw
* ---
* {
*    a: toTime("23:57:59Z"),
* }
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
* {
*    a: |23:57:59Z|
*  }
* ----
**/
@Since(version = "2.4.0")
fun toTime(str: String, format: String | Null = null, locale: String | Null = null ): Time =
    str as Time {
                   (format: format) if (format != null),
                   (locale: locale) if (locale != null)
                }


/**
* Transforms a String into a LocalTime using the given format
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str | The string to transform
* | format | The format to be used
* | locale | The locale of the format
* |===
*
* === Example
*
* This example shows how the `toLocalTime` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toLocalTime from dw::util::Coercions
* output application/dw
* ---
* {
*    a: toLocalTime("23:57:59"),
* }
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
* {
*      a: |23:57:59|
* }
* ----
**/
@Since(version = "2.4.0")
fun toLocalTime(str: String, format: String | Null = null, locale: String | Null = null ): LocalTime =
    str as LocalTime {
                        (format: format) if (format != null),
                        (locale: locale) if (locale != null)
                     }

/**
*  Transform a String into a Period
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str | The string to be transformed
* |===
*
* === Example
*
* This example shows how the `toPeriod` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toPeriod from dw::util::Coercions
* output application/dw
* ---
* {
*   a: toPeriod("P1D"),
*   b: toPeriod("PT1H1M"),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    a: |P1D|,
*    b: |PT1H1M|
*  }
* ----
**/
@Since(version = "2.4.0")
fun toPeriod(str: String): Period =
    str as Period

/**
*  Transforms a String into a Regex
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str | The string to be transformed
* |===
*
* === Example
*
* This example shows how the `toRegex` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/dw
* import toRegex from dw::util::Coercions
* ---
* {
*   a: toRegex("a-Z"),
*   b: toRegex("0-9+")
* }
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
* {
*    a: /a-Z/,
*    b: /0-9+/
* }
* ----
**/
@Since(version = "2.4.0")
fun toRegex(str: String): Regex =
    str as Regex


/**
* Transform a String into a Boolean
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str | The string to be transformed
* |===
*
* === Example
*
* This example shows how the `toBoolean` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import toBoolean from dw::util::Coercions
* ---
* {
*   a: toBoolean("true"),
*   b: toBoolean("false"),
*   c: toBoolean("FALSE"),
*   d: toBoolean("TrUe"),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "a": true,
*    "b": false,
*    "c": false,
*    "d": true
*  }
* ----
**/
@Since(version = "2.4.0")
fun toBoolean(str: String): Boolean =
    str as Boolean


/**
* Transform a String into a TimeZone
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str | The string to be transformed
* |===
*
* === Example
*
* This example shows how the `toBoolean` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/dw
* import toTimeZone from dw::util::Coercions
* ---
* {
*   a: toTimeZone("-03:00"),
*   b: toTimeZone("Z"),
*   c: toTimeZone("America/Argentina/Buenos_Aires"),
* }
*
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
* {
*    a: |-03:00|,
*    b: |Z|,
*    c: |America/Argentina/Buenos_Aires|
*  }
* ----
**/
@Since(version = "2.4.0")
fun toTimeZone(str: String): TimeZone =
    str as TimeZone

/**
* Transforms a String into a URI
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str | The String to be transformed
* |===
*
* === Example
*
* This example shows how the `toUri` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import toUri from dw::util::Coercions
* ---
* {
*   a: toUri("http://google.com")
* }
*
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
* {
*    a: "http://google.com"
* }
* ----
**/
@Since(version = "2.4.0")
fun toUri(str: String): Uri =
  str as Uri

/**
* Transforms a String into a Binary using the given encoding
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | str |
* | encoding |
* |===
*
* === Example
*
* This example shows how the `toBinary` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import toBinary from dw::util::Coercions
* output application/dw
* ---
* {
*   a: toBinary("DW Binary", "UTF-32"),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* ----
**/
@Since(version = "2.4.0")
fun toBinary(str: String, encoding: String): Binary =
  str as Binary {
                   encoding: encoding
                }


