/**
* This module contains Math utility functions
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*/
%dw 2.0

/**
* The constant value of E, the base of the natural logarithms.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*/
@Since(version = "2.4.0")
var E = 2.7182818284590452354

/**
* The constant value of PI, the ratio of the circumference of a circle to its diameter.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*/
@Since(version = "2.4.0")
var PI = 3.14159265358979323846

/**
* Returns the trigonometric sine of an angle.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | angle | an angle, in radians.
* |===
*
* === Example
*
* This example shows how the `sin` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "sin0": sin(0),
*   "sin13": sin(0.13),
*   "sin-1": sin(-1)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*{
*   "sin0": 0.0,
*   "sin13": 0.12963414261969486,
*   "sin-1": -0.8414709848078965
* }
* ----
**/
@Since(version = "2.4.0")
fun sin(angle: Number):Number = native("system::SinFunctionValue")

/**
* Returns the trigonometric cosine of an angle.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | angle | an angle, in radians
* |===
*
* === Example
*
* This example shows how the `cos` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "cos0": cos(0),
*   "cos13": cos(0.13),
*   "cos-1": cos(-1)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "cos0": 1.0,
*   "cos13": 0.9915618937147881,
*   "cos-1": 0.5403023058681398
* }
* ----
**/
@Since(version = "2.4.0")
fun cos(angle: Number):Number = native("system::CosFunctionValue")

/**
* Returns the trigonometric tangent of an angle.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | angle | an angle, in radians
* |===
*
* === Example
*
* This example shows how the `tan` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* %dw 2.0
*  import * from dw::util::Math
*  output application/json
*  ---
*  {
*    "tan0": tan(0),
*    "tan13": tan(0.13),
*    "tan-1": tan(-1)
*  }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "tan0": 0.0,
*    "tan13": 0.13073731800446006,
*    "tan-1": -1.5574077246549023
*  }
* ----
**/
@Since(version = "2.4.0")
fun tan(angle: Number):Number = native("system::TanFunctionValue")

/**
* Returns the arc sine of a value; the returned angle is in the range -pi/2 through pi/2. Special cases:
* - Argument absolute value is greater than 1, then the result is NaN.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | angle | the value whose arc sine is to be returned.
* |===
*
* === Example
*
* This example shows how the `asin` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "asin0": asin(0),
*   "asin13": asin(0.13),
*   "asin-1": asin(-1)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "asin0": 0.0,
*    "asin13": 0.1303689797031455,
*    "asin-1": -1.5707963267948966
*  }
* ----
**/
@Since(version = "2.4.0")
fun asin(angle: Number):Number = native("system::ASinFunctionValue")

/**
* Returns the arc cosine of a value; the returned angle is in the range 0.0 through pi. Special case:
* - Argument absolute value is greater than 1, then the result is NaN.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | angle | the value whose arc cosine is to be returned.
* |===
*
* === Example
*
* This example shows how the `acos` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "acos0": acos(0),
*   "acos13": acos(0.13),
*   "acos-1": acos(-1)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "acos0": 1.5707963267948966,
*    "acos13": 1.440427347091751,
*    "acos-1": 3.141592653589793
*  }
* ----
**/
@Since(version = "2.4.0")
fun acos(angle: Number):Number | NaN = native("system::ACosFunctionValue")

/**
* Returns the arc tangent of a value; the returned angle is in the range -pi/2 through pi/2.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | angle | the value whose arc tangent is to be returned.
* |===
*
* === Example
*
* This example shows how the `atan` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "atan0":  atan(0),
*   "atan13": atan(0.13),
*   "atan-1": atan(-1)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "atan0": 1.5707963267948966,
*   "atan13": 1.440427347091751,
*   "atan-1": 3.141592653589793
* }
* ----
**/
@Since(version = "2.4.0")
fun atan(angle: Number):Number = native("system::ATanFunctionValue")

/**
*
* Converts an angle measured in degrees to an approximately equivalent angle measured in radians
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | angdeg | an angle, in degree
* |===
*
* === Example
*
* This example shows how the `toRadians` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "toRadians10":  toRadians(10),
*   "toRadians013": toRadians(0.13),
*   "toRadians-20": toRadians(-20)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* {
*    "toRadians10": 0.1745329251994329576922222222222222,
*    "toRadians013": 0.002268928027592628449998888888888889,
*    "toRadians-20": -0.3490658503988659153844444444444444
*  }
* ----
**/
@Since(version = "2.4.0")
fun toRadians(angdeg:Number): Number = angdeg / 180.0 * PI

/**
* Converts an angle measured in radians to an approximately equivalent angle measured in degrees.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | angrad | an angle, in radians
* |===
*
* === Example
*
* This example shows how the `toDegrees` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "toDegrees0.17":  toDegrees(0.174),
*   "toDegrees0": toDegrees(0),
*   "toDegrees-20": toDegrees(-0.20)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "toDegrees0.17": 9.969465635276323832571267395889251,
*    "toDegrees0": 0E+19,
*    "toDegrees-20": -11.45915590261646417536927286883822
*  }
* ----
**/
@Since(version = "2.4.0")
fun toDegrees(angrad:Number):Number = angrad * 180.0 / PI

/**
*  Returns the natural logarithm (base e) of a double value. Special cases:
*  - Argument less or equal to zero , then the result is NaN.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | a | a value
* |===
*
* === Example
*
* This example shows how the `logn` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*    "logn10":  logn(10),
*    "logn13": logn(0.13),
*    "logn-20": logn(-20)
*  }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "logn10": 2.302585092994046,
*    "logn13": -2.0402208285265546,
*    "logn-20": null
*  }
* ----
**/
@Since(version = "2.4.0")
fun logn(a: Number): Number | NaN = native("system::LognFunctionValue")

/**
* Returns the base 10 logarithm of a double value
*
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | a | a value
* |===
*
* === Example
*
* This example shows how the `log10` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "log1010":  log10(10),
*   "log1013": log10(0.13),
*   "log10-20": log10(-20)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*  {
*     "log1010": 2.302585092994046,
*     "log1013": -2.0402208285265546,
*     "log10-20": null
*   }
* ----
**/
@Since(version = "2.4.0")
fun log10(a: Number): Number | NaN = native("system::Log10FunctionValue")