/**
*
* This module contains all functions required to create a Data Weave test
*
* .Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::test::Tests
*  ---
*
*  "Matcher api" describedBy [
*      "It should support nested matching" in  do {
*          var payload = {}
*          ---
*          payload must [
*              beObject(),
*              $.foo must [
*                  beNull()
*              ]
*          ]
*      },
*      "It should support simple matching" in do {
*          var payload = {}
*          ---
*          payload must beObject()
*      },
*
*      "It should support multiple root cases" in do {
*          var payload = {}
*          var flowVar = {a: 123}
*          ---
*          [
*              payload must beObject(),
*              flowVar must [
*                  beObject(),
*                  $.a must equalTo(123)
*              ]
*          ]
*      },
*      "It should support using custom assertions" in  do {
*          var payload = []
*          ---
*          payload must sizeOf($) > 2
*      }
*  ]
* ----
*/
%dw 2.0


import * from dw::util::Timer
import some from dw::core::Arrays
import * from dw::test::Asserts
import interceptor from dw::test::internal::Functions
import red,green from dw::test::internal::Ansi

var ERROR_STATUS = "ERROR"
var OK_STATUS = "OK"
var FAIL_STATUS = "FAIL"

/**
* Data Type that describes the result of a Test Execution
*/
type TestResult = {
    name: String,
    time: Number,
    status: String,
    tests?: Array<TestResult>,
    errorMessages?: Array<String>
}

/**
* Defines a new test case inside a test suite with a single assertion.
*
* .Example
*
* [source,DataWeave,linenums]
* ----
* "It should support nested matching" in  do {
*    "foo" must beString()
* }
* ---
*/
fun in(testName:String, testCases: () -> MatcherResult): TestResult = do {
  in(testName, [testCases])
}


/**
* Defines a new test case with multiple assertions
*
*
* .Example
*
* [source,DataWeave,linenums]
* ----
*  "It should support multiple root cases" in do {
*      var payload = {}
*      var flowVar = {a: 123}
*      ---
*     [
*         payload must beObject(),
*         flowVar must [
*              beObject(),
*              $.a must equalTo(123)
*           ]
*       ]
*  }
* ----
*/
fun in(testName:String, callback: Array<() -> MatcherResult>): TestResult = do {
    var OK_RESULT = {name: testName, status: OK_STATUS, time: 0}

    fun doIn(testName:String, testCases: () -> MatcherResult) = do {
      //Unicode chars for helping visualizing
        var CHECK = "\u2713"
        var CROSS = "\u274C"
        var WARNING = "\u26A0"

        var testWithTime = duration(() -> dw::Runtime::try(testCases))
        var timeExecution = testWithTime.time
        var testExecution = testWithTime.result
        var result =
            if(testExecution.success)
                {
                   name: testName,
                   time: timeExecution,
                   status: if(testExecution.result.matches!) OK_STATUS else FAIL_STATUS,
                   errorMessage:
                    if(testExecution.result.reasons?)
                        testExecution.result.reasons! joinBy "\n"
                    else do {
                         var description = testExecution.result.description
                         ---
                         "Expecting $(description.expected) but was $(description.actual)."
                    }
                }
            else
              {
                name: testName,
                status: ERROR_STATUS,
                time: timeExecution,
                errorMessage: testExecution.error.message! ++
                                 if(testExecution.error.location?) (" at:\n" ++ testExecution.error.location!) else "" ++
                                 if(testExecution.error.stack?) ("\n" ++ testExecution.error.stack! reduce ($$ ++ "\n" ++ "$")) else " NO STACK "
              }
        ---
         result
    }
    ---
    interceptor(
                () -> log("wtf", {event: "TestStarted", name: testName}),
                () -> do {
                    callback reduce (assertion, result = OK_RESULT) ->
                        if(result.status == OK_STATUS)
                            doIn(testName, assertion)
                        else
                            result
                },
                (result) -> log("wtf", result ++ {event: "TestFinished"}))

}

/**
* Defines a new test suite with the list of test cases.
*
* .Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::test::Tests
*  ---
*
*  "Matcher api" describedBy [
*      "It should support nested matching" in  do {
*          var payload = {}
*          ---
*          payload must [
*              beObject(),
*              $.foo must [
*                  beNull()
*              ]
*          ]
*      },
* ]
* ----
*/
fun describedBy(suite: String, testsToRun:Array<() -> TestResult> ): TestResult = do {
    var measuredResult = interceptor(
                                        () -> log("wtf", {event: "TestSuiteStarted", name:suite}),
                                        () -> duration(() -> testsToRun reduce ((test, acc = []) -> acc + test())),
                                        (result) -> log("wtf", {event: "TestSuiteFinished", name:suite})
                                    )
    var testResults = measuredResult.result
    var testExecutionTime = measuredResult.time
    ---
      {
        name: suite,
        time: testExecutionTime,
        status: OK_STATUS,
        tests: testResults
      }
}

