/**
*
* This module contains all functions required to create a Data Weave test
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::test::Tests
* ---
*  "Matcher api" describedBy [
*      "It should support nested matching" in  do {
*          var payload = {}
*          ---
*          payload must [
*              beObject(),
*              $.foo must [
*                  beNull()
*              ]
*          ]
*      },
*      "It should support simple matching" in do {
*          var payload = {}
*          ---
*          payload must beObject()
*      },
*
*      "It should support multiple root cases" in do {
*          var payload = {}
*          var flowVar = {a: 123}
*          ---
*          [
*              payload must beObject(),
*              flowVar must [
*                  beObject(),
*                  $.a must equalTo(123)
*              ]
*          ]
*      },
*      "It should support using custom assertions" in  do {
*          var payload = []
*          ---
*          payload must sizeOf($) > 2
*      }
*  ]
* ----
*/
%dw 2.0


import * from dw::util::Timer
import some from dw::core::Arrays
import * from dw::test::Asserts
import interceptor from dw::test::internal::Functions
import red,green from dw::test::internal::Ansi
import location from dw::Runtime

var ERROR_STATUS = "ERROR"
var OK_STATUS    = "OK"
var FAIL_STATUS  = "FAIL"
var SKIP_STATUS  = "SKIP"

/**
* Data Type that describes the result of a Test Execution
*/
type TEST_STATUS = "ERROR" | "OK" | "FAIL" | "SKIP"

type TestResult = {
    name: String,
    time: Number,
    status: TEST_STATUS,
    tests?: Array<TestResult>,
    errorMessage?: String
}

/**
* Defines a new test case inside a test suite with a single assertion.
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* "It should support nested matching" in  do {
*    "foo" must beString()
* }
* ----
*/
fun in(testName: String, testCases: () -> MatcherResult): TestResult = do {
  in(testName, [testCases])
}


/**
* Defines a new test case with multiple assertions
*
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
*  "It should support multiple root cases" in do {
*      var payload = {}
*      var flowVar = {a: 123}
*      ---
*     [
*         payload must beObject(),
*         flowVar must [
*              beObject(),
*              $.a must equalTo(123)
*           ]
*       ]
*  }
* ----
*/
fun in(testName: String, callback: Array<() -> MatcherResult>): TestResult = do {
    var skipAll: Boolean = (dw::Runtime::prop("skipAll") default false) as Boolean
    ---
    in(testName, callback, skipAll)
}

fun in(testName: String, callback: Array<() -> MatcherResult>, skipAll: Boolean): TestResult = do {
    var OK_RESULT = {name: testName, status: OK_STATUS, time: 0}

    var testToRun: String | Null = dw::Runtime::prop("testToRun")

    fun doIn(testName:String, testCases: () -> MatcherResult) = do {
      //Unicode chars for helping visualizing
        var CHECK = "\u2713"
        var CROSS = "\u274C"
        var WARNING = "\u26A0"

        var testWithTime = duration(() -> dw::Runtime::try(testCases))
        var timeExecution = testWithTime.time
        var testExecution = testWithTime.result
        var result =
            if(testExecution.success)
                {
                   name:        testName,
                   time:        timeExecution,
                   status:      if(testExecution.result.matches!) OK_STATUS else FAIL_STATUS,
                   errorMessage:
                                if(testExecution.result.reasons?)
                                    testExecution.result.reasons! joinBy "\n"
                                else do {
                                     var description = testExecution.result.description
                                     ---
                                     "Expecting $(description.expected) but was $(description.actual)."
                                }
                }
            else
                {
                    name:         testName,
                    status:       ERROR_STATUS,
                    time:         timeExecution,
                    errorMessage: testExecution.error.kind ++  ":" ++ testExecution.error.message! ++
                                     if(testExecution.error.location?)
                                      (" at:\n" ++ testExecution.error.location!)
                                     else "" ++
                                       if(testExecution.error.stack?)
                                        ("\n" ++ testExecution.error.stack! reduce ($$ ++ "\n" ++ "$"))
                                       else
                                        "\n" ++ (testExecution.error.stacktrace as String default " NO STACK ")
                }
        ---
         result
    }

    fun exec() =
        interceptor(
                    () -> log("wtf", {event: "TestStarted", name: testName, location: location(testName)}),
                    () -> do {
                        callback reduce (assertion, result = OK_RESULT) ->
                            if(result.status == OK_STATUS)
                                doIn(testName, assertion)
                            else
                                result
                    },
                    (result) -> log("wtf", result ++ {event: "TestFinished"}))
    fun skipRun() =
        interceptor(
     () -> log("wtf", {event: "TestStarted", name: testName, location: location(testName)}),
      () -> {
                 name: testName,
                 time: 0,
                 status: SKIP_STATUS
               },
      (result) -> log("wtf", result ++ {event: "TestFinished"})
        )
    var patternMatches = testToRun match {
        case is Null -> true
        case testToRunPattern is String ->
          isBlank(testToRunPattern) or !isEmpty(testToRunPattern splitBy "," filter ((acceptedTest, index) -> testName contains acceptedTest))
    }
    ---
    if (skipAll or (not patternMatches))
      skipRun()
    else
      exec()
}

/**
* Defines a new test suite with the list of test cases.
*
* === Example
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::test::Tests
*  ---
*
*  "Matcher api" describedBy [
*      "It should support nested matching" in  do {
*          var payload = {}
*          ---
*          payload must [
*              beObject(),
*              $.foo must [
*                  beNull()
*              ]
*          ]
*      },
* ]
* ----
*/
fun describedBy(suite: String, testsToRun: Array<() -> TestResult> ): TestResult = do {
    var result = interceptor(
                              () -> log("wtf", {event: "TestSuiteStarted", name: suite, location: location(suite)}),
                               () -> do {
                                  var measuredResult = duration(() -> testsToRun reduce ((test, acc = []) -> acc + test()))
                                  var testResults = measuredResult.result
                                  var testExecutionTime = measuredResult.time
                                  ---
                                  {
                                      name: suite,
                                      time: testExecutionTime,
                                      status: OK_STATUS,
                                      tests: testResults
                                  }
                               },
                               (result) -> log("wtf", result ++ {event: "TestSuiteFinished"})
                              )
    ---
    result
}

