/**
*
* This module contains all functions required to create a Data Weave test
*
* .Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::test::Tests
*  ---
*
*  "Matcher api" describedBy [
*      "It should support nested matching" in  do {
*          var payload = {}
*          ---
*          payload must [
*              beObject(),
*              $.foo must [
*                  beNull()
*              ]
*          ]
*      },
*      "It should support simple matching" in do {
*          var payload = {}
*          ---
*          payload must beObject()
*      },
*
*      "It should support multiple root cases" in do {
*          var payload = {}
*          var flowVar = {a: 123}
*          ---
*          [
*              payload must beObject(),
*              flowVar must [
*                  beObject(),
*                  $.a must equalTo(123)
*              ]
*          ]
*      },
*      "It should support using custom assertions" in  do {
*          var payload = []
*          ---
*          payload must sizeOf($) > 2
*      }
*  ]
* ----
*/
%dw 2.0

import diff from dw::util::Diff
import * from dw::util::Timer
import some from dw::core::Arrays
import interceptor from dw::test::internal::Functions
import red,green from dw::test::internal::Ansi

var ERROR_STATUS = "ERROR"
var OK_STATUS = "OK"
var FAIL_STATUS = "FAIL"

/**
* Data Type that describes the result of a Test Execution
*/
type TestResult = {
    name: String,
    time: Number,
    status: String,
    tests?: Array<TestResult>,
    errorMessages?: Array<String>
}

/**
* Defines a new test case inside a test suite with a single assertion.
*
* .Example
*
* [source,DataWeave,linenums]
* ----
* "It should support nested matching" in  do {
*    "foo" must beString()
* }
* ---
*/
fun in(testName:String, testCases: () -> MatcherResult): TestResult = do {
  in(testName, [testCases])
}


/**
* Defines a new test case with multiple assertions
*
*
* .Example
*
* [source,DataWeave,linenums]
* ----
*  "It should support multiple root cases" in do {
*      var payload = {}
*      var flowVar = {a: 123}
*      ---
*     [
*         payload must beObject(),
*         flowVar must [
*              beObject(),
*              $.a must equalTo(123)
*           ]
*       ]
*  }
* ----
*/
fun in(testName:String, callback: Array<() -> MatcherResult>): TestResult = do {
    var OK_RESULT = {name: testName, status: OK_STATUS, time: 0}

    fun doIn(testName:String, testCases: () -> MatcherResult) = do {
      //Unicode chars for helping visualizing
        var CHECK = "\u2713"
        var CROSS = "\u274C"
        var WARNING = "\u26A0"

        var testWithTime = duration(() -> dw::Runtime::try(testCases))
        var timeExecution = testWithTime.time
        var testExecution = testWithTime.result
        var result =
            if(testExecution.success)
                {
                   name: testName,
                   time: timeExecution,
                   status: if(testExecution.result.matches!) OK_STATUS else FAIL_STATUS,
                   errorMessage:
                    if(testExecution.result.reasons?)
                        testExecution.result.reasons! joinBy "\n"
                    else do {
                         var description = testExecution.result.description
                         ---
                         "Expecting $(description.expected) but was $(description.actual)."
                    }
                }
            else
              {
                name: testName,
                status: ERROR_STATUS,
                time: timeExecution,
                errorMessage: testExecution.error.message! ++
                                 if(testExecution.error.location?) (" at:\n" ++ testExecution.error.location!) else "" ++
                                 if(testExecution.error.stack?) ("\n" ++ testExecution.error.stack! reduce ($$ ++ "\n" ++ "$")) else " NO STACK "
              }
        ---
         result
    }
    ---
    interceptor(
                () -> log("wtf", {event: "TestStarted", name: testName}),
                () -> do {
                    callback reduce (assertion, result = OK_RESULT) ->
                        if(result.status == OK_STATUS)
                            doIn(testName, assertion)
                        else
                            result
                },
                (result) -> log("wtf", result ++ {event: "TestFinished"}))

}

/**
* Defines a new test suite with the list of test cases.
*
* .Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::test::Tests
*  ---
*
*  "Matcher api" describedBy [
*      "It should support nested matching" in  do {
*          var payload = {}
*          ---
*          payload must [
*              beObject(),
*              $.foo must [
*                  beNull()
*              ]
*          ]
*      },
* ]
* ----
*/
fun describedBy(suite: String, testsToRun:Array<() -> TestResult> ): TestResult = do {
    var messuredResult = interceptor(
                                        () -> log("wtf", {event: "TestSuiteStarted", name:suite}),
                                        () -> duration(() -> testsToRun reduce ((test, acc = []) -> acc << test())),
                                        (result) -> log("wtf", {event: "TestSuiteFinished", name:suite})
                                    )
    var testResults = messuredResult.result
    var testExecutionTime = messuredResult.time
    ---
      {
        name: suite,
        time: testExecutionTime,
        status: OK_STATUS,
        tests: testResults
      }
}


//MATCHERS

type MatcherResult = {
    "matches": Boolean,
    description: { expected: String, actual: String},
    reasons?: Array<String>
}


type Matcher<T> = (value: T) -> MatcherResult

var MATCHED = {"matches": true, description: {expected: "", actual: ""}}

/**
* This function allows to assert a value with with a list of Matcher of Expressions
*
* .Example
*
* [source, DataWeave,linenum]
* ----
* payload must [
*     beObject(),
*     $.foo is Null
* ]
* ----
*/
fun must<T>(value: T, matchExpressions: Array<(value:T) -> Matcher<T> | MatcherResult | Boolean>): MatcherResult = do {
    matchExpressions reduce (assertion, result = MATCHED) ->
        if(result.matches) do {
           var assertionResult = assertion(value) match {
               case is MatcherResult -> $
               case is Matcher -> $(value)
               case is Boolean ->  { matches: $, description: { expected: "`" ++ dw::Runtime::locationString(assertion) ++ "`", actual: write(value) as String} }
           }
           ---
           assertionResult
        } else
           result

}

/**
* This function allows to assert a value with with a list of Matcher of Expressions
*
* .Example
*
* [source, DataWeave,linenum]
* ----
* payload must beObject()
* ----
*/
fun must<T>(value: T, matcher: (value: T) -> Matcher<T> | Boolean): MatcherResult = do {
    matcher(value) match {
        case is Matcher -> $(value)
        case is Boolean ->
            { matches: $, description: { expected: "`" ++ dw::Runtime::locationString(matcher) ++ "`", actual: write(value) as String} }
    }
}

/**
* Validates that the specified matcher does not validate
*/
fun notBe<T>(matcher:Matcher<T>): Matcher<T> = (value) -> do {
    var matcherResult = matcher(value)
    ---
     {
        matches: not matcherResult.matches,
        description: {
            expected: "Not to be $(matcherResult.description.expected)",
            actual: matcherResult.description.actual
        }
     }
}

/**
* Validates if any of the matchers are successful.
*/
fun anyOf(matchers: Array<Matcher<Any>>): Matcher<Any> = (value) -> do {
    var matchersResult = matchers map $(value)
    var matchesResult = matchersResult some $.matches
    ---
    {
      matches: matchesResult,
      description: {
        expected: "Any of " ++ (matchersResult map $.description.expected joinBy " or ") ,
        actual: "$(write(value) as String)"
      }
    }
}

/**
* Will validate the value is equal to the expected. If not a list of diffs will be reported
*/
fun equalTo(expected: Any): Matcher =
    (actual) -> do {
        var diffResult = diff(actual, expected)
        ---
        {
          matches: diffResult.matches,
          description:
            {
               expected: "$(write(expected) as String) to be equal to",
               actual: "$(write(actual) as String)"
            },
          reasons: diffResult.diffs map ((diff,index) -> "Expecting `$(diff.path)` to be $(diff.expected) but was $(diff.actual)")
        }
    }



/**
* Validates that the value is at least one of the expected values.
*/
fun oneOf(expected:Array<Any>): Matcher =
    (actual: Any) -> do{
        var exists = expected contains actual
        ---
        {
            matches: exists,
            description:
                {
                    expected: "one of $(write(expected) as String)",
                    actual: "$(write(actual) as String)"
                },
        }
    }

/**
* Validate that the asserted value (Array, String or Object) has the expected size.
*/
fun haveSize(expectedSize:Number): Matcher<Array | String | Object> =
    (actual) -> do {
        var actualSize = sizeOf(actual)
        ---
        {
            matches: actualSize == expectedSize,
            description: { expected: "size $(expectedSize)", actual: "$(actualSize)" }
        }

    }

/**
* Validate that the asserted value is of type Array.
*/
fun beArray(): Matcher =
    (actual: Any) -> do {
        var isArray = actual is Array
        ---
        {
            matches: isArray,
            description: {expected: "Array type", actual: "$(typeOf(actual) as String)"}
        }

    }

/**
* Validate that the asserted value is of type String.
*/
fun beString(): Matcher =
    (actual: Any) -> do {
        var isArray = actual is String
        ---
        {
            matches: isArray,
            description: {expected: "String type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validate that the asserted value is of type Number.
*/
fun beNumber(): Matcher =
    (actual: Any) -> do {
        var isArray = actual is Number
        ---

        {
            matches: isArray,
            description: {expected: "Number type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validate that the asserted value is of type Boolean.
*/
fun beBoolean(): Matcher =
    (actual: Any) -> do {
        var isArray = actual is Boolean
        ---
        {
            matches: isArray,
            description: {expected: "Boolean type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validate that the asserted value is of type Object.
*/
fun beObject(): Matcher =
    (actual: Any) -> do {
        var isArray = actual is Object
        ---
        {
            matches: isArray,
            description: { expected: "Object type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validate that the asserted value is of type Null.
*/
fun beNull(): Matcher =
    (actual: Any) -> do {
        var isArray = actual is Null
        ---
        {
            matches: isArray,
            description: { expected: "Null type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validate that the asserted value is not Null.
*/
fun notBeNull(): Matcher =
    (actual: Any) -> do {
        var isArray = not (actual is Null)
        ---
        {
            matches: isArray,
            description: { expected: "Not Null type", actual: "$(typeOf(actual) as String)"}
        }
    }

/**
* Validate that the asserted string contains the specified string.
*/
fun contain(expected:String): Matcher<String> =
    (actual:String) -> do {
        var matchContains = actual contains expected
        ---
        {
            matches: matchContains,
            description: {expected: " to contain $(expected)", actual: actual}
        }
    }


/**
* Validate that the asserted string ends with the specified string.
*/
fun endWith(expected:String): Matcher<String> =
    (actual:String) -> do {
        var matchContains = actual endsWith expected
        ---
        {
            matches: matchContains,
            description: {expected: " to end with $(expected)", actual: actual}
        }
    }


/**
* Validate that the asserted string starts with the specified string.
*/
fun startWith(expected:String): Matcher<String> =
    (actual:String) -> do {
        var matchContains = actual startsWith expected
        ---
        {
            matches: matchContains,
            description: {expected: " to start with $(expected)", actual: actual}
        }
    }


/**
* Validates that the asserted value (String or Object or Array) is empty.
*/
fun beEmpty(): Matcher<String | Object | Array> =
     (actual:String | Object | Array) -> do {
         var matchContains = isEmpty(actual)
         ---
         {
             matches: matchContains,
             description: {expected: " to be empty.", actual: actual}
         }
     }

/**
* Validates that the asserted string is blank
*/
fun beBlank(): Matcher<String> =
     (actual: String) -> do {
         var matchContains = isBlank(actual)
         ---
         {
             matches: matchContains,
             description: {expected: " to be blank", actual: actual}
         }
     }


/**
* Validates that the asserted object contains the specified key
*/
fun haveKey(keyName:String): Matcher<Object> =
    (actual: Object) -> do {
        var matchKeyPresent = actual[keyName]?
        ---
        {
            matches: matchKeyPresent,
            description: {expected: " to contain key $(keyName)", actual: write(actual) as String}
        }
    }

