package org.mule.weave.v2.module.xmlschema

import org.mule.weave.v2.module.reader.BOMInputStream
import org.mule.weave.v2.module.reader.CharsetHelper

import java.io.ByteArrayInputStream
import java.io.File
import java.io.FileInputStream
import java.io.FileNotFoundException
import java.io.IOException
import java.io.InputStream
import java.net.MalformedURLException
import java.net.URL
import java.nio.charset.Charset
import scala.collection.mutable

object SchemaCollector {

  def getInstance = new SchemaCollector(Charset.defaultCharset.toString)

  def getInstance(charset: String) = new SchemaCollector(charset)
}

class SchemaCollector private (val charset: String) {

  val schemas: mutable.Map[String, InputStream] = mutable.Map()

  def addSchema(schema: File): SchemaCollector = {
    try schemas.put(schema.toURI.toString, wrapInBOMInputStream(new FileInputStream(schema)))
    catch {
      case e: FileNotFoundException =>
        throw new RuntimeException("Error while adding schema file [" + schema.getName + "]", e)
    }
    this
  }

  def addSchema(schemaUrl: URL): SchemaCollector = {
    try schemas.put(schemaUrl.toString, wrapInBOMInputStream(schemaUrl.openStream))
    catch {
      case e: IOException =>
        throw new RuntimeException("Error while adding schema of URL [" + schemaUrl + "]", e)
    }
    this
  }

  def addSchema(name: String, schemaContent: String): SchemaCollector = {
    schemas.put(name, wrapInBOMInputStream(new ByteArrayInputStream(schemaContent.getBytes(Charset.forName(charset)))))
    this
  }

  def addSchema(schemaUrl: String): SchemaCollector = {
    try addSchema(new URL(schemaUrl))
    catch {
      case e: MalformedURLException =>
        throw new RuntimeException("The specified URL [" + schemaUrl + "] is invalid", e)
    }
    this
  }

  def addSchema(name: String, schemaContent: InputStream): SchemaCollector = {
    schemas.put(name, wrapInBOMInputStream(schemaContent))
    this
  }

  def collect: mutable.Map[String, InputStream] = schemas

  /**
    * Wraps the input strema in a {@link BOMInputStream} to make sure that we can handle this scenario.
    *
    * Otherwise XML Parsers with fail with error "Content not valid in prolog."
    *
    * @param streamToWrap the InputStream to be wrapped.
    * @return a {@link BOMInputStream} that can handle xmls with and without BOM markers
    */
  private def wrapInBOMInputStream(streamToWrap: InputStream) = CharsetHelper.getBom(Charset.defaultCharset()).map(bomMarker => new BOMInputStream(streamToWrap, bomMarker)).getOrElse(streamToWrap)
}