package org.mule.weave.v2.module.xmlschema

import org.mule.apache.xerces.impl.dv.xs.XSSimpleTypeDecl
import org.mule.apache.xerces.impl.xs._
import org.mule.apache.xerces.xs._
import org.mule.weave.v2.module.xmlschema.handler.HandlerManager
import org.mule.weave.v2.module.xmlschema.utils.SchemaHelper
import org.mule.weave.v2.module.xmlschema.utils.SchemaHelper._
import org.mule.weave.v2.module.xmlschema.utils.XmlConstants.URI_2001_SCHEMA_XSD
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.SafeStringBasedParserInput
import org.mule.weave.v2.parser.ast.header.directives._
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.types._
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase._
import org.mule.weave.v2.sdk.NameIdentifierHelper
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.mule.weave.v2.sdk.WeaveResourceResolverAware
import org.mule.weave.v2.utils.StringEscapeHelper
import org.mule.weave.v2.utils.WeaveNameHelper

import java.net.URL
import java.nio.file.Paths
import java.util.function.Function
import javax.xml.namespace.QName
import scala.collection.mutable.ArrayBuffer
import scala.collection.mutable.ListBuffer

class XmlSchemaModuleLoader extends ModuleLoader with WeaveResourceResolverAware {

  private var resolver: WeaveResourceResolver = _
  val ROOT_TYPE_NAME = "Root"
  val XSD_LOADER_NAME = "xmlschema"
  val TYPE_DEFINITION_SUFFIX = "Definition"
  val XSD_EXTENSION = ".xsd"
  val ELEMENT_DEFINITION = "ElementDefinition"

  override def loadModule(nameIdentifier: NameIdentifier, parsingContext: ParsingContext): Option[PhaseResult[ParsingResult[ModuleNode]]] = {
    val path = getXsdPath(nameIdentifier)
    val maybeResource = resolver.resolvePath(path)
    maybeResource match {
      case Some(resource) => {
        val xmlSchemaLoaderContext = new XmlSchemaTransformationContext(resource, nameIdentifier, resolver)
        val schemaParserResult = xmlSchemaLoaderContext.modelResult
        if (schemaParserResult.hasErrors()) {
          Some(new PhaseResult(None, schemaParserResult.messages()))
        } else {
          loadSchema(nameIdentifier, xmlSchemaLoaderContext)
        }
      }
      case None => None
    }
  }

  private def getXsdPath(nameIdentifier: NameIdentifier) = {
    NameIdentifierHelper.toWeaveFilePath(nameIdentifier, NameIdentifierHelper.fileSeparator, XSD_EXTENSION)
  }

  private def loadSchema(nameIdentifier: NameIdentifier, xmlSchemaTransformationContext: XmlSchemaTransformationContext) = {
    val directives = new ArrayBuffer[DirectiveNode]()
    directives ++= loadTypeDefinitions(xmlSchemaTransformationContext)
    val rootElements = loadElements(xmlSchemaTransformationContext)
    if (rootElements.nonEmpty) {
      val rootTypeValue = if (rootElements.size > 1) SchemaHelper.buildUnionTypeNode(rootElements.map(rootElement => TypeReferenceNode(NameIdentifier(WeaveNameHelper.toValidNameIdentifier(rootElement.nameIdentifier.name, true)))), (x: WeaveTypeNode) => x) else TypeReferenceNode(NameIdentifier(WeaveNameHelper.toValidNameIdentifier(rootElements(0).nameIdentifier.name, true)))
      val rootType = TypeDirective(NameIdentifier(ROOT_TYPE_NAME), None, rootTypeValue)
      directives ++= rootElements
      directives += (rootType)
    }
    val versionDirectives = Seq(VersionDirective(VersionMajor("2"), VersionMinor("0")))
    val namespaceDirectives: List[NamespaceDirective] = xmlSchemaTransformationContext.getNamespaceDirectives
    val moduleNode: ModuleNode = ModuleNode(nameIdentifier, versionDirectives ++ namespaceDirectives ++ directives)
    val input: ParsingContentInput = ParsingContentInput(xmlSchemaTransformationContext.resource, nameIdentifier, SafeStringBasedParserInput(xmlSchemaTransformationContext.resource.content()))
    Some(new PhaseResult(Some(ParsingResult(input, moduleNode)), MessageCollector()))
  }

  private def loadElements(xmlSchemaTransformationContext: XmlSchemaTransformationContext): List[TypeDirective] = {
    val directiveNodes: ArrayBuffer[TypeDirective] = new ArrayBuffer[TypeDirective]()
    val elementDeclarationMap = xmlSchemaTransformationContext.modelResult.getResult().getComponents(XSConstants.ELEMENT_DECLARATION)
    elementDeclarationMap.forEach((_, value) => {
      val xsElementDeclaration = value.asInstanceOf[XSElementDeclaration]
      val typeDefinition = xsElementDeclaration.getTypeDefinition
      val attributesNodes = processAttributes(typeDefinition, xmlSchemaTransformationContext)
      val name = xsElementDeclaration.getName
      val namespace = xsElementDeclaration.getNamespace
      val dwPrefix = processNamespace(xmlSchemaTransformationContext, namespace)
      val typeName = getTypeName(typeDefinition)
      val valueTypeNode = if (typeName.isDefined && xmlSchemaTransformationContext.types.contains(typeName.get)) {
        xmlSchemaTransformationContext.types(typeName.get)
      } else {
        val stringNode = StringNode(name)
        val stringNodeWithQuotation = if (StringEscapeHelper.keyRequiresQuotes(name)) stringNode.withQuotation('"') else stringNode
        val node = TypeSelectorNode(NameNode(stringNodeWithQuotation, dwPrefix), TypeReferenceNode(NameIdentifier(WeaveNameHelper.toValidNameIdentifier(name + ELEMENT_DEFINITION, true))), None)
        if (typeName.isDefined && !typeName.get.getNamespaceURI.equals(URI_2001_SCHEMA_XSD)) {
          xmlSchemaTransformationContext.types.put(typeName.get, node)
        }
        processValue(xsElementDeclaration, xmlSchemaTransformationContext, node)
      }
      val elementDefinitionNode = KeyValueTypeNode(KeyTypeNode(NameTypeNode(Option(name), dwPrefix), attributesNodes), valueTypeNode, false, false)
      directiveNodes += TypeDirective(NameIdentifier(WeaveNameHelper.toValidNameIdentifier(name + ELEMENT_DEFINITION, true)), None, ObjectTypeNode(Seq(elementDefinitionNode), close = true))
    })
    directiveNodes.toList
  }

  private def loadTypeDefinitions(xmlSchemaTransformationContext: XmlSchemaTransformationContext): List[DirectiveNode] = {
    val typedirectives = new ArrayBuffer[DirectiveNode]()
    val model: XSModel = xmlSchemaTransformationContext.modelResult.getResult()
    model.asInstanceOf[XSModelImpl].forEach((schemaGrammar) => {
      typedirectives ++= loadImports(schemaGrammar.asInstanceOf[SchemaGrammar], xmlSchemaTransformationContext)
    })
    val typeDefinitions = model.getComponents(XSConstants.TYPE_DEFINITION)
    typeDefinitions.forEach((_, value) => {
      val xsTypeDefinition = value.asInstanceOf[XSTypeDefinition]
      if (!Option(xsTypeDefinition.getNamespace).getOrElse("").equals(SchemaSymbols.URI_SCHEMAFORSCHEMA)) {
        cacheTypeSelectorNodes(xsTypeDefinition, xmlSchemaTransformationContext)
      }
    })
    typeDefinitions.forEach((_, value) => {
      val xsTypeDefinition = value.asInstanceOf[XSTypeDefinition]
      loadTypeDefinition(xmlSchemaTransformationContext, xsTypeDefinition).map(directiveNode => typedirectives += directiveNode)
    })
    typedirectives.toList
  }

  private def loadTypeDefinition(xmlSchemaTransformationContext: XmlSchemaTransformationContext, xsTypeDefinition: XSTypeDefinition): Option[DirectiveNode] = {
    if (isLocalDefinition(xmlSchemaTransformationContext, xsTypeDefinition)) {
      getTypeName(xsTypeDefinition).map(typeName => {
        val definitionName = typeName.getLocalPart.capitalize + TYPE_DEFINITION_SUFFIX
        val uri = typeName.getNamespaceURI
        val dwPrefix = processNamespace(xmlSchemaTransformationContext, uri)
        val nameNode = NameTypeNode(Option(typeName.getLocalPart), dwPrefix)
        val localName = nameNode.localName.get
        val stringNode = StringNode(localName)
        val stringNodeWithQuotation = if (StringEscapeHelper.keyRequiresQuotes(localName)) stringNode.withQuotation('"') else stringNode
        val typeSelector = TypeSelectorNode(NameNode(stringNodeWithQuotation, nameNode.ns), TypeReferenceNode(NameIdentifier(WeaveNameHelper.toValidNameIdentifier(typeName.getLocalPart.capitalize + ELEMENT_DEFINITION, true))), None)
        val typeNodeWithoutKey = processTypeDeclaration(None, xsTypeDefinition, xmlSchemaTransformationContext, typeSelector)
        val typeNodeWithKey = ObjectTypeNode(Seq(KeyValueTypeNode(KeyTypeNode(nameNode, processAttributes(xsTypeDefinition, xmlSchemaTransformationContext)), typeNodeWithoutKey, false, false)))
        TypeDirective(NameIdentifier(WeaveNameHelper.toValidNameIdentifier(definitionName, true)), None, typeNodeWithKey)
      })
    } else {
      None
    }
  }

  private def isLocalDefinition(xmlSchemaTransformationContext: XmlSchemaTransformationContext, xsTypeDefinition: XSTypeDefinition) = {
    !Option(xsTypeDefinition.getNamespace).getOrElse("").equals(SchemaSymbols.URI_SCHEMAFORSCHEMA) && xsTypeDefinition.getNamespaceItem.getDocumentLocations.contains(xmlSchemaTransformationContext.resource.url())
  }

  private def cacheTypeSelectorNodes(xsTypeDefinition: XSTypeDefinition, xmlSchemaTransformationContext: XmlSchemaTransformationContext): Unit = {
    getTypeName(xsTypeDefinition).map(typeName => {
      val uri = typeName.getNamespaceURI
      val dwPrefix = processNamespace(xmlSchemaTransformationContext, uri)
      val nameNode = NameTypeNode(Option(typeName.getLocalPart), dwPrefix)
      val definitionName = typeName.getLocalPart.capitalize + TYPE_DEFINITION_SUFFIX
      val documentLocations = xsTypeDefinition.getNamespaceItem.getDocumentLocations
      val locations = documentLocations.toArray(new Array[String](documentLocations.getLength))
      val foundLocation = locations.find(location => !location.equals(xmlSchemaTransformationContext.resource.url()) && xmlSchemaTransformationContext.fileNameToLocation.contains(location)).flatMap(location => xmlSchemaTransformationContext.fileNameToLocation.get(location))
      val validNameIdentifier = WeaveNameHelper.toValidNameIdentifier(definitionName, true)
      val nameIdentifier = NameIdentifier(foundLocation.map(location => location + "::" + validNameIdentifier).getOrElse(validNameIdentifier))
      val stringNode = StringNode(nameNode.localName.get)
      val stringNodeWithQuotation = if (StringEscapeHelper.keyRequiresQuotes(nameNode.localName.get)) stringNode.withQuotation('"') else stringNode
      val typeSelectorRefNode = TypeSelectorNode(NameNode(stringNodeWithQuotation, nameNode.ns), TypeReferenceNode(nameIdentifier), None)
      xmlSchemaTransformationContext.types.put(typeName, typeSelectorRefNode)
    })
  }

  private def loadImports(schemaGrammar: SchemaGrammar, xmlSchemaTransformationContext: XmlSchemaTransformationContext): List[DirectiveNode] = {
    val typeDirectives: ArrayBuffer[DirectiveNode] = new ArrayBuffer[DirectiveNode]()
    val documentLocations = schemaGrammar.getDocumentLocations
    val locations = documentLocations.toArray(new Array[String](documentLocations.getLength))
    locations.foreach(location => {
      if (!location.equals(xmlSchemaTransformationContext.resource.url())) {
        val localPath = Paths.get(new URL(location).toURI.getPath)
        val moduleFilePath = Paths.get(new URL(xmlSchemaTransformationContext.resource.url()).toURI)
        val pathFromClasspath = Paths.get(getXsdPath(xmlSchemaTransformationContext.moduleIdentifier))
        val pathToTransform = pathFromClasspath.resolve(moduleFilePath.relativize(localPath)).normalize().toString
        val fileNameIdentifier = NameIdentifierHelper.fromWeaveFilePath(pathToTransform)
        xmlSchemaTransformationContext.fileNameToLocation.put(location, fileNameIdentifier.localName().name)
        typeDirectives += ImportDirective(ImportedElement(NameIdentifier(fileNameIdentifier.name, Some(XSD_LOADER_NAME))))
      }
    })
    typeDirectives.toList
  }

  private def processValue(elementDeclaration: XSElementDeclaration, xmlSchemaTransformationContext: XmlSchemaTransformationContext, parentReferenceNode: WeaveTypeNode): WeaveTypeNode = {
    val typeDefinition = getXsTypeDefinition(elementDeclaration, xmlSchemaTransformationContext.modelResult.getResult())
    processTypeDeclaration(Some(elementDeclaration), typeDefinition, xmlSchemaTransformationContext, parentReferenceNode)
  }

  private def processTypeDeclaration(elementDeclaration: Option[XSElementDeclaration], typeDefinition: XSTypeDefinition, xmlSchemaTransformationContext: XmlSchemaTransformationContext, parentReferenceNode: WeaveTypeNode): WeaveTypeNode = {
    if (typeDefinition.isInstanceOf[XSComplexTypeDecl]) {
      val complexType = typeDefinition.asInstanceOf[XSComplexTypeDecl]
      val groupType = getGroupType(complexType)
      if (groupType.isDefined && groupType.get == XSModelGroup.COMPOSITOR_CHOICE) {
        processUnionType(complexType, xmlSchemaTransformationContext, parentReferenceNode)
      } else if (XSComplexTypeDefinition.CONTENTTYPE_SIMPLE == complexType.getContentType) {
        TypeReferenceNode(NameIdentifier("String"))
      } else {
        val propertiesNodes: ListBuffer[WeaveTypeNode] = new ListBuffer[WeaveTypeNode]()
        if (XSComplexTypeDefinition.CONTENTTYPE_MIXED == complexType.getContentType) {
          propertiesNodes += KeyValueTypeNode(KeyTypeNode(NameTypeNode(Some("__text"))), TypeReferenceNode(NameIdentifier("String")), true, true)
        }
        val fields = processElements(complexType, xmlSchemaTransformationContext, parentReferenceNode)
        propertiesNodes ++= fields.fields
        val objectTypeNode = ObjectTypeNode(propertiesNodes, None, ordered = fields.ordered)
        objectTypeNode
      }
    } else if (typeDefinition.isInstanceOf[XSSimpleTypeDecl]) {
      HandlerManager.handle(typeDefinition.asInstanceOf[XSSimpleTypeDecl], elementDeclaration.flatMap(declr => getDefaultValue(declr)))
    } else {
      TypeReferenceNode(NameIdentifier("Any"))
    }
  }

  private def processElements(complexType: XSComplexTypeDecl, xmlSchemaTransformationContext: XmlSchemaTransformationContext, parentReferenceNode: WeaveTypeNode): ObjectFields = {
    val particle = complexType.getParticle
    if (particle == null) {
      ObjectFields(List())
    } else {
      processElements(particle, xmlSchemaTransformationContext, parentReferenceNode)
    }
  }

  private def processElements(particle: XSParticle, xmlSchemaTransformationContext: XmlSchemaTransformationContext, parentReferenceNode: WeaveTypeNode): ObjectFields = {
    processElements(particle, (part: XSParticle) => part.getMinOccurs > 0, xmlSchemaTransformationContext, parentReferenceNode)
  }

  private def processElements(particle: XSParticle, requiredSupplier: Function[XSParticle, Boolean], xmlSchemaTransformationContext: XmlSchemaTransformationContext, parentReferenceNode: WeaveTypeNode): ObjectFields = {
    var ordered = false
    var openObject = false
    if (particle == null) {
      return null
    }
    val keyValueNodes: ListBuffer[WeaveTypeNode] = new ListBuffer[WeaveTypeNode]()
    val term = particle.getTerm
    if (term.isInstanceOf[XSModelGroup]) {
      val xsModelGroup = term.asInstanceOf[XSModelGroup]
      ordered = xsModelGroup.getCompositor == XSModelGroup.COMPOSITOR_SEQUENCE
      val particles = xsModelGroup.getParticles
      import scala.collection.JavaConversions._
      for (xsObject <- particles) {
        if (xsObject.isInstanceOf[XSParticle]) {
          val part = xsObject.asInstanceOf[XSParticle]
          val element = part.getTerm
          if (element.isInstanceOf[XSElementDeclaration]) {
            val xsElementDeclaration = element.asInstanceOf[XSElementDeclaration]
            val name = element.getName
            val namespace = element.getNamespace
            val typeDefinition = element.asInstanceOf[XSElementDeclaration].getTypeDefinition
            val attributesNodes = processAttributes(typeDefinition, xmlSchemaTransformationContext)
            val dwPrefix = processNamespace(xmlSchemaTransformationContext, namespace)
            val typeName = getTypeName(typeDefinition)
            val valueTypeNode = if (typeName.isDefined && xmlSchemaTransformationContext.types.containsKey(typeName.get)) {
              xmlSchemaTransformationContext.types(typeName.get)
            } else {
              val stringNode = StringNode(name)
              val stringNodeWithQuotation = if (StringEscapeHelper.keyRequiresQuotes(name)) stringNode.withQuotation('"') else stringNode
              val node = TypeSelectorNode(NameNode(stringNodeWithQuotation, dwPrefix), parentReferenceNode, None)
              if (typeName.isDefined && !typeName.get.getNamespaceURI.equals(URI_2001_SCHEMA_XSD)) {
                xmlSchemaTransformationContext.types.put(typeName.get, node)
              }
              processValue(xsElementDeclaration, xmlSchemaTransformationContext, node)
            }
            keyValueNodes += KeyValueTypeNode(KeyTypeNode(NameTypeNode(Option(name), dwPrefix), attributesNodes), valueTypeNode, isRepeated(part), !requiredSupplier.apply(part))
          } else if (element.isInstanceOf[XSWildcard] && element.asInstanceOf[XSWildcard].getConstraintType == XSWildcard.NSCONSTRAINT_ANY) {
            openObject = true
          } else if (element.isInstanceOf[XSModelGroup]) {
            val fields = processElements(part, (xsParticle: XSParticle) => false, xmlSchemaTransformationContext, parentReferenceNode)
            ordered |= fields.ordered
            keyValueNodes ++= fields.fields
          }
        }
      }
    }
    ObjectFields(keyValueNodes.toList, ordered = ordered, openObject = openObject)
  }

  private def buildUnionTypeNode(values: Array[WeaveTypeNode]): WeaveTypeNode = {
    SchemaHelper.buildUnionTypeNode(values, (x: WeaveTypeNode) => x)
  }

  private def processUnionType(xsModelGroup: XSModelGroup, xmlSchemaTransformationContext: XmlSchemaTransformationContext, parentReferenceNode: WeaveTypeNode): WeaveTypeNode = {
    val particles = xsModelGroup.getParticles
    import scala.collection.JavaConversions._
    val objectTypeNodes: ListBuffer[WeaveTypeNode] = ListBuffer[WeaveTypeNode]()
    for (xsObject <- particles) {
      if (xsObject.isInstanceOf[XSParticle]) {
        val part = xsObject.asInstanceOf[XSParticle]
        val element = part.getTerm
        val groupType = getGroupType(element)
        if (groupType.isDefined && groupType.get == XSModelGroup.COMPOSITOR_CHOICE) {
          objectTypeNodes += processUnionType(element.asInstanceOf[XSModelGroup], xmlSchemaTransformationContext, parentReferenceNode)
        } else {
          val keyValueTypeNodes: ObjectFields = if (element.isInstanceOf[XSElementDeclaration]) {
            val elementDeclaration = element.asInstanceOf[XSElementDeclaration]
            val name = element.getName
            val namespace = element.getNamespace
            val dwPrefix = processNamespace(xmlSchemaTransformationContext, namespace)
            val typeDefinition = element.asInstanceOf[XSElementDeclaration].getTypeDefinition
            val attributes = processAttributes(typeDefinition, xmlSchemaTransformationContext)
            val required = part.getMinOccurs > 0
            val typeName: Option[QName] = getTypeName(typeDefinition)
            val valueTypeNode = if (typeName.isDefined && xmlSchemaTransformationContext.types.containsKey(typeName.get)) {
              xmlSchemaTransformationContext.types(typeName.get)
            } else {
              val stringNode = StringNode(name)
              val stringNodeWithQuotation = if (StringEscapeHelper.keyRequiresQuotes(name)) stringNode.withQuotation('"') else stringNode
              val node = TypeSelectorNode(NameNode(stringNodeWithQuotation, dwPrefix), parentReferenceNode, None)
              if (typeName.isDefined && !typeName.get.getNamespaceURI.equals(URI_2001_SCHEMA_XSD)) {
                xmlSchemaTransformationContext.types.put(typeName.get, node)
              }
              processValue(elementDeclaration, xmlSchemaTransformationContext, node)
            }
            ObjectFields(List(KeyValueTypeNode(KeyTypeNode(NameTypeNode(Some(name), dwPrefix), attributes), valueTypeNode, repeated = false, optional = !required)))
          } else {
            processElements(part, xmlSchemaTransformationContext, parentReferenceNode)
          }
          objectTypeNodes += ObjectTypeNode(keyValueTypeNodes.fields, ordered = keyValueTypeNodes.ordered, close = !keyValueTypeNodes.openObject)
        }
      }
    }
    buildUnionTypeNode(objectTypeNodes.toArray)
  }

  private def processUnionType(complexType: XSComplexTypeDecl, xmlSchemaTransformationContext: XmlSchemaTransformationContext, parentReferenceNode: WeaveTypeNode): WeaveTypeNode = {
    var weaveTypeNode: WeaveTypeNode = null
    val particle = complexType.getParticle
    if (particle != null) {
      val term = particle.getTerm
      term match {
        case group: XSModelGroup =>
          weaveTypeNode = processUnionType(group, xmlSchemaTransformationContext, parentReferenceNode)
        case _ =>
      }
    }
    weaveTypeNode
  }

  private def getXsTypeDefinition(elementDeclaration: XSElementDeclaration, model: XSModel): XSTypeDefinition = {
    if (elementDeclaration.isInstanceOf[XSElementDecl]) {
      val unresolved = elementDeclaration.asInstanceOf[XSElementDecl].fUnresolvedTypeName
      if (unresolved != null) {
        val definition = model.getTypeDefinition(unresolved.localpart, unresolved.uri)
        if (definition != null) return definition
      }
    }
    elementDeclaration.getTypeDefinition
  }

  private def processAttributes(typeDefinition: XSTypeDefinition, xmlSchemaTransformationContext: XmlSchemaTransformationContext): List[WeaveTypeNode] = {
    val typeNodes = new ListBuffer[WeaveTypeNode]()
    if (typeDefinition.isInstanceOf[XSComplexTypeDecl]) {
      val attributeUses = typeDefinition.asInstanceOf[XSComplexTypeDefinition].getAttributeUses
      attributeUses.forEach {
        case xsAttributeUse: XSAttributeUse =>
          val attrDeclaration = xsAttributeUse.getAttrDeclaration
          val localName = attrDeclaration.getName
          val dwPrefix: Option[NamespaceNode] = processNamespace(xmlSchemaTransformationContext, attrDeclaration.getNamespace)
          val valueNode = HandlerManager.handle(attrDeclaration.getTypeDefinition.asInstanceOf[XSSimpleTypeDecl], getDefaultValue(xsAttributeUse))
          typeNodes += NameValueTypeNode(NameTypeNode(Option(localName), dwPrefix), valueNode, optional = !xsAttributeUse.getRequired)
        case _ =>
      }
    }
    typeNodes.toList
  }

  private def processNamespace(xmlSchemaTransformationContext: XmlSchemaTransformationContext, namespace: String) = {
    val dwNamespace = if (namespace == null || namespace.isEmpty) None else Option(namespace)
    val dwPrefix = dwNamespace match {
      case Some(uri) => {
        Option(NamespaceNode(NameIdentifier(xmlSchemaTransformationContext.namespaces.getOrElseUpdate(uri, xmlSchemaTransformationContext.getNextNsPrefix))))
      }
      case None => None
    }
    dwPrefix
  }

  override def name(): Option[String] = Some(XSD_LOADER_NAME)

  override def resolver(resolver: WeaveResourceResolver): Unit = {
    this.resolver = resolver
  }

  case class ObjectFields(fields: List[WeaveTypeNode], var openObject: Boolean = false, var ordered: Boolean = false)

  override def canResolveModule(nameIdentifier: NameIdentifier): Boolean = resolver.canResolveResource(NameIdentifierHelper.fromWeaveFilePath(getXsdPath(nameIdentifier)))
}
