package org.mule.weave.v2.module.yaml

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.structure.schema.SchemaProperty
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.reader.ConfigurableMaxEntityCount
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.reader.SourceProviderAwareReader
import org.mule.weave.v2.module.yaml.exception.MaxExpansionException
import org.mule.weave.v2.module.yaml.exception.YamlReaderException
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.location.SimpleLocation
import org.mulesoft.common.client.lexical.SourceLocation
import org.yaml.model.ParseErrorHandler
import org.yaml.model.SyamlException
import org.yaml.model.YComment
import org.yaml.model.YDocument
import org.yaml.model.YMap
import org.yaml.model.YNode
import org.yaml.model.YNode.Alias
import org.yaml.model.YPart
import org.yaml.model.YScalar
import org.yaml.model.YSequence
import org.yaml.model.YTag
import org.yaml.model.YType
import org.yaml.parser.YamlParser

import java.io.File
import java.io.InputStream
import scala.collection.mutable.ArrayBuffer
import scala.io.Source

class YamlReader(override val sourceProvider: SourceProvider, override val settings: YamlReaderSettings)(implicit ctx: EvaluationContext) extends Reader with SourceProviderAwareReader {

  override def dataFormat: Option[DataFormat[_, _]] = Some(new YamlDataFormat)

  private val inputStream = sourceProvider.asInputStream

  var expansionCounter: Int = 0

  private val currentComments: ArrayBuffer[StringValue] = new ArrayBuffer[StringValue]()

  /**
    * Should return the root value of the document that was read
    *
    * @param name The name of the variable that is being referenced example payload
    * @return The root value
    */
  override protected def doRead(name: String): Value[_] = {
    val source = Source.fromInputStream(inputStream, sourceProvider.charset.name())
    try {
      val inputStr: String = source.mkString
      val elements: IndexedSeq[YPart] = YamlParser(inputStr, name)(new ParseErrorHandler() {
        override def handle(location: SourceLocation, e: SyamlException): Unit = {
          throw new YamlReaderException(e.getMessage, new SimpleLocation(location.toString()))
        }
      }).parse()
      val roots: IndexedSeq[Value[_]] = (elements.flatMap((element) => traverse(element)))
      if (roots.size > 1) {
        val schema: Schema = Schema(Seq(SchemaProperty(StringValue(YamlReader.DOCUMENTS_SCHEMA_PROPERTY), BooleanValue.TRUE_BOOL)))
        ArrayValue(ArraySeq(roots), UnknownLocationCapable, Some(schema))
      } else if (roots.size == 1) {
        roots(0)
      } else {
        NullValue
      }
    } finally {
      source.close()
    }
  }

  def traverse(part: YPart, insideKey: Boolean = false, ytag: Option[YTag] = None): Option[Value[_]] = {
    part match {
      case document: YDocument =>
        document.children.flatMap((n) => traverse(n, insideKey)).headOption
      case alias: Alias =>
        //the alias has to be before YNode, since it is included in it.
        if (settings.maxEntityCount == -1 || expansionCounter < settings.maxEntityCount) {
          expansionCounter = expansionCounter + 1
          traverse(alias.target)
        } else {
          throw new MaxExpansionException(settings.maxEntityCount, SimpleLocation(alias.location.toString()))
        }
      case node: YNode =>
        val childValues: IndexedSeq[Value[_]] = node.children
          .flatMap((n) => {
            traverse(n, insideKey, Some(node.tag))
          })
        val headOption = childValues.headOption
        headOption
      case sequence: YSequence =>
        val comments: Array[StringValue] = retrieveComments()
        val seq: IndexedSeq[Value[_]] = sequence.children.flatMap((n) => traverse(n, insideKey))
        Some(new YamlArrayValue(ArraySeq(seq), SimpleLocation(sequence.location.toString()), comments))
      case map: YMap =>
        val comments = retrieveComments()
        val kvps: IndexedSeq[KeyValuePair] =
          for (entry <- map.entries) yield {
            var insideKey = true
            val mappedChildren = entry.children.flatMap((n) => {
              val result = traverse(n, insideKey)
              if (insideKey) {
                insideKey = result.isEmpty
              }
              result
            })
            KeyValuePair(mappedChildren.head.asInstanceOf[YamlKeyValue], mappedChildren.apply(1))
          }
        Some(new YamlObjectValue(ObjectSeq(kvps), SimpleLocation(map.location.toString()), comments))
      case scalar: YScalar =>
        //Load comments
        scalar.children.foreach((n) => traverse(n, insideKey))
        val comments: Array[StringValue] = retrieveComments()
        val location: SimpleLocation = SimpleLocation(scalar.location.toString())
        if (insideKey) {
          Some(new YamlKeyValue(QualifiedName(scalar.text, None), location, comments))
        } else {
          ytag match {
            case Some(tag) => {
              tag.tagType match {
                case YType.Int => {
                  try {
                    if (scalar.text.contains("0x")) {
                      Some(new YamlNumberValue(Number(Integer.decode(scalar.text)), location, comments))
                    } else {
                      Some(new YamlNumberValue(Number(location, scalar.text), location, comments))
                    }
                  } catch {
                    case _: Exception => throw new YamlReaderException(s"Unable to parse `${scalar.text}` with type Int as Number value.", location)
                  }
                }
                case YType.Float => {
                  try {
                    if (scalar.text == ".NaN") {
                      Some(new YamlNullValue(location, comments, NumberValue.NaN_PropertySeq))
                    } else if (scalar.text == "-.inf") {
                      Some(new YamlNullValue(location, comments, Seq(NumberValue.NEGATIVE_INF_Property)))
                    } else if (scalar.text == ".inf") {
                      Some(new YamlNullValue(location, comments, Seq(NumberValue.POSTIVE_INF_Property)))
                    } else {
                      Some(new YamlNumberValue(Number(location, scalar.text), location, comments))
                    }
                  } catch {
                    case _: Exception => throw new YamlReaderException(s"Unable to parse `${scalar.text}` with type Float as Number value.", location)
                  }
                }
                case YType.Null      => Some(new YamlNullValue(location, comments))
                case YType.Bool      => Some(new YamlBooleanValue(scalar.text.toBoolean, location, comments))
                case YType.Str       => Some(new YamlStringValue(scalar.text, location, comments))
                //TODO support timestamp
                case YType.Timestamp => Some(new YamlStringValue(scalar.text, location, comments))
                case _ => {
                  val tagProperty = Seq(SchemaProperty(StringValue(YamlReader.TAG_SCHEMA_PROPERTY), StringValue(tag.text)))
                  Some(new YamlStringValue(scalar.text, location, comments, tagProperty))
                }
              }
            }
            case None => Some(new YamlStringValue(scalar.text, location, comments))
          }

        }
      case comment: YComment => {
        this.currentComments.+=(StringValue(comment.toString()))
        None
      }
      case _ =>
        None
    }
  }

  private def retrieveComments(): Array[StringValue] = {
    val commentArray: Array[StringValue] = currentComments.toArray
    currentComments.clear()
    commentArray
  }
}

trait YamlValue[T] extends Value[T] {

  val comments: Array[StringValue]

  override def schema(implicit ctx: EvaluationContext): Option[Schema] = {
    Some(Schema(schemaProperties()))
  }

  def schemaProperties(): Seq[SchemaProperty] = {
    if (comments.nonEmpty) {
      Seq(SchemaProperty(StringValue(YamlReader.COMMENTS_SCHEMA_PROPERTY), ArrayValue(comments)))
    } else {
      Seq.empty
    }
  }
}

class YamlKeyValue(qualifiedName: QualifiedName, override val location: Location, val comments: Array[StringValue]) extends KeyValue with YamlValue[QualifiedName] {
  override def evaluate(implicit ctx: EvaluationContext): T = qualifiedName

}

class YamlObjectValue(objectSeq: ObjectSeq, override val location: Location, val comments: Array[StringValue]) extends ObjectValue with YamlValue[ObjectSeq] {
  override def evaluate(implicit ctx: EvaluationContext): T = objectSeq

}

class YamlArrayValue(arraySeq: ArraySeq, override val location: Location, val comments: Array[StringValue]) extends ArrayValue with YamlValue[ArraySeq] {
  override def evaluate(implicit ctx: EvaluationContext): T = arraySeq

}

class YamlStringValue(value: String, override val location: Location, val comments: Array[StringValue], additionalProperties: Seq[SchemaProperty] = Seq()) extends StringValue with YamlValue[String] {
  override def evaluate(implicit ctx: EvaluationContext): T = value

  override def schemaProperties(): Seq[SchemaProperty] = super.schemaProperties() ++ additionalProperties

}

class YamlNullValue(override val location: Location, val comments: Array[StringValue], additionalProperties: Seq[SchemaProperty] = Seq()) extends NullValue with YamlValue[Null] {
  override def evaluate(implicit ctx: EvaluationContext): T = null

  override def schemaProperties(): Seq[SchemaProperty] = super.schemaProperties() ++ additionalProperties

}

class YamlNumberValue(value: Number, override val location: Location, val comments: Array[StringValue]) extends NumberValue with YamlValue[Number] {
  override def evaluate(implicit ctx: EvaluationContext): T = value

}

class YamlBooleanValue(value: Boolean, override val location: Location, val comments: Array[StringValue]) extends BooleanValue with YamlValue[Boolean] {
  override def evaluate(implicit ctx: EvaluationContext): T = value

}

class YamlReaderSettings extends ConfigurableMaxEntityCount

object YamlReader {

  val COMMENTS_SCHEMA_PROPERTY = "comments"
  val TAG_SCHEMA_PROPERTY = "tag"
  val DOCUMENTS_SCHEMA_PROPERTY = "documents"

  def apply(file: File, settings: YamlReaderSettings)(implicit ctx: EvaluationContext): YamlReader = {
    new YamlReader(SourceProvider(file), settings)
  }

  def apply(inputStream: InputStream, settings: YamlReaderSettings)(implicit ctx: EvaluationContext): YamlReader = {
    new YamlReader(SourceProvider(inputStream), settings)
  }

  def apply(sourceProvider: SourceProvider, settings: YamlReaderSettings)(implicit ctx: EvaluationContext): YamlReader = {
    new YamlReader(sourceProvider, settings)
  }

  def apply(content: String, settings: YamlReaderSettings)(implicit ctx: EvaluationContext): YamlReader = {
    new YamlReader(SourceProvider(content), settings)
  }
}
