package org.mule.weave.v2.module.yaml

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.BooleanType
import org.mule.weave.v2.model.types.NullType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.RangeType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.reader.ConfigurableDeferred
import org.mule.weave.v2.module.writer.BufferedIOWriter
import org.mule.weave.v2.module.writer.ConfigurableEncoding
import org.mule.weave.v2.module.writer.ConfigurableSkipDeclaration
import org.mule.weave.v2.module.writer.ConfigurableSkipNullWriter
import org.mule.weave.v2.module.writer.TargetProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.module.yaml.YamlWriter.spaceIndents
import org.mule.weave.v2.parser.location.LocationCapable
import org.yaml.model.NoMark
import org.yaml.render.ScalarRender

import java.io.OutputStream
import scala.collection.mutable

class YamlWriter(var os: OutputStream, val settings: YamlWriterSettings)(implicit ctx: EvaluationContext) extends Writer {
  //Lazy uses the settings
  lazy val writer: BufferedIOWriter = {
    BufferedIOWriter(os, settings.charset(ctx), settings.bufferSize)
  }

  private var indent: Int = 0

  //0 means no value has been written
  //1 means is the root value
  // bigger means is not the root value
  var rootValue: Int = 0

  protected override def doWriteValue(v: Value[_])(implicit ctx: EvaluationContext): Unit = {
    writeAllComments(v)
    writeValueWithoutComment(v)
  }

  def writeStringValue(v: Value[_]): Unit = {
    v.schema match {
      case Some(theSchema) => {
        theSchema.valueOf(YamlReader.TAG_SCHEMA_PROPERTY) match {
          case Some(value) => {
            writer.write(StringType.coerce(value).evaluate.toString)
            writer.write(" ")
            writeString(v.evaluate.toString)
          }
          case None => {
            writeString(v.evaluate.toString)
          }
        }
      }
      case None => {
        writeString(v.evaluate.toString)
      }
    }
  }

  private def writeValueWithoutComment(v: Value[_]): Unit = {
    rootValue = rootValue + 1
    v match {
      case obj if ObjectType.accepts(obj) => writeObject(ObjectType.coerce(v))
      case array if ArrayType.accepts(array) => {
        if (rootValue == 1 && isRootOfDocuments(v)) {
          val arraySeq = v.evaluate.asInstanceOf[ArrayType.T]
          val iterator = arraySeq.toIterator()
          var root = true
          while (iterator.hasNext) {
            val documentRoot = iterator.next()
            if (!root) {
              writer.write("\n---\n")
            }
            writeValue(documentRoot)
            root = false
          }
        } else {
          writeArray(ArrayType.coerce(v))
        }
      }
      case range if RangeType.accepts(range)       => writeArray(ArrayType.coerce(v))
      case number if NumberType.accepts(number)    => writeNumber(NumberType.coerce(v))
      case boolean if BooleanType.accepts(boolean) => writeBoolean(BooleanType.coerce(v))
      case nill if NullType.accepts(nill)          => writeNull(v)
      case str if StringType.accepts(str)          => writeStringValue(v)
      case _                                       => writeString(StringType.withSchema(v.schema).coerce(v).evaluate.asInstanceOf[String])
    }
  }

  private def writeAllComments(v: Value[_]): Unit = {
    v.schema.foreach((s) => {
      s.valueOf(YamlReader.COMMENTS_SCHEMA_PROPERTY)
        .foreach((sv) => {
          sv.evaluate match {
            case arr: types.ArrayType.T => {
              val comments = arr.toIterator()
              comments.foreach((s) => {
                val commentValue: String = StringType.coerce(s).evaluate.toString
                writeComment(commentValue)
              })
            }
            case s: types.StringType.T => {
              writeComment(s.toString)
            }
            case _ => {}
          }
        })
    })

  }

  private def writeComment(commentValue: String): Unit = {
    val linesIterator = commentValue.linesIterator
    linesIterator.foreach((line) => {
      writer.write("#" + line)
      newLine()
    })
  }

  private def newLine() = {
    newline(indent)
  }

  private def isScalar(v: Value[_])(implicit ctx: EvaluationContext): Boolean = {
    v match {
      case number if NumberType.accepts(number)    => true
      case boolean if BooleanType.accepts(boolean) => true
      case nill if NullType.accepts(nill)          => true
      case str if StringType.accepts(str)          => true
      case _                                       => false
    }
  }

  def isRootOfDocuments(value: Value[_])(implicit ctx: EvaluationContext): Boolean = {
    value.schema
      .flatMap(s => s.valueOf(YamlReader.DOCUMENTS_SCHEMA_PROPERTY))
      .exists((v) =>
        v.evaluate match {
          case b: Boolean => b
          case _          => false
        })
  }

  def writeArray(value: Value[ArraySeq])(implicit ctx: EvaluationContext): Unit = {
    val arraySeq = value.evaluate
    val filtered = if (settings.skipNullOnArrays) {
      arraySeq.toIterator().filterNot((v: Value[_]) => NullType.accepts(v))
    } else {
      arraySeq.toIterator()
    }

    if (filtered.isEmpty) {
      writer.write("[]")
    } else {
      //first item
      if (filtered.hasNext) {
        val v = filtered.next()
        writeAllComments(v)
        writer.write(YamlWriter.hyphen)
        indent += 1
        writeArrayValue(v)
        indent -= 1
      }

      filtered.foreach(v => {
        newLine()
        writeAllComments(v)
        writer.write(YamlWriter.hyphen)
        indent += 1
        writeArrayValue(v)
        indent -= 1
      })
    }

  }

  def writeObject(value: Value[ObjectSeq])(implicit ctx: EvaluationContext): Unit = {

    val objectSeq: ObjectSeq = value.evaluate
    val filtered = if (settings.skipNullOnObjects) {
      objectSeq.toIterator().filterNot((v: KeyValuePair) => NullType.accepts(v._2))
    } else {
      objectSeq.toIterator()
    }

    if (filtered.isEmpty) {
      writer.write("{}")
    } else {
      if (filtered.hasNext) {
        val KeyValuePair(k, v, _) = filtered.next()
        writeAllComments(k)
        writeString(k.evaluate.name)
        indent += 1
        keyValueSeparator(v)
        writeValue(v)
        indent -= 1
      }
      filtered.foreach {
        case KeyValuePair(k, v, _) =>
          newLine()
          writeAllComments(k)
          writeString(k.evaluate.name)
          indent += 1
          keyValueSeparator(v)
          writeValue(v)
          indent -= 1
      }
    }

  }

  private def writeArrayValue(v: Value[_])(implicit ctx: EvaluationContext): Unit = {
    if (isScalar(v)) {
      writer.write(YamlWriter.space)
    } else {
      newLine()
    }
    writeValueWithoutComment(v)
  }

  private def keyValueSeparator(v: Value[_])(implicit ctx: EvaluationContext): Unit = {
    writer.write(YamlWriter.colon)
    if (isScalar(v)) {
      writer.write(YamlWriter.space)
    } else {
      newLine()
    }
  }

  @inline
  private def newline(indent: Int): Unit = {
    writer.write("\n")
    if (indent > 0) {
      writer.write(YamlWriter.getIndentSpaces(indent))
    }
  }

  def writeBoolean(value: Value[Boolean])(implicit ctx: EvaluationContext): Unit = {
    writer.write(value.evaluate.toString)
  }

  def writeNull(value: Value[_])(implicit ctx: EvaluationContext): Unit = {
    value.schema match {
      case Some(theSchema) if (theSchema.inf.isDefined) => {
        theSchema.inf match {
          case Some(NumberValue.NEGATIVE_INF_VALUE) => {
            writer.write("-.inf")
          }
          case _ => {
            writer.write(".inf")
          }
        }
      }
      case Some(theSchema) if (theSchema.nan.isDefined) => {
        writer.write(".NaN")
      }
      case _ => {
        writer.write("null")
      }
    }
  }

  private def writeNumber(value: Value[Number])(implicit ctx: EvaluationContext): Unit = {
    writer.write(value.evaluate.toCanonicalString)
  }

  private def writeString(strValue: String): Unit = {
    val scalarMark = ScalarRender.renderScalar(strValue, mustBeString = true, mark = NoMark, indentation = (indent - 1) * spaceIndents)
    writer.write(scalarMark.toString)
  }

  override def flush(): Unit = writer.flush()

  override def result: OutputStream = os

  override def close(): Unit = writer.close()

  override def startDocument(location: LocationCapable): Unit = {
    if (settings.writeDeclaration) {
      writer.write("%YAML 1.2\n---\n")
    }
  }

  override def dataFormat: Option[DataFormat[_, _]] = Some(new YamlDataFormat)
}

object YamlWriter {
  val colon: String = ":"
  val pipe: String = "|"
  val space: String = " "
  val comma: String = ","
  val open_curly: String = "{"
  val close_curly: String = "}"
  val open_square: String = "["
  val close_square: String = "]"
  val hyphen: String = "-"

  val spaceIndents = 2

  private val indentSpaceCache = new mutable.ListBuffer[String]()
  getIndentSpaces(25)

  def getIndentSpaces(amount: Int): String = {
    if (indentSpaceCache.lengthCompare(amount) <= 0) {
      this.synchronized {
        if (indentSpaceCache.lengthCompare(amount) <= 0) {
          var i = indentSpaceCache.size
          while (amount >= i) {
            indentSpaceCache.+=(" " * i * spaceIndents)
            i = i + 1
          }
        }
      }
    }
    indentSpaceCache.apply(amount)
  }

  def apply(tp: TargetProvider, settings: YamlWriterSettings)(implicit ctx: EvaluationContext): YamlWriter = new YamlWriter(tp.asOutputStream, settings)
}

class YamlWriterSettings(val dataFormat: DataFormat[_, _]) extends ConfigurableEncoding with ConfigurableDeferred with ConfigurableSkipNullWriter with ConfigurableSkipDeclaration {
  val SKIP_NULL_ON_ARRAYS: String = "arrays"
  val SKIP_NULL_ON_OBJECTS: String = "objects"
  val SKIP_NULL_ON_EVERYWHERE: String = "everywhere"

  lazy val skipNullOnArrays: Boolean = skipNullOn.isDefined && (skipNullOn.get == SKIP_NULL_ON_ARRAYS || skipNullOn.get == SKIP_NULL_ON_EVERYWHERE)
  lazy val skipNullOnObjects: Boolean = skipNullOn.isDefined && (skipNullOn.get == SKIP_NULL_ON_OBJECTS || skipNullOn.get == SKIP_NULL_ON_EVERYWHERE)

  override def possibleToSkipNullsOn: List[String] = List(SKIP_NULL_ON_ARRAYS, SKIP_NULL_ON_OBJECTS, SKIP_NULL_ON_EVERYWHERE)

  override def skipOnNullDescriptionUrl: String = "data-format/yaml/skipNullOn.asciidoc"
}
