package org.mule.weave.v2.module.yaml

import java.nio.charset.Charset
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.service.CharsetProviderService
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.writer.DeferredWriter
import org.mule.weave.v2.module.writer.TargetProvider
import org.mule.weave.v2.parser.module.MimeType

class YamlDataFormat extends DataFormat[YamlReaderSettings, YamlWriterSettings] {

  /**
    * The label of this DataFormat
    *
    * @return The label
    */
  override def label(): String = "YAML"

  override def defaultCharset: Option[CharsetProviderService#Charset] = {
    Option(Charset.forName("UTF-8"))
  }

  /**
    * The name of this module
    *
    * @return
    */
  override def name(): String = "yaml"

  /**
    * Create a new instance of the writer using the provided target if present else it will create a new target
    *
    * @param target The target where the result will be written
    * @return The writer
    */
  override def writer(target: Option[Any], outputMimeType: MimeType = defaultMimeType)(implicit ctx: EvaluationContext): DeferredWriter[YamlWriterSettings] = {
    DeferredWriter(YamlWriter.apply, TargetProvider(target), createWriterSettings())
  }

  /**
    * Creates a new instance of the writer using the given source provider
    *
    * @param source The source provider for this reader
    * @return The Reader
    */
  override def reader(source: SourceProvider)(implicit ctx: EvaluationContext): YamlReader = {
    new YamlReader(source, createReaderSettings())
  }

  /**
    * The content types supported by this module
    *
    * @return
    */
  override val defaultMimeType: MimeType = new MimeType("application", "yaml")

  //

  /**
    * Returns the mimeTypes accepted by this
    *
    * @return The list of mime types
    */
  override val acceptedMimeTypes: Seq[MimeType] = Seq(
    new MimeType("application", "yaml"),
    new MimeType("text", "yaml"),
    new MimeType("application", "x-yaml"),
    new MimeType("text", "x-yaml"))

  /**
    * The file extensions associated with this DataFormat. (".json", ".xml", etc...)
    * Note: This can be empty
    */
  override def fileExtensions: Seq[String] = Seq(".yml", ".yaml")

  /**
    * Creates a new instance of the reader configuration settings
    *
    * @return
    */
  override def readerSettings(): YamlReaderSettings = new YamlReaderSettings()

  /**
    * Creates a new instance of the writer configuraion settings
    *
    * @return
    */
  override def writerSettings(): YamlWriterSettings = new YamlWriterSettings(this)
}
