export type Attributes = {
    [key: string]: string;
};
export interface IParser {
    startElement(tag: string, attrs: Attributes): void;
    endElement(tag: string): void;
    characterData(content: string): void;
}
/**
 * Expat XML parser WASM library, provides a simplified wrapper around the Expat XML Parser library.
 *
 * See [libexpat.github.io](https://libexpat.github.io/) for c++ details.
 *
 * ```ts
 * import { Expat } from "@hpcc-js/wasm/expat";
 *
 * const expat = await Expat.load();
 *
 * const xml = ` \
 *     <root>
 *         <child xxx="yyy">content</child>
 *     </root>
 * `;
 *
 * const callback = {
 *     startElement(tag, attrs) { console.log("start", tag, attrs); },
 *     endElement(tag) { console.log("end", tag); },
 *     characterData(content) { console.log("characterData", content); }
 * };
 *
 * expat.parse(xml, callback);
 * ```
 
 */
export declare class Expat {
    protected _module: any;
    private constructor();
    /**
     * Compiles and instantiates the raw wasm.
     *
     * ::: info
     * In general WebAssembly compilation is disallowed on the main thread if the buffer size is larger than 4KB, hence forcing `load` to be asynchronous;
     * :::
     *
     * @returns A promise to an instance of the Expat class.
     */
    static load(): Promise<Expat>;
    /**
     * Unloades the compiled wasm instance.
     */
    static unload(): void;
    /**
     *
     * @returns The Expat c++ version
     */
    version(): string;
    /**
     * Parses the XML with suitable callbacks.
     *
     * :::tip
     * The _IParser.characterData_ callback method can get called several times for a single tag element.
     * :::
     *
     * @param xml string containing XML
     * @param callback Callback interface
     * @returns `true`|`false` if the XML parse succeeds.
     */
    parse(xml: string, callback: IParser): boolean;
}
export declare class StackElement {
    readonly tag: string;
    readonly attrs: Attributes;
    private _content;
    get content(): string;
    constructor(tag: string, attrs: Attributes);
    appendContent(content: string): void;
}
export declare class StackParser implements IParser {
    private _stack;
    parse(xml: string): Promise<boolean>;
    top(): StackElement;
    startElement(tag: string, attrs: Attributes): StackElement;
    endElement(tag: string): StackElement;
    characterData(content: string): void;
}
//# sourceMappingURL=expat.d.ts.map