/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.server.http.cypher;

import java.time.Clock;
import java.time.Duration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Predicate;
import org.neo4j.function.Predicates;
import org.neo4j.internal.kernel.api.security.LoginContext;
import org.neo4j.logging.InternalLog;
import org.neo4j.logging.InternalLogProvider;
import org.neo4j.memory.HeapEstimator;
import org.neo4j.memory.MemoryPool;
import org.neo4j.server.http.cypher.InvalidConcurrentTransactionAccess;
import org.neo4j.server.http.cypher.InvalidTransactionId;
import org.neo4j.server.http.cypher.TransactionHandle;
import org.neo4j.server.http.cypher.TransactionLifecycleException;
import org.neo4j.server.http.cypher.TransactionRegistry;
import org.neo4j.server.http.cypher.TransactionTerminationHandle;
import org.neo4j.util.VisibleForTesting;

public class TransactionHandleRegistry
implements TransactionRegistry {
    @VisibleForTesting
    public static final long ACTIVE_TRANSACTION_SHALLOW_SIZE = HeapEstimator.shallowSizeOfInstance(ActiveTransaction.class);
    @VisibleForTesting
    public static final long SUSPENDED_TRANSACTION_SHALLOW_SIZE = HeapEstimator.shallowSizeOf(SuspendedTransaction.class);
    private final AtomicLong idGenerator = new AtomicLong(0L);
    private final ConcurrentHashMap<Long, TransactionMarker> registry = new ConcurrentHashMap(64);
    private final Clock clock;
    private final InternalLog log;
    private final Duration transactionTimeout;
    private final MemoryPool memoryPool;

    public TransactionHandleRegistry(Clock clock, Duration transactionTimeout, InternalLogProvider logProvider, MemoryPool memoryPool) {
        this.clock = clock;
        this.transactionTimeout = transactionTimeout;
        this.log = logProvider.getLog(this.getClass());
        this.memoryPool = memoryPool;
    }

    @Override
    public long begin(TransactionHandle handle) {
        this.memoryPool.reserveHeap(ACTIVE_TRANSACTION_SHALLOW_SIZE);
        long id = this.idGenerator.incrementAndGet();
        if (null == this.registry.putIfAbsent(id, new ActiveTransaction(handle))) {
            return id;
        }
        this.memoryPool.releaseHeap(ACTIVE_TRANSACTION_SHALLOW_SIZE);
        throw new IllegalStateException("Attempt to begin transaction for id that was already registered");
    }

    @Override
    public long release(long id, TransactionHandle transactionHandle) {
        TransactionMarker marker = this.registry.get(id);
        if (null == marker) {
            throw new IllegalStateException("Trying to suspend unregistered transaction");
        }
        if (marker.isSuspended()) {
            throw new IllegalStateException("Trying to suspend transaction that was already suspended");
        }
        this.memoryPool.reserveHeap(SUSPENDED_TRANSACTION_SHALLOW_SIZE);
        SuspendedTransaction suspendedTx = new SuspendedTransaction(this, marker.getActiveTransaction(), transactionHandle);
        if (!this.registry.replace(id, marker, suspendedTx)) {
            this.memoryPool.releaseHeap(SUSPENDED_TRANSACTION_SHALLOW_SIZE);
            throw new IllegalStateException("Trying to suspend transaction that has been concurrently suspended");
        }
        return this.computeNewExpiryTime(suspendedTx.getLastActiveTimestamp());
    }

    private long computeNewExpiryTime(long lastActiveTimestamp) {
        return lastActiveTimestamp + this.transactionTimeout.toMillis();
    }

    @Override
    public LoginContext getLoginContextForTransaction(long id) throws InvalidTransactionId {
        TransactionMarker marker = this.registry.get(id);
        if (marker == null) {
            throw InvalidTransactionId.transactionDoesNotExists(id);
        }
        return marker.getLoginContext();
    }

    @Override
    public TransactionHandle acquire(long id) throws TransactionLifecycleException {
        TransactionMarker marker = this.registry.get(id);
        if (null == marker) {
            throw InvalidTransactionId.transactionDoesNotExists(id);
        }
        SuspendedTransaction transaction = marker.getSuspendedTransaction();
        if (this.registry.replace(id, marker, marker.getActiveTransaction())) {
            this.memoryPool.releaseHeap(SUSPENDED_TRANSACTION_SHALLOW_SIZE);
            return transaction.transactionHandle;
        }
        throw new InvalidConcurrentTransactionAccess();
    }

    @Override
    public void forget(long id) {
        TransactionMarker marker = this.registry.get(id);
        if (null == marker) {
            throw new IllegalStateException("Could not finish unregistered transaction");
        }
        if (marker.isSuspended()) {
            throw new IllegalStateException("Cannot finish suspended registered transaction");
        }
        if (!this.registry.remove(id, marker)) {
            throw new IllegalStateException("Trying to finish transaction that has been concurrently finished or suspended");
        }
        this.memoryPool.releaseHeap(ACTIVE_TRANSACTION_SHALLOW_SIZE);
    }

    @Override
    public TransactionHandle terminate(long id) throws TransactionLifecycleException {
        TransactionMarker marker = this.registry.get(id);
        if (null == marker) {
            throw InvalidTransactionId.transactionDoesNotExists(id);
        }
        this.memoryPool.releaseHeap(ACTIVE_TRANSACTION_SHALLOW_SIZE);
        TransactionTerminationHandle handle = marker.getActiveTransaction().getTerminationHandle();
        handle.terminate();
        try {
            SuspendedTransaction transaction = marker.getSuspendedTransaction();
            if (this.registry.replace(id, marker, marker.getActiveTransaction())) {
                this.memoryPool.releaseHeap(SUSPENDED_TRANSACTION_SHALLOW_SIZE);
                return transaction.transactionHandle;
            }
        }
        catch (InvalidConcurrentTransactionAccess invalidConcurrentTransactionAccess) {
            // empty catch block
        }
        return null;
    }

    @Override
    public void rollbackAllSuspendedTransactions() {
        this.rollbackSuspended(Predicates.alwaysTrue());
    }

    @Override
    public void rollbackSuspendedTransactionsIdleSince(long oldestLastActiveTime) {
        this.rollbackSuspended(item -> {
            try {
                SuspendedTransaction transaction = item.getSuspendedTransaction();
                return transaction.lastActiveTimestamp < oldestLastActiveTime;
            }
            catch (InvalidConcurrentTransactionAccess concurrentTransactionAccessError) {
                throw new RuntimeException((Throwable)((Object)concurrentTransactionAccessError));
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void rollbackSuspended(Predicate<TransactionMarker> predicate) {
        HashSet<Long> candidateTransactionIdsToRollback = new HashSet<Long>();
        for (Map.Entry<Long, TransactionMarker> entry : this.registry.entrySet()) {
            TransactionMarker marker = entry.getValue();
            if (!marker.isSuspended() || !predicate.test(marker)) continue;
            candidateTransactionIdsToRollback.add(entry.getKey());
        }
        Iterator<Map.Entry<Long, TransactionMarker>> iterator = candidateTransactionIdsToRollback.iterator();
        while (iterator.hasNext()) {
            TransactionHandle handle;
            long id = (Long)((Object)iterator.next());
            try {
                handle = this.acquire(id);
            }
            catch (TransactionLifecycleException invalidTransactionId) {
                continue;
            }
            try {
                handle.forceRollback();
                this.log.info(String.format("Transaction with id %d has been automatically rolled back due to transaction timeout.", id));
            }
            catch (Throwable e) {
                this.log.error(String.format("Transaction with id %d failed to roll back.", id), e);
            }
            finally {
                this.forget(id);
            }
        }
    }

    private static class ActiveTransaction
    extends TransactionMarker {
        final TransactionTerminationHandle terminationHandle;

        private ActiveTransaction(TransactionHandle terminationHandle) {
            super(terminationHandle);
            this.terminationHandle = terminationHandle;
        }

        TransactionTerminationHandle getTerminationHandle() {
            return this.terminationHandle;
        }

        @Override
        ActiveTransaction getActiveTransaction() {
            return this;
        }

        @Override
        SuspendedTransaction getSuspendedTransaction() throws InvalidConcurrentTransactionAccess {
            throw new InvalidConcurrentTransactionAccess();
        }

        @Override
        boolean isSuspended() {
            return false;
        }
    }

    private static abstract class TransactionMarker {
        protected TransactionHandle transactionHandle;

        protected TransactionMarker(TransactionHandle transactionHandle) {
            this.transactionHandle = transactionHandle;
        }

        abstract ActiveTransaction getActiveTransaction();

        abstract SuspendedTransaction getSuspendedTransaction() throws InvalidConcurrentTransactionAccess;

        abstract boolean isSuspended();

        LoginContext getLoginContext() {
            return this.transactionHandle.loginContext();
        }
    }

    private class SuspendedTransaction
    extends TransactionMarker {
        final ActiveTransaction activeMarker;
        final long lastActiveTimestamp;

        private SuspendedTransaction(TransactionHandleRegistry transactionHandleRegistry, ActiveTransaction activeMarker, TransactionHandle transactionHandle) {
            super(transactionHandle);
            this.activeMarker = activeMarker;
            this.lastActiveTimestamp = transactionHandleRegistry.clock.millis();
        }

        @Override
        ActiveTransaction getActiveTransaction() {
            return this.activeMarker;
        }

        @Override
        SuspendedTransaction getSuspendedTransaction() {
            return this;
        }

        @Override
        boolean isSuspended() {
            return true;
        }

        @Override
        LoginContext getLoginContext() {
            return this.transactionHandle.loginContext();
        }

        long getLastActiveTimestamp() {
            return this.lastActiveTimestamp;
        }
    }
}

