/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.integration;

import io.netty.bootstrap.Bootstrap;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import org.junit.Assert;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.neo4j.driver.AuthToken;
import org.neo4j.driver.Config;
import org.neo4j.driver.Driver;
import org.neo4j.driver.Logging;
import org.neo4j.driver.QueryRunner;
import org.neo4j.driver.Result;
import org.neo4j.driver.Session;
import org.neo4j.driver.Transaction;
import org.neo4j.driver.Values;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.internal.BoltServerAddress;
import org.neo4j.driver.internal.ConnectionSettings;
import org.neo4j.driver.internal.DriverFactory;
import org.neo4j.driver.internal.async.connection.ChannelConnector;
import org.neo4j.driver.internal.async.pool.ConnectionPoolImpl;
import org.neo4j.driver.internal.async.pool.PoolSettings;
import org.neo4j.driver.internal.cluster.RoutingContext;
import org.neo4j.driver.internal.cluster.RoutingSettings;
import org.neo4j.driver.internal.metrics.DevNullMetricsListener;
import org.neo4j.driver.internal.metrics.MetricsListener;
import org.neo4j.driver.internal.metrics.MetricsProvider;
import org.neo4j.driver.internal.retry.RetrySettings;
import org.neo4j.driver.internal.security.SecurityPlan;
import org.neo4j.driver.internal.security.SecurityPlanImpl;
import org.neo4j.driver.internal.spi.Connection;
import org.neo4j.driver.internal.spi.ConnectionPool;
import org.neo4j.driver.internal.util.Clock;
import org.neo4j.driver.internal.util.EnabledOnNeo4jWith;
import org.neo4j.driver.internal.util.Neo4jFeature;
import org.neo4j.driver.reactive.RxResult;
import org.neo4j.driver.reactive.RxSession;
import org.neo4j.driver.reactive.RxTransaction;
import org.neo4j.driver.summary.ResultSummary;
import org.neo4j.driver.util.DatabaseExtension;
import org.neo4j.driver.util.ParallelizableIT;
import org.neo4j.driver.util.TestUtil;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.test.StepVerifier;

@ParallelizableIT
class ConnectionHandlingIT {
    @RegisterExtension
    static final DatabaseExtension neo4j = new DatabaseExtension();
    private Driver driver;
    private MemorizingConnectionPool connectionPool;

    ConnectionHandlingIT() {
    }

    @BeforeEach
    void createDriver() {
        DriverFactoryWithConnectionPool driverFactory = new DriverFactoryWithConnectionPool();
        AuthToken auth = neo4j.authToken();
        RoutingSettings routingSettings = RoutingSettings.DEFAULT;
        RetrySettings retrySettings = RetrySettings.DEFAULT;
        this.driver = driverFactory.newInstance(neo4j.uri(), auth, routingSettings, retrySettings, Config.builder().withFetchSize(1L).build(), SecurityPlanImpl.insecure());
        this.connectionPool = driverFactory.connectionPool;
        this.connectionPool.startMemorizing();
    }

    @AfterEach
    void closeDriver() {
        this.driver.close();
    }

    @Test
    void connectionUsedForSessionRunReturnedToThePoolWhenResultConsumed() {
        Result result = this.createNodesInNewSession(12);
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.never())).release();
        result.consume();
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assertions.assertSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection1)).release();
    }

    @Test
    void connectionUsedForSessionRunReturnedToThePoolWhenResultSummaryObtained() {
        Result result = this.createNodesInNewSession(5);
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.never())).release();
        ResultSummary summary = result.consume();
        Assertions.assertEquals((int)5, (int)summary.counters().nodesCreated());
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assertions.assertSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection1)).release();
    }

    @Test
    void connectionUsedForSessionRunReturnedToThePoolWhenResultFetchedInList() {
        Result result = this.createNodesInNewSession(2);
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.never())).release();
        List records = result.list();
        Assertions.assertEquals((int)2, (int)records.size());
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assertions.assertSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection1)).release();
    }

    @Test
    void connectionUsedForSessionRunReturnedToThePoolWhenSingleRecordFetched() {
        Result result = this.createNodesInNewSession(1);
        Assertions.assertNotNull((Object)result.single());
        Connection connection = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection)).release();
    }

    @Test
    void connectionUsedForSessionRunReturnedToThePoolWhenResultFetchedAsIterator() {
        Result result = this.createNodesInNewSession(6);
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.never())).release();
        int seenRecords = 0;
        while (result.hasNext()) {
            Assertions.assertNotNull((Object)result.next());
            ++seenRecords;
        }
        Assertions.assertEquals((int)6, (int)seenRecords);
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assertions.assertSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection1)).release();
    }

    @Test
    void connectionUsedForSessionRunReturnedToThePoolOnServerFailure() {
        try (Session session = this.driver.session();){
            Assertions.assertThrows(ClientException.class, () -> session.run("UNWIND range(10, -1, 0) AS i CREATE (n {index: 10/i}) RETURN n").consume());
            Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
            ((Connection)Mockito.verify((Object)connection1)).release();
        }
    }

    @Test
    void connectionUsedForTransactionReturnedToThePoolWhenTransactionCommitted() {
        Session session = this.driver.session();
        Transaction tx = session.beginTransaction();
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.never())).release();
        Result result = this.createNodes(5, (QueryRunner)tx);
        int size = result.list().size();
        tx.commit();
        tx.close();
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assertions.assertSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection1)).release();
        Assertions.assertEquals((int)5, (int)size);
    }

    @Test
    void connectionUsedForTransactionReturnedToThePoolWhenTransactionRolledBack() {
        Session session = this.driver.session();
        Transaction tx = session.beginTransaction();
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.never())).release();
        Result result = this.createNodes(8, (QueryRunner)tx);
        int size = result.list().size();
        tx.rollback();
        tx.close();
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assertions.assertSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection1)).release();
        Assertions.assertEquals((int)8, (int)size);
    }

    @Test
    void connectionUsedForTransactionReturnedToThePoolWhenTransactionFailsToCommitted() {
        Assumptions.assumeTrue((boolean)neo4j.isNeo4j44OrEarlier());
        try (Session session = this.driver.session();){
            session.run("CREATE CONSTRAINT ON (book:Library) ASSERT exists(book.isbn)");
        }
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.atLeastOnce())).release();
        Session session = this.driver.session();
        Transaction tx = session.beginTransaction();
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection2, (VerificationMode)Mockito.never())).release();
        tx.run("CREATE (:Library)");
        Assertions.assertThrows(ClientException.class, () -> ((Transaction)tx).commit());
        ((Connection)Mockito.verify((Object)connection2)).release();
    }

    @Test
    void connectionUsedForSessionRunReturnedToThePoolWhenSessionClose() {
        Session session = this.driver.session();
        this.createNodes(12, (QueryRunner)session);
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.never())).release();
        session.close();
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assertions.assertSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.times((int)2))).release();
    }

    @Test
    void connectionUsedForBeginTxReturnedToThePoolWhenSessionClose() {
        Session session = this.driver.session();
        Transaction tx = session.beginTransaction();
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.never())).release();
        session.close();
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assertions.assertSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.times((int)2))).release();
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void sessionCloseShouldReleaseConnectionUsedBySessionRun() {
        RxSession session = this.driver.rxSession();
        RxResult res = session.run("UNWIND [1,2,3,4] AS a RETURN a");
        StepVerifier.create((Publisher)Flux.from((Publisher)res.keys())).expectNext(Collections.singletonList("a")).verifyComplete();
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.never())).release();
        StepVerifier.create((Publisher)Mono.from((Publisher)session.close())).verifyComplete();
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assertions.assertSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.times((int)2))).release();
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void resultRecordsShouldReleaseConnectionUsedBySessionRun() {
        RxSession session = this.driver.rxSession();
        RxResult res = session.run("UNWIND [1,2,3,4] AS a RETURN a");
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        Assert.assertNull((Object)connection1);
        StepVerifier.create((Publisher)Flux.from((Publisher)res.records()).map(record -> record.get("a").asInt())).expectNext((Object)1, (Object)2, (Object)3, (Object)4).verifyComplete();
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assert.assertNotSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection2)).release();
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void resultSummaryShouldReleaseConnectionUsedBySessionRun() {
        RxSession session = this.driver.rxSession();
        RxResult res = session.run("UNWIND [1,2,3,4] AS a RETURN a");
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        Assert.assertNull((Object)connection1);
        StepVerifier.create((Publisher)Mono.from((Publisher)res.consume())).expectNextCount(1L).verifyComplete();
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assert.assertNotSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection2)).release();
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void txCommitShouldReleaseConnectionUsedByBeginTx() {
        AtomicReference connection1Ref = new AtomicReference();
        Function<RxSession, Publisher> sessionToRecordPublisher = session -> Flux.usingWhen((Publisher)Mono.fromDirect((Publisher)session.beginTransaction()), tx -> {
            connection1Ref.set(this.connectionPool.lastAcquiredConnectionSpy);
            ((Connection)Mockito.verify(connection1Ref.get(), (VerificationMode)Mockito.never())).release();
            return tx.run("UNWIND [1,2,3,4] AS a RETURN a").records();
        }, RxTransaction::commit, (tx, error) -> tx.rollback(), RxTransaction::rollback);
        Flux resultsFlux = Flux.usingWhen((Publisher)Mono.fromSupplier(() -> ((Driver)this.driver).rxSession()), sessionToRecordPublisher, session -> {
            Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
            Assertions.assertSame(connection1Ref.get(), (Object)connection2);
            ((Connection)Mockito.verify(connection1Ref.get())).release();
            return Mono.empty();
        }, (session, error) -> session.close(), RxSession::close).map(record -> record.get("a").asInt());
        StepVerifier.create((Publisher)resultsFlux).expectNext((Object)1, (Object)2, (Object)3, (Object)4).expectComplete().verify();
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void txRollbackShouldReleaseConnectionUsedByBeginTx() {
        AtomicReference connection1Ref = new AtomicReference();
        Function<RxSession, Publisher> sessionToRecordPublisher = session -> Flux.usingWhen((Publisher)Mono.fromDirect((Publisher)session.beginTransaction()), tx -> {
            connection1Ref.set(this.connectionPool.lastAcquiredConnectionSpy);
            ((Connection)Mockito.verify(connection1Ref.get(), (VerificationMode)Mockito.never())).release();
            return tx.run("UNWIND [1,2,3,4] AS a RETURN a").records();
        }, RxTransaction::rollback, (tx, error) -> tx.rollback(), RxTransaction::rollback);
        Flux resultsFlux = Flux.usingWhen((Publisher)Mono.fromSupplier(() -> ((Driver)this.driver).rxSession()), sessionToRecordPublisher, session -> {
            Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
            Assertions.assertSame(connection1Ref.get(), (Object)connection2);
            ((Connection)Mockito.verify(connection1Ref.get())).release();
            return Mono.empty();
        }, (session, error) -> session.close(), RxSession::close).map(record -> record.get("a").asInt());
        StepVerifier.create((Publisher)resultsFlux).expectNext((Object)1, (Object)2, (Object)3, (Object)4).expectComplete().verify();
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void sessionCloseShouldReleaseConnectionUsedByBeginTx() {
        RxSession session = this.driver.rxSession();
        Publisher tx = session.beginTransaction();
        StepVerifier.create((Publisher)Mono.from((Publisher)tx)).expectNextCount(1L).verifyComplete();
        Connection connection1 = this.connectionPool.lastAcquiredConnectionSpy;
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.never())).release();
        StepVerifier.create((Publisher)Mono.from((Publisher)session.close())).verifyComplete();
        Connection connection2 = this.connectionPool.lastAcquiredConnectionSpy;
        Assertions.assertSame((Object)connection1, (Object)connection2);
        ((Connection)Mockito.verify((Object)connection1, (VerificationMode)Mockito.times((int)2))).release();
    }

    private Result createNodesInNewSession(int nodesToCreate) {
        return this.createNodes(nodesToCreate, (QueryRunner)this.driver.session());
    }

    private Result createNodes(int nodesToCreate, QueryRunner queryRunner) {
        return queryRunner.run("UNWIND range(1, $nodesToCreate) AS i CREATE (n {index: i}) RETURN n", Values.parameters((Object[])new Object[]{"nodesToCreate", nodesToCreate}));
    }

    private static class MemorizingConnectionPool
    extends ConnectionPoolImpl {
        Connection lastAcquiredConnectionSpy;
        boolean memorize;

        MemorizingConnectionPool(ChannelConnector connector, Bootstrap bootstrap, PoolSettings settings, Logging logging, Clock clock, boolean ownsEventLoopGroup) {
            super(connector, bootstrap, settings, (MetricsListener)DevNullMetricsListener.INSTANCE, logging, clock, ownsEventLoopGroup);
        }

        void startMemorizing() {
            this.memorize = true;
        }

        public CompletionStage<Connection> acquire(BoltServerAddress address) {
            Connection connection = (Connection)TestUtil.await(super.acquire(address));
            if (this.memorize) {
                if (!Mockito.mockingDetails((Object)connection).isSpy()) {
                    connection = (Connection)Mockito.spy((Object)connection);
                }
                this.lastAcquiredConnectionSpy = connection;
            }
            return CompletableFuture.completedFuture(connection);
        }
    }

    private static class DriverFactoryWithConnectionPool
    extends DriverFactory {
        MemorizingConnectionPool connectionPool;

        private DriverFactoryWithConnectionPool() {
        }

        protected ConnectionPool createConnectionPool(AuthToken authToken, SecurityPlan securityPlan, Bootstrap bootstrap, MetricsProvider ignored, Config config, boolean ownsEventLoopGroup, RoutingContext routingContext) {
            ConnectionSettings connectionSettings = new ConnectionSettings(authToken, "test", 1000);
            PoolSettings poolSettings = new PoolSettings(config.maxConnectionPoolSize(), config.connectionAcquisitionTimeoutMillis(), config.maxConnectionLifetimeMillis(), config.idleTimeBeforeConnectionTest());
            Clock clock = this.createClock();
            ChannelConnector connector = super.createConnector(connectionSettings, securityPlan, config, clock, routingContext);
            this.connectionPool = new MemorizingConnectionPool(connector, bootstrap, poolSettings, config.logging(), clock, ownsEventLoopGroup);
            return this.connectionPool;
        }
    }
}

