/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.integration;

import io.netty.channel.Channel;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URI;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Stream;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.hamcrest.junit.MatcherAssert;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.neo4j.driver.AuthToken;
import org.neo4j.driver.Config;
import org.neo4j.driver.Driver;
import org.neo4j.driver.GraphDatabase;
import org.neo4j.driver.Result;
import org.neo4j.driver.Session;
import org.neo4j.driver.Transaction;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.exceptions.ServiceUnavailableException;
import org.neo4j.driver.internal.cluster.RoutingSettings;
import org.neo4j.driver.internal.logging.DevNullLogging;
import org.neo4j.driver.internal.messaging.response.FailureMessage;
import org.neo4j.driver.internal.retry.RetrySettings;
import org.neo4j.driver.internal.security.SecurityPlanImpl;
import org.neo4j.driver.internal.util.FailingMessageFormat;
import org.neo4j.driver.internal.util.FakeClock;
import org.neo4j.driver.internal.util.Iterables;
import org.neo4j.driver.internal.util.io.ChannelTrackingDriverFactory;
import org.neo4j.driver.internal.util.io.ChannelTrackingDriverFactoryWithFailingMessageFormat;
import org.neo4j.driver.util.ParallelizableIT;
import org.neo4j.driver.util.SessionExtension;

@ParallelizableIT
class ErrorIT {
    @RegisterExtension
    static final SessionExtension session = new SessionExtension();

    ErrorIT() {
    }

    @Test
    void shouldThrowHelpfulSyntaxError() {
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> {
            Result result = session.run("invalid query");
            result.consume();
        });
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.startsWith((String)"Invalid input"));
    }

    @Test
    void shouldNotAllowMoreTxAfterClientException() {
        Transaction tx = session.beginTransaction();
        try {
            tx.run("invalid").consume();
        }
        catch (ClientException clientException) {
            // empty catch block
        }
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> {
            Result cursor = tx.run("RETURN 1");
            cursor.single().get("1").asInt();
        });
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.startsWith((String)"Cannot run more queries in this transaction"));
    }

    @Test
    void shouldAllowNewQueryAfterRecoverableError() {
        try {
            session.run("invalid").consume();
        }
        catch (ClientException clientException) {
            // empty catch block
        }
        Result cursor = session.run("RETURN 1");
        int val = cursor.single().get("1").asInt();
        MatcherAssert.assertThat((Object)val, (Matcher)CoreMatchers.equalTo((Object)1));
    }

    @Test
    void shouldAllowNewTransactionAfterRecoverableError() {
        Throwable throwable;
        Transaction tx2;
        try {
            tx2 = session.beginTransaction();
            throwable = null;
            try {
                tx2.run("invalid").consume();
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (tx2 != null) {
                    if (throwable != null) {
                        try {
                            tx2.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                    } else {
                        tx2.close();
                    }
                }
            }
        }
        catch (ClientException tx2) {
            // empty catch block
        }
        tx2 = session.beginTransaction();
        throwable = null;
        try {
            Result cursor = tx2.run("RETURN 1");
            int val = cursor.single().get("1").asInt();
            MatcherAssert.assertThat((Object)val, (Matcher)CoreMatchers.equalTo((Object)1));
        }
        catch (Throwable throwable4) {
            throwable = throwable4;
            throw throwable4;
        }
        finally {
            if (tx2 != null) {
                if (throwable != null) {
                    try {
                        tx2.close();
                    }
                    catch (Throwable throwable5) {
                        throwable.addSuppressed(throwable5);
                    }
                } else {
                    tx2.close();
                }
            }
        }
    }

    @Test
    void shouldExplainConnectionError() {
        Driver driver = GraphDatabase.driver((String)"bolt://localhost:7777");
        ServiceUnavailableException e = (ServiceUnavailableException)Assertions.assertThrows(ServiceUnavailableException.class, () -> ((Driver)driver).verifyConnectivity());
        Assertions.assertEquals((Object)"Unable to connect to localhost:7777, ensure the database is running and that there is a working network connection to it.", (Object)e.getMessage());
    }

    @Test
    void shouldHandleFailureAtRunTime() {
        if (session.isNeo4j44OrEarlier()) {
            String label = UUID.randomUUID().toString();
            Transaction tx = session.beginTransaction();
            tx.run("CREATE CONSTRAINT ON (a:`" + label + "`) ASSERT a.name IS UNIQUE");
            tx.commit();
            Transaction anotherTx = session.beginTransaction();
            ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> anotherTx.run("CREATE INDEX ON :`" + label + "`(name)"));
            anotherTx.rollback();
            MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)label));
            MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"name"));
        }
    }

    @Test
    void shouldGetHelpfulErrorWhenTryingToConnectToHttpPort() {
        Config config = Config.builder().withoutEncryption().build();
        URI boltUri = session.uri();
        URI uri = URI.create(String.format("%s://%s:%d", boltUri.getScheme(), boltUri.getHost(), session.httpPort()));
        Driver driver = GraphDatabase.driver((URI)uri, (Config)config);
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> ((Driver)driver).verifyConnectivity());
        Assertions.assertEquals((Object)"Server responded HTTP. Make sure you are not trying to connect to the http endpoint (HTTP defaults to port 7474 whereas BOLT defaults to port 7687)", (Object)e.getMessage());
    }

    @Test
    void shouldCloseChannelOnRuntimeExceptionInOutboundMessage() throws InterruptedException {
        RuntimeException error = new RuntimeException("Unable to encode message");
        Throwable queryError = this.testChannelErrorHandling(messageFormat -> messageFormat.makeWriterThrow(error));
        Assertions.assertEquals((Object)error, (Object)queryError);
    }

    @Test
    void shouldCloseChannelOnIOExceptionInOutboundMessage() throws InterruptedException {
        IOException error = new IOException("Unable to write");
        Throwable queryError = this.testChannelErrorHandling(messageFormat -> messageFormat.makeWriterThrow(error));
        MatcherAssert.assertThat((Object)queryError, (Matcher)Matchers.instanceOf(ServiceUnavailableException.class));
        Assertions.assertEquals((Object)"Connection to the database failed", (Object)queryError.getMessage());
        Assertions.assertEquals((Object)error, (Object)queryError.getCause());
    }

    @Test
    void shouldCloseChannelOnRuntimeExceptionInInboundMessage() throws InterruptedException {
        RuntimeException error = new RuntimeException("Unable to decode message");
        Throwable queryError = this.testChannelErrorHandling(messageFormat -> messageFormat.makeReaderThrow(error));
        Assertions.assertEquals((Object)error, (Object)queryError);
    }

    @Test
    void shouldCloseChannelOnIOExceptionInInboundMessage() throws InterruptedException {
        IOException error = new IOException("Unable to read");
        Throwable queryError = this.testChannelErrorHandling(messageFormat -> messageFormat.makeReaderThrow(error));
        MatcherAssert.assertThat((Object)queryError, (Matcher)Matchers.instanceOf(ServiceUnavailableException.class));
        Assertions.assertEquals((Object)"Connection to the database failed", (Object)queryError.getMessage());
        Assertions.assertEquals((Object)error, (Object)queryError.getCause());
    }

    @Test
    void shouldCloseChannelOnInboundFatalFailureMessage() throws InterruptedException {
        String errorCode = "Neo.ClientError.Request.Invalid";
        String errorMessage = "Very wrong request";
        FailureMessage failureMsg = new FailureMessage(errorCode, errorMessage);
        Throwable queryError = this.testChannelErrorHandling(messageFormat -> messageFormat.makeReaderFail(failureMsg));
        MatcherAssert.assertThat((Object)queryError, (Matcher)Matchers.instanceOf(ClientException.class));
        Assertions.assertEquals((Object)((ClientException)queryError).code(), (Object)errorCode);
        Assertions.assertEquals((Object)queryError.getMessage(), (Object)errorMessage);
    }

    @Test
    void shouldThrowErrorWithNiceStackTrace(TestInfo testInfo) {
        ClientException error = (ClientException)Assertions.assertThrows(ClientException.class, () -> session.run("RETURN 10 / 0").consume());
        StackTraceElement[] stackTrace = error.getStackTrace();
        Assertions.assertTrue((boolean)Stream.of(stackTrace).anyMatch(element -> ErrorIT.testClassAndMethodMatch(testInfo, element)), () -> "Expected stacktrace element is absent:\n" + Arrays.toString(stackTrace));
        MatcherAssert.assertThat(Arrays.asList(error.getSuppressed()), (Matcher)Matchers.hasSize((Matcher)Matchers.greaterThanOrEqualTo((Comparable)Integer.valueOf(1))));
    }

    private Throwable testChannelErrorHandling(Consumer<FailingMessageFormat> messageFormatSetup) throws InterruptedException {
        ChannelTrackingDriverFactoryWithFailingMessageFormat driverFactory = new ChannelTrackingDriverFactoryWithFailingMessageFormat(new FakeClock());
        URI uri = session.uri();
        AuthToken authToken = session.authToken();
        RoutingSettings routingSettings = RoutingSettings.DEFAULT;
        RetrySettings retrySettings = RetrySettings.DEFAULT;
        Config config = Config.builder().withLogging(DevNullLogging.DEV_NULL_LOGGING).build();
        Throwable queryError = null;
        try (Driver driver = driverFactory.newInstance(uri, authToken, routingSettings, retrySettings, config, SecurityPlanImpl.insecure());){
            driver.verifyConnectivity();
            try (Session session = driver.session();){
                messageFormatSetup.accept(driverFactory.getFailingMessageFormat());
                try {
                    session.run("RETURN 1").consume();
                    Assertions.fail((String)"Exception expected");
                }
                catch (Throwable error) {
                    queryError = error;
                }
                this.assertSingleChannelIsClosed(driverFactory);
                this.assertNewQueryCanBeExecuted(session, driverFactory);
            }
        }
        return queryError;
    }

    private void assertSingleChannelIsClosed(ChannelTrackingDriverFactory driverFactory) throws InterruptedException {
        Channel channel = (Channel)Iterables.single(driverFactory.channels());
        Assertions.assertTrue((boolean)channel.closeFuture().await(10L, TimeUnit.SECONDS));
        Assertions.assertFalse((boolean)channel.isActive());
    }

    private void assertNewQueryCanBeExecuted(Session session, ChannelTrackingDriverFactory driverFactory) {
        Assertions.assertEquals((int)42, (int)session.run("RETURN 42").single().get(0).asInt());
        List<Channel> channels = driverFactory.channels();
        Channel lastChannel = channels.get(channels.size() - 1);
        Assertions.assertTrue((boolean)lastChannel.isActive());
    }

    private static boolean testClassAndMethodMatch(TestInfo testInfo, StackTraceElement element) {
        return ErrorIT.testClassMatches(testInfo, element) && ErrorIT.testMethodMatches(testInfo, element);
    }

    private static boolean testClassMatches(TestInfo testInfo, StackTraceElement element) {
        String expectedName = testInfo.getTestClass().map(Class::getName).orElse("");
        String actualName = element.getClassName();
        return Objects.equals(expectedName, actualName);
    }

    private static boolean testMethodMatches(TestInfo testInfo, StackTraceElement element) {
        String expectedName = testInfo.getTestMethod().map(Method::getName).orElse("");
        String actualName = element.getMethodName();
        return Objects.equals(expectedName, actualName);
    }
}

