/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.async;

import java.util.concurrent.CompletableFuture;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.junit.MatcherAssert;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.ArgumentMatchers;
import org.mockito.InOrder;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.neo4j.driver.AccessMode;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Query;
import org.neo4j.driver.TransactionConfig;
import org.neo4j.driver.async.ResultCursor;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.internal.InternalBookmark;
import org.neo4j.driver.internal.async.ConnectionContext;
import org.neo4j.driver.internal.async.NetworkSession;
import org.neo4j.driver.internal.async.UnmanagedTransaction;
import org.neo4j.driver.internal.messaging.BoltProtocol;
import org.neo4j.driver.internal.messaging.Message;
import org.neo4j.driver.internal.messaging.request.PullMessage;
import org.neo4j.driver.internal.messaging.request.RunWithMetadataMessage;
import org.neo4j.driver.internal.messaging.v4.BoltProtocolV4;
import org.neo4j.driver.internal.spi.Connection;
import org.neo4j.driver.internal.spi.ConnectionProvider;
import org.neo4j.driver.internal.spi.ResponseHandler;
import org.neo4j.driver.internal.util.Futures;
import org.neo4j.driver.util.TestUtil;

class NetworkSessionTest {
    private Connection connection;
    private ConnectionProvider connectionProvider;
    private NetworkSession session;

    NetworkSessionTest() {
    }

    @BeforeEach
    void setUp() {
        this.connection = TestUtil.connectionMock(BoltProtocolV4.INSTANCE);
        this.connectionProvider = (ConnectionProvider)Mockito.mock(ConnectionProvider.class);
        Mockito.when((Object)this.connectionProvider.acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class))).thenReturn(CompletableFuture.completedFuture(this.connection));
        this.session = TestUtil.newSession(this.connectionProvider);
    }

    @Test
    void shouldFlushOnRunAsync() {
        TestUtil.setupSuccessfulRunAndPull(this.connection);
        TestUtil.await(this.session.runAsync(new Query("RETURN 1"), TransactionConfig.empty()));
        TestUtil.verifyRunAndPull(this.connection, "RETURN 1");
    }

    @Test
    void shouldFlushOnRunRx() {
        TestUtil.setupSuccessfulRunRx(this.connection);
        TestUtil.await(this.session.runRx(new Query("RETURN 1"), TransactionConfig.empty()));
        TestUtil.verifyRunRx(this.connection, "RETURN 1");
    }

    @Test
    void shouldNotAllowNewTxWhileOneIsRunning() {
        NetworkSessionTest.beginTransaction(this.session);
        Assertions.assertThrows(ClientException.class, () -> NetworkSessionTest.beginTransaction(this.session));
    }

    @Test
    void shouldBeAbleToOpenTxAfterPreviousIsClosed() {
        TestUtil.await(NetworkSessionTest.beginTransaction(this.session).closeAsync());
        UnmanagedTransaction tx = NetworkSessionTest.beginTransaction(this.session);
        Assertions.assertNotNull((Object)tx);
        TestUtil.verifyRollbackTx(this.connection);
    }

    @Test
    void shouldNotBeAbleToUseSessionWhileOngoingTransaction() {
        NetworkSessionTest.beginTransaction(this.session);
        Assertions.assertThrows(ClientException.class, () -> NetworkSessionTest.run(this.session, "RETURN 1"));
    }

    @Test
    void shouldBeAbleToUseSessionAgainWhenTransactionIsClosed() {
        TestUtil.await(NetworkSessionTest.beginTransaction(this.session).closeAsync());
        String query = "RETURN 1";
        TestUtil.setupSuccessfulRunAndPull(this.connection, query);
        NetworkSessionTest.run(this.session, query);
        TestUtil.verifyRunAndPull(this.connection, query);
    }

    @Test
    void shouldNotCloseAlreadyClosedSession() {
        NetworkSessionTest.beginTransaction(this.session);
        NetworkSessionTest.close(this.session);
        NetworkSessionTest.close(this.session);
        NetworkSessionTest.close(this.session);
        TestUtil.verifyRollbackTx(this.connection);
    }

    @Test
    void runThrowsWhenSessionIsClosed() {
        NetworkSessionTest.close(this.session);
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> NetworkSessionTest.run(this.session, "CREATE ()"));
        MatcherAssert.assertThat((Object)e, (Matcher)CoreMatchers.instanceOf(ClientException.class));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"session is already closed"));
    }

    @Test
    void acquiresNewConnectionForRun() {
        String query = "RETURN 1";
        TestUtil.setupSuccessfulRunAndPull(this.connection, query);
        NetworkSessionTest.run(this.session, query);
        ((ConnectionProvider)Mockito.verify((Object)this.connectionProvider)).acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class));
    }

    @Test
    void releasesOpenConnectionUsedForRunWhenSessionIsClosed() {
        String query = "RETURN 1";
        TestUtil.setupSuccessfulRunAndPull(this.connection, query);
        NetworkSessionTest.run(this.session, query);
        NetworkSessionTest.close(this.session);
        InOrder inOrder = Mockito.inOrder((Object[])new Object[]{this.connection});
        ((Connection)inOrder.verify((Object)this.connection)).write((Message)ArgumentMatchers.any(RunWithMetadataMessage.class), (ResponseHandler)ArgumentMatchers.any());
        ((Connection)inOrder.verify((Object)this.connection)).writeAndFlush((Message)ArgumentMatchers.any(PullMessage.class), (ResponseHandler)ArgumentMatchers.any());
        ((Connection)inOrder.verify((Object)this.connection, Mockito.atLeastOnce())).release();
    }

    @Test
    void resetDoesNothingWhenNoTransactionAndNoConnection() {
        TestUtil.await(this.session.resetAsync());
        ((ConnectionProvider)Mockito.verify((Object)this.connectionProvider, (VerificationMode)Mockito.never())).acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class));
    }

    @Test
    void closeWithoutConnection() {
        NetworkSession session = TestUtil.newSession(this.connectionProvider);
        NetworkSessionTest.close(session);
        ((ConnectionProvider)Mockito.verify((Object)this.connectionProvider, (VerificationMode)Mockito.never())).acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class));
    }

    @Test
    void acquiresNewConnectionForBeginTx() {
        UnmanagedTransaction tx = NetworkSessionTest.beginTransaction(this.session);
        Assertions.assertNotNull((Object)tx);
        ((ConnectionProvider)Mockito.verify((Object)this.connectionProvider)).acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class));
    }

    @Test
    void updatesBookmarkWhenTxIsClosed() {
        Bookmark bookmarkAfterCommit = InternalBookmark.parse((String)"TheBookmark");
        BoltProtocol protocol = (BoltProtocol)Mockito.spy((Object)BoltProtocolV4.INSTANCE);
        ((BoltProtocol)Mockito.doReturn(CompletableFuture.completedFuture(bookmarkAfterCommit)).when((Object)protocol)).commitTransaction((Connection)ArgumentMatchers.any(Connection.class));
        Mockito.when((Object)this.connection.protocol()).thenReturn((Object)protocol);
        UnmanagedTransaction tx = NetworkSessionTest.beginTransaction(this.session);
        MatcherAssert.assertThat((Object)this.session.lastBookmark(), (Matcher)CoreMatchers.instanceOf(InternalBookmark.class));
        InternalBookmark bookmark = (InternalBookmark)this.session.lastBookmark();
        Assertions.assertTrue((boolean)bookmark.isEmpty());
        TestUtil.await(tx.commitAsync());
        Assertions.assertEquals((Object)bookmarkAfterCommit, (Object)this.session.lastBookmark());
    }

    @Test
    void releasesConnectionWhenTxIsClosed() {
        String query = "RETURN 42";
        TestUtil.setupSuccessfulRunAndPull(this.connection, query);
        UnmanagedTransaction tx = NetworkSessionTest.beginTransaction(this.session);
        TestUtil.await(tx.runAsync(new Query(query)));
        ((ConnectionProvider)Mockito.verify((Object)this.connectionProvider)).acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class));
        TestUtil.verifyRunAndPull(this.connection, query);
        TestUtil.await(tx.closeAsync());
        ((Connection)Mockito.verify((Object)this.connection)).release();
    }

    @Test
    void bookmarkIsPropagatedFromSession() {
        Bookmark bookmark = InternalBookmark.parse((String)"Bookmarks");
        NetworkSession session = TestUtil.newSession(this.connectionProvider, bookmark);
        UnmanagedTransaction tx = NetworkSessionTest.beginTransaction(session);
        Assertions.assertNotNull((Object)tx);
        TestUtil.verifyBeginTx(this.connection);
    }

    @Test
    void bookmarkIsPropagatedBetweenTransactions() {
        Bookmark bookmark1 = InternalBookmark.parse((String)"Bookmark1");
        Bookmark bookmark2 = InternalBookmark.parse((String)"Bookmark2");
        NetworkSession session = TestUtil.newSession(this.connectionProvider);
        BoltProtocol protocol = (BoltProtocol)Mockito.spy((Object)BoltProtocolV4.INSTANCE);
        ((BoltProtocol)Mockito.doReturn(CompletableFuture.completedFuture(bookmark1), (Object[])new Object[]{CompletableFuture.completedFuture(bookmark2)}).when((Object)protocol)).commitTransaction((Connection)ArgumentMatchers.any(Connection.class));
        Mockito.when((Object)this.connection.protocol()).thenReturn((Object)protocol);
        UnmanagedTransaction tx1 = NetworkSessionTest.beginTransaction(session);
        TestUtil.await(tx1.commitAsync());
        Assertions.assertEquals((Object)bookmark1, (Object)session.lastBookmark());
        UnmanagedTransaction tx2 = NetworkSessionTest.beginTransaction(session);
        TestUtil.verifyBeginTx(this.connection, 2);
        TestUtil.await(tx2.commitAsync());
        Assertions.assertEquals((Object)bookmark2, (Object)session.lastBookmark());
    }

    @Test
    void accessModeUsedToAcquireReadConnections() {
        this.accessModeUsedToAcquireConnections(AccessMode.READ);
    }

    @Test
    void accessModeUsedToAcquireWriteConnections() {
        this.accessModeUsedToAcquireConnections(AccessMode.WRITE);
    }

    private void accessModeUsedToAcquireConnections(AccessMode mode) {
        NetworkSession session2 = TestUtil.newSession(this.connectionProvider, mode);
        NetworkSessionTest.beginTransaction(session2);
        ArgumentCaptor argument = ArgumentCaptor.forClass(ConnectionContext.class);
        ((ConnectionProvider)Mockito.verify((Object)this.connectionProvider)).acquireConnection((ConnectionContext)argument.capture());
        Assertions.assertEquals((Object)mode, (Object)((ConnectionContext)argument.getValue()).mode());
    }

    @Test
    void testPassingNoBookmarkShouldRetainBookmark() {
        NetworkSession session = TestUtil.newSession(this.connectionProvider, InternalBookmark.parse((String)"X"));
        NetworkSessionTest.beginTransaction(session);
        MatcherAssert.assertThat((Object)session.lastBookmark(), (Matcher)CoreMatchers.equalTo((Object)InternalBookmark.parse((String)"X")));
    }

    @Test
    void connectionShouldBeResetAfterSessionReset() {
        String query = "RETURN 1";
        TestUtil.setupSuccessfulRunAndPull(this.connection, query);
        NetworkSessionTest.run(this.session, query);
        InOrder connectionInOrder = Mockito.inOrder((Object[])new Object[]{this.connection});
        ((Connection)connectionInOrder.verify((Object)this.connection, Mockito.never())).reset();
        ((Connection)connectionInOrder.verify((Object)this.connection)).release();
        TestUtil.await(this.session.resetAsync());
        ((Connection)connectionInOrder.verify((Object)this.connection)).reset();
        ((Connection)connectionInOrder.verify((Object)this.connection, Mockito.never())).release();
    }

    @Test
    void shouldHaveEmptyLastBookmarkInitially() {
        MatcherAssert.assertThat((Object)this.session.lastBookmark(), (Matcher)CoreMatchers.instanceOf(InternalBookmark.class));
        InternalBookmark bookmark = (InternalBookmark)this.session.lastBookmark();
        Assertions.assertTrue((boolean)bookmark.isEmpty());
    }

    @Test
    void shouldDoNothingWhenClosingWithoutAcquiredConnection() {
        RuntimeException error = new RuntimeException("Hi");
        Mockito.when((Object)this.connectionProvider.acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class))).thenReturn((Object)Futures.failedFuture((Throwable)error));
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> NetworkSessionTest.run(this.session, "RETURN 1"));
        Assertions.assertEquals((Object)error, (Object)e);
        NetworkSessionTest.close(this.session);
    }

    @Test
    void shouldRunAfterRunFailure() {
        RuntimeException error = new RuntimeException("Hi");
        Mockito.when((Object)this.connectionProvider.acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class))).thenReturn((Object)Futures.failedFuture((Throwable)error)).thenReturn(CompletableFuture.completedFuture(this.connection));
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> NetworkSessionTest.run(this.session, "RETURN 1"));
        Assertions.assertEquals((Object)error, (Object)e);
        String query = "RETURN 2";
        TestUtil.setupSuccessfulRunAndPull(this.connection, query);
        NetworkSessionTest.run(this.session, query);
        ((ConnectionProvider)Mockito.verify((Object)this.connectionProvider, (VerificationMode)Mockito.times((int)2))).acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class));
        TestUtil.verifyRunAndPull(this.connection, query);
    }

    @Test
    void shouldRunAfterBeginTxFailureOnBookmark() {
        RuntimeException error = new RuntimeException("Hi");
        Connection connection1 = TestUtil.connectionMock(BoltProtocolV4.INSTANCE);
        TestUtil.setupFailingBegin(connection1, error);
        Connection connection2 = TestUtil.connectionMock(BoltProtocolV4.INSTANCE);
        Mockito.when((Object)this.connectionProvider.acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class))).thenReturn(CompletableFuture.completedFuture(connection1)).thenReturn(CompletableFuture.completedFuture(connection2));
        Bookmark bookmark = InternalBookmark.parse((String)"neo4j:bookmark:v1:tx42");
        NetworkSession session = TestUtil.newSession(this.connectionProvider, bookmark);
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> NetworkSessionTest.beginTransaction(session));
        Assertions.assertEquals((Object)error, (Object)e);
        String query = "RETURN 2";
        TestUtil.setupSuccessfulRunAndPull(connection2, query);
        NetworkSessionTest.run(session, query);
        ((ConnectionProvider)Mockito.verify((Object)this.connectionProvider, (VerificationMode)Mockito.times((int)2))).acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class));
        TestUtil.verifyBeginTx(connection1);
        TestUtil.verifyRunAndPull(connection2, "RETURN 2");
    }

    @Test
    void shouldBeginTxAfterBeginTxFailureOnBookmark() {
        RuntimeException error = new RuntimeException("Hi");
        Connection connection1 = TestUtil.connectionMock(BoltProtocolV4.INSTANCE);
        TestUtil.setupFailingBegin(connection1, error);
        Connection connection2 = TestUtil.connectionMock(BoltProtocolV4.INSTANCE);
        Mockito.when((Object)this.connectionProvider.acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class))).thenReturn(CompletableFuture.completedFuture(connection1)).thenReturn(CompletableFuture.completedFuture(connection2));
        Bookmark bookmark = InternalBookmark.parse((String)"neo4j:bookmark:v1:tx42");
        NetworkSession session = TestUtil.newSession(this.connectionProvider, bookmark);
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> NetworkSessionTest.beginTransaction(session));
        Assertions.assertEquals((Object)error, (Object)e);
        NetworkSessionTest.beginTransaction(session);
        ((ConnectionProvider)Mockito.verify((Object)this.connectionProvider, (VerificationMode)Mockito.times((int)2))).acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class));
        TestUtil.verifyBeginTx(connection1);
        TestUtil.verifyBeginTx(connection2);
    }

    @Test
    void shouldBeginTxAfterRunFailureToAcquireConnection() {
        RuntimeException error = new RuntimeException("Hi");
        Mockito.when((Object)this.connectionProvider.acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class))).thenReturn((Object)Futures.failedFuture((Throwable)error)).thenReturn(CompletableFuture.completedFuture(this.connection));
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> NetworkSessionTest.run(this.session, "RETURN 1"));
        Assertions.assertEquals((Object)error, (Object)e);
        NetworkSessionTest.beginTransaction(this.session);
        ((ConnectionProvider)Mockito.verify((Object)this.connectionProvider, (VerificationMode)Mockito.times((int)2))).acquireConnection((ConnectionContext)ArgumentMatchers.any(ConnectionContext.class));
        TestUtil.verifyBeginTx(this.connection);
    }

    @Test
    void shouldMarkTransactionAsTerminatedAndThenResetConnectionOnReset() {
        UnmanagedTransaction tx = NetworkSessionTest.beginTransaction(this.session);
        Assertions.assertTrue((boolean)tx.isOpen());
        ((Connection)Mockito.verify((Object)this.connection, (VerificationMode)Mockito.never())).reset();
        TestUtil.await(this.session.resetAsync());
        ((Connection)Mockito.verify((Object)this.connection)).reset();
    }

    private static ResultCursor run(NetworkSession session, String query) {
        return (ResultCursor)TestUtil.await(session.runAsync(new Query(query), TransactionConfig.empty()));
    }

    private static UnmanagedTransaction beginTransaction(NetworkSession session) {
        return (UnmanagedTransaction)TestUtil.await(session.beginTransactionAsync(TransactionConfig.empty()));
    }

    private static void close(NetworkSession session) {
        TestUtil.await(session.closeAsync());
    }
}

