/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.async;

import java.util.Collections;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Stream;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.ArgumentMatchers;
import org.mockito.BDDMockito;
import org.mockito.InOrder;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Query;
import org.neo4j.driver.TransactionConfig;
import org.neo4j.driver.exceptions.AuthorizationExpiredException;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.exceptions.ConnectionReadTimeoutException;
import org.neo4j.driver.internal.BookmarkHolder;
import org.neo4j.driver.internal.DefaultBookmarkHolder;
import org.neo4j.driver.internal.FailableCursor;
import org.neo4j.driver.internal.InternalBookmark;
import org.neo4j.driver.internal.async.ResultCursorsHolder;
import org.neo4j.driver.internal.async.UnmanagedTransaction;
import org.neo4j.driver.internal.messaging.BoltProtocol;
import org.neo4j.driver.internal.messaging.Message;
import org.neo4j.driver.internal.messaging.v4.BoltProtocolV4;
import org.neo4j.driver.internal.spi.Connection;
import org.neo4j.driver.internal.spi.ResponseHandler;
import org.neo4j.driver.util.TestUtil;

class UnmanagedTransactionTest {
    UnmanagedTransactionTest() {
    }

    @Test
    void shouldFlushOnRunAsync() {
        Connection connection = TestUtil.connectionMock(BoltProtocolV4.INSTANCE);
        UnmanagedTransaction tx = UnmanagedTransactionTest.beginTx(connection);
        TestUtil.setupSuccessfulRunAndPull(connection);
        TestUtil.await(tx.runAsync(new Query("RETURN 1")));
        TestUtil.verifyRunAndPull(connection, "RETURN 1");
    }

    @Test
    void shouldFlushOnRunRx() {
        Connection connection = TestUtil.connectionMock(BoltProtocolV4.INSTANCE);
        UnmanagedTransaction tx = UnmanagedTransactionTest.beginTx(connection);
        TestUtil.setupSuccessfulRunRx(connection);
        TestUtil.await(tx.runRx(new Query("RETURN 1")));
        TestUtil.verifyRunRx(connection, "RETURN 1");
    }

    @Test
    void shouldRollbackOnImplicitFailure() {
        Connection connection = TestUtil.connectionMock();
        UnmanagedTransaction tx = UnmanagedTransactionTest.beginTx(connection);
        TestUtil.await(tx.closeAsync());
        InOrder order = Mockito.inOrder((Object[])new Object[]{connection});
        TestUtil.verifyBeginTx(connection);
        TestUtil.verifyRollbackTx(connection);
        ((Connection)order.verify((Object)connection)).release();
    }

    @Test
    void shouldOnlyQueueMessagesWhenNoBookmarkGiven() {
        Connection connection = TestUtil.connectionMock();
        UnmanagedTransactionTest.beginTx(connection, InternalBookmark.empty());
        TestUtil.verifyBeginTx(connection);
        ((Connection)Mockito.verify((Object)connection, (VerificationMode)Mockito.never())).writeAndFlush((Message)ArgumentMatchers.any(), (ResponseHandler)ArgumentMatchers.any(), (Message)ArgumentMatchers.any(), (ResponseHandler)ArgumentMatchers.any());
    }

    @Test
    void shouldFlushWhenBookmarkGiven() {
        Bookmark bookmark = InternalBookmark.parse((String)"hi, I'm bookmark");
        Connection connection = TestUtil.connectionMock();
        UnmanagedTransactionTest.beginTx(connection, bookmark);
        TestUtil.verifyBeginTx(connection);
        ((Connection)Mockito.verify((Object)connection, (VerificationMode)Mockito.never())).write((Message)ArgumentMatchers.any(), (ResponseHandler)ArgumentMatchers.any(), (Message)ArgumentMatchers.any(), (ResponseHandler)ArgumentMatchers.any());
    }

    @Test
    void shouldBeOpenAfterConstruction() {
        UnmanagedTransaction tx = UnmanagedTransactionTest.beginTx(TestUtil.connectionMock());
        Assertions.assertTrue((boolean)tx.isOpen());
    }

    @Test
    void shouldBeClosedWhenMarkedAsTerminated() {
        UnmanagedTransaction tx = UnmanagedTransactionTest.beginTx(TestUtil.connectionMock());
        tx.markTerminated(null);
        Assertions.assertTrue((boolean)tx.isOpen());
    }

    @Test
    void shouldBeClosedWhenMarkedTerminatedAndClosed() {
        UnmanagedTransaction tx = UnmanagedTransactionTest.beginTx(TestUtil.connectionMock());
        tx.markTerminated(null);
        TestUtil.await(tx.closeAsync());
        Assertions.assertFalse((boolean)tx.isOpen());
    }

    @Test
    void shouldReleaseConnectionWhenBeginFails() {
        RuntimeException error = new RuntimeException("Wrong bookmark!");
        Connection connection = UnmanagedTransactionTest.connectionWithBegin(handler -> handler.onFailure((Throwable)error));
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        Bookmark bookmark = InternalBookmark.parse((String)"SomeBookmark");
        TransactionConfig txConfig = TransactionConfig.empty();
        RuntimeException e = (RuntimeException)Assertions.assertThrows(RuntimeException.class, () -> {
            UnmanagedTransaction cfr_ignored_0 = (UnmanagedTransaction)TestUtil.await(tx.beginAsync(bookmark, txConfig));
        });
        Assertions.assertEquals((Object)error, (Object)e);
        ((Connection)Mockito.verify((Object)connection)).release();
    }

    @Test
    void shouldNotReleaseConnectionWhenBeginSucceeds() {
        Connection connection = UnmanagedTransactionTest.connectionWithBegin(handler -> handler.onSuccess(Collections.emptyMap()));
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        Bookmark bookmark = InternalBookmark.parse((String)"SomeBookmark");
        TransactionConfig txConfig = TransactionConfig.empty();
        TestUtil.await(tx.beginAsync(bookmark, txConfig));
        ((Connection)Mockito.verify((Object)connection, (VerificationMode)Mockito.never())).release();
    }

    @Test
    void shouldReleaseConnectionWhenTerminatedAndCommitted() {
        Connection connection = TestUtil.connectionMock();
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        tx.markTerminated(null);
        Assertions.assertThrows(ClientException.class, () -> {
            Void cfr_ignored_0 = (Void)TestUtil.await(tx.commitAsync());
        });
        Assertions.assertFalse((boolean)tx.isOpen());
        ((Connection)Mockito.verify((Object)connection)).release();
    }

    @Test
    void shouldNotCreateCircularExceptionWhenTerminationCauseEqualsToCursorFailure() {
        Connection connection = TestUtil.connectionMock();
        ClientException terminationCause = new ClientException("Custom exception");
        ResultCursorsHolder resultCursorsHolder = this.mockResultCursorWith(terminationCause);
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L, resultCursorsHolder);
        tx.markTerminated((Throwable)terminationCause);
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> {
            Void cfr_ignored_0 = (Void)TestUtil.await(tx.commitAsync());
        });
        TestUtil.assertNoCircularReferences(e);
        Assertions.assertEquals((Object)((Object)terminationCause), (Object)((Object)e));
    }

    @Test
    void shouldNotCreateCircularExceptionWhenTerminationCauseDifferentFromCursorFailure() {
        Connection connection = TestUtil.connectionMock();
        ClientException terminationCause = new ClientException("Custom exception");
        ResultCursorsHolder resultCursorsHolder = this.mockResultCursorWith(new ClientException("Cursor error"));
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L, resultCursorsHolder);
        tx.markTerminated((Throwable)terminationCause);
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> {
            Void cfr_ignored_0 = (Void)TestUtil.await(tx.commitAsync());
        });
        TestUtil.assertNoCircularReferences(e);
        Assertions.assertEquals((int)1, (int)e.getSuppressed().length);
        Throwable suppressed = e.getSuppressed()[0];
        Assertions.assertEquals((Object)((Object)terminationCause), (Object)suppressed.getCause());
    }

    @Test
    void shouldNotCreateCircularExceptionWhenTerminatedWithoutFailure() {
        Connection connection = TestUtil.connectionMock();
        ClientException terminationCause = new ClientException("Custom exception");
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        tx.markTerminated((Throwable)terminationCause);
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> {
            Void cfr_ignored_0 = (Void)TestUtil.await(tx.commitAsync());
        });
        TestUtil.assertNoCircularReferences(e);
        Assertions.assertEquals((Object)((Object)terminationCause), (Object)e.getCause());
    }

    @Test
    void shouldReleaseConnectionWhenTerminatedAndRolledBack() {
        Connection connection = TestUtil.connectionMock();
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        tx.markTerminated(null);
        TestUtil.await(tx.rollbackAsync());
        ((Connection)Mockito.verify((Object)connection)).release();
    }

    @Test
    void shouldReleaseConnectionWhenClose() throws Throwable {
        Connection connection = TestUtil.connectionMock();
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        TestUtil.await(tx.closeAsync());
        ((Connection)Mockito.verify((Object)connection)).release();
    }

    @Test
    void shouldReleaseConnectionOnConnectionAuthorizationExpiredExceptionFailure() {
        AuthorizationExpiredException exception = new AuthorizationExpiredException("code", "message");
        Connection connection = UnmanagedTransactionTest.connectionWithBegin(handler -> handler.onFailure((Throwable)exception));
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        Bookmark bookmark = InternalBookmark.parse((String)"SomeBookmark");
        TransactionConfig txConfig = TransactionConfig.empty();
        AuthorizationExpiredException actualException = (AuthorizationExpiredException)Assertions.assertThrows(AuthorizationExpiredException.class, () -> {
            UnmanagedTransaction cfr_ignored_0 = (UnmanagedTransaction)TestUtil.await(tx.beginAsync(bookmark, txConfig));
        });
        Assertions.assertSame((Object)exception, (Object)actualException);
        ((Connection)Mockito.verify((Object)connection)).terminateAndRelease("Authorization information kept on the server has expired, this connection is no longer valid.");
        ((Connection)Mockito.verify((Object)connection, (VerificationMode)Mockito.never())).release();
    }

    @Test
    void shouldReleaseConnectionOnConnectionReadTimeoutExceptionFailure() {
        Connection connection = UnmanagedTransactionTest.connectionWithBegin(handler -> handler.onFailure((Throwable)ConnectionReadTimeoutException.INSTANCE));
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        Bookmark bookmark = InternalBookmark.parse((String)"SomeBookmark");
        TransactionConfig txConfig = TransactionConfig.empty();
        ConnectionReadTimeoutException actualException = (ConnectionReadTimeoutException)Assertions.assertThrows(ConnectionReadTimeoutException.class, () -> {
            UnmanagedTransaction cfr_ignored_0 = (UnmanagedTransaction)TestUtil.await(tx.beginAsync(bookmark, txConfig));
        });
        Assertions.assertSame((Object)ConnectionReadTimeoutException.INSTANCE, (Object)actualException);
        ((Connection)Mockito.verify((Object)connection)).terminateAndRelease(ConnectionReadTimeoutException.INSTANCE.getMessage());
        ((Connection)Mockito.verify((Object)connection, (VerificationMode)Mockito.never())).release();
    }

    private static Stream<Arguments> similarTransactionCompletingActionArgs() {
        return Stream.of(Arguments.of((Object[])new Object[]{true, "commit", "commit"}), Arguments.of((Object[])new Object[]{false, "rollback", "rollback"}), Arguments.of((Object[])new Object[]{false, "rollback", "close"}), Arguments.of((Object[])new Object[]{false, "close", "rollback"}), Arguments.of((Object[])new Object[]{false, "close", "close"}));
    }

    @ParameterizedTest
    @MethodSource(value={"similarTransactionCompletingActionArgs"})
    void shouldReturnExistingStageOnSimilarCompletingAction(boolean protocolCommit, String initialAction, String similarAction) {
        Connection connection = (Connection)Mockito.mock(Connection.class);
        BoltProtocol protocol = (BoltProtocol)Mockito.mock(BoltProtocol.class);
        BDDMockito.given((Object)connection.protocol()).willReturn((Object)protocol);
        BDDMockito.given((Object)(protocolCommit ? protocol.commitTransaction(connection) : protocol.rollbackTransaction(connection))).willReturn(new CompletableFuture());
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        CompletionStage<Void> initialStage = this.mapTransactionAction(initialAction, tx).get();
        CompletionStage<Void> similarStage = this.mapTransactionAction(similarAction, tx).get();
        Assertions.assertSame(initialStage, similarStage);
        if (protocolCommit) {
            ((BoltProtocol)BDDMockito.then((Object)protocol).should(Mockito.times((int)1))).commitTransaction(connection);
        } else {
            ((BoltProtocol)BDDMockito.then((Object)protocol).should(Mockito.times((int)1))).rollbackTransaction(connection);
        }
    }

    private static Stream<Arguments> conflictingTransactionCompletingActionArgs() {
        return Stream.of(Arguments.of((Object[])new Object[]{true, true, "commit", "commit", "Can't commit, transaction has been committed"}), Arguments.of((Object[])new Object[]{true, true, "commit", "rollback", "Can't rollback, transaction has been committed"}), Arguments.of((Object[])new Object[]{true, false, "commit", "rollback", "Can't rollback, transaction has been requested to be committed"}), Arguments.of((Object[])new Object[]{true, false, "commit", "close", "Can't rollback, transaction has been requested to be committed"}), Arguments.of((Object[])new Object[]{false, true, "rollback", "rollback", "Can't rollback, transaction has been rolled back"}), Arguments.of((Object[])new Object[]{false, true, "rollback", "commit", "Can't commit, transaction has been rolled back"}), Arguments.of((Object[])new Object[]{false, false, "rollback", "commit", "Can't commit, transaction has been requested to be rolled back"}), Arguments.of((Object[])new Object[]{false, true, "close", "commit", "Can't commit, transaction has been rolled back"}), Arguments.of((Object[])new Object[]{false, true, "close", "rollback", "Can't rollback, transaction has been rolled back"}), Arguments.of((Object[])new Object[]{false, false, "close", "commit", "Can't commit, transaction has been requested to be rolled back"}));
    }

    @ParameterizedTest
    @MethodSource(value={"conflictingTransactionCompletingActionArgs"})
    void shouldReturnFailingStageOnConflictingCompletingAction(boolean protocolCommit, boolean protocolActionCompleted, String initialAction, String conflictingAction, String expectedErrorMsg) {
        Connection connection = (Connection)Mockito.mock(Connection.class);
        BoltProtocol protocol = (BoltProtocol)Mockito.mock(BoltProtocol.class);
        BDDMockito.given((Object)connection.protocol()).willReturn((Object)protocol);
        BDDMockito.given((Object)(protocolCommit ? protocol.commitTransaction(connection) : protocol.rollbackTransaction(connection))).willReturn(protocolActionCompleted ? CompletableFuture.completedFuture(null) : new CompletableFuture());
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        CompletionStage<Void> originalActionStage = this.mapTransactionAction(initialAction, tx).get();
        CompletionStage<Void> conflictingActionStage = this.mapTransactionAction(conflictingAction, tx).get();
        Assertions.assertNotNull(originalActionStage);
        if (protocolCommit) {
            ((BoltProtocol)BDDMockito.then((Object)protocol).should(Mockito.times((int)1))).commitTransaction(connection);
        } else {
            ((BoltProtocol)BDDMockito.then((Object)protocol).should(Mockito.times((int)1))).rollbackTransaction(connection);
        }
        Assertions.assertTrue((boolean)conflictingActionStage.toCompletableFuture().isCompletedExceptionally());
        Throwable throwable = ((ExecutionException)Assertions.assertThrows(ExecutionException.class, () -> {
            Void cfr_ignored_0 = (Void)conflictingActionStage.toCompletableFuture().get();
        })).getCause();
        Assertions.assertTrue((boolean)(throwable instanceof ClientException));
        Assertions.assertEquals((Object)expectedErrorMsg, (Object)throwable.getMessage());
    }

    private static Stream<Arguments> closingNotActionTransactionArgs() {
        return Stream.of(Arguments.of((Object[])new Object[]{true, 1, "commit", null}), Arguments.of((Object[])new Object[]{false, 1, "rollback", null}), Arguments.of((Object[])new Object[]{false, 0, "terminate", null}), Arguments.of((Object[])new Object[]{true, 1, "commit", true}), Arguments.of((Object[])new Object[]{false, 1, "rollback", true}), Arguments.of((Object[])new Object[]{true, 1, "commit", false}), Arguments.of((Object[])new Object[]{false, 1, "rollback", false}), Arguments.of((Object[])new Object[]{false, 0, "terminate", false}));
    }

    @ParameterizedTest
    @MethodSource(value={"closingNotActionTransactionArgs"})
    void shouldReturnCompletedWithNullStageOnClosingInactiveTransactionExceptCommittingAborted(boolean protocolCommit, int expectedProtocolInvocations, String originalAction, Boolean commitOnClose) {
        Connection connection = (Connection)Mockito.mock(Connection.class);
        BoltProtocol protocol = (BoltProtocol)Mockito.mock(BoltProtocol.class);
        BDDMockito.given((Object)connection.protocol()).willReturn((Object)protocol);
        BDDMockito.given((Object)(protocolCommit ? protocol.commitTransaction(connection) : protocol.rollbackTransaction(connection))).willReturn(CompletableFuture.completedFuture(null));
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        CompletionStage<Void> originalActionStage = this.mapTransactionAction(originalAction, tx).get();
        CompletionStage closeStage = commitOnClose != null ? tx.closeAsync(commitOnClose.booleanValue()) : tx.closeAsync();
        Assertions.assertTrue((boolean)originalActionStage.toCompletableFuture().isDone());
        Assertions.assertFalse((boolean)originalActionStage.toCompletableFuture().isCompletedExceptionally());
        if (protocolCommit) {
            ((BoltProtocol)BDDMockito.then((Object)protocol).should(Mockito.times((int)expectedProtocolInvocations))).commitTransaction(connection);
        } else {
            ((BoltProtocol)BDDMockito.then((Object)protocol).should(Mockito.times((int)expectedProtocolInvocations))).rollbackTransaction(connection);
        }
        Assertions.assertNull(closeStage.toCompletableFuture().join());
    }

    private static UnmanagedTransaction beginTx(Connection connection) {
        return UnmanagedTransactionTest.beginTx(connection, InternalBookmark.empty());
    }

    private static UnmanagedTransaction beginTx(Connection connection, Bookmark initialBookmark) {
        UnmanagedTransaction tx = new UnmanagedTransaction(connection, (BookmarkHolder)new DefaultBookmarkHolder(), -1L);
        return (UnmanagedTransaction)TestUtil.await(tx.beginAsync(initialBookmark, TransactionConfig.empty()));
    }

    private static Connection connectionWithBegin(Consumer<ResponseHandler> beginBehaviour) {
        Connection connection = TestUtil.connectionMock();
        ((Connection)Mockito.doAnswer(invocation -> {
            ResponseHandler beginHandler = (ResponseHandler)invocation.getArgument(1);
            beginBehaviour.accept(beginHandler);
            return null;
        }).when((Object)connection)).writeAndFlush((Message)ArgumentMatchers.argThat(TestUtil.beginMessage()), (ResponseHandler)ArgumentMatchers.any());
        return connection;
    }

    private ResultCursorsHolder mockResultCursorWith(ClientException clientException) {
        ResultCursorsHolder resultCursorsHolder = new ResultCursorsHolder();
        FailableCursor cursor = (FailableCursor)Mockito.mock(FailableCursor.class);
        ((FailableCursor)Mockito.doReturn(CompletableFuture.completedFuture(clientException)).when((Object)cursor)).discardAllFailureAsync();
        resultCursorsHolder.add(CompletableFuture.completedFuture(cursor));
        return resultCursorsHolder;
    }

    private Supplier<CompletionStage<Void>> mapTransactionAction(String actionName, UnmanagedTransaction tx) {
        Supplier<CompletionStage<Void>> action;
        if ("commit".equals(actionName)) {
            action = () -> ((UnmanagedTransaction)tx).commitAsync();
        } else if ("rollback".equals(actionName)) {
            action = () -> ((UnmanagedTransaction)tx).rollbackAsync();
        } else if ("terminate".equals(actionName)) {
            action = () -> {
                tx.markTerminated((Throwable)Mockito.mock(Throwable.class));
                return CompletableFuture.completedFuture(null);
            };
        } else if ("close".equals(actionName)) {
            action = () -> ((UnmanagedTransaction)tx).closeAsync();
        } else {
            throw new RuntimeException(String.format("Unknown completing action type '%s'", actionName));
        }
        return action;
    }
}

