/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.util;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.junit.MatcherAssert;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Query;
import org.neo4j.driver.Value;
import org.neo4j.driver.Values;
import org.neo4j.driver.exceptions.UntrustedServerException;
import org.neo4j.driver.exceptions.value.Uncoercible;
import org.neo4j.driver.internal.BoltServerAddress;
import org.neo4j.driver.internal.InternalBookmark;
import org.neo4j.driver.internal.messaging.v43.BoltProtocolV43;
import org.neo4j.driver.internal.spi.Connection;
import org.neo4j.driver.internal.summary.InternalInputPosition;
import org.neo4j.driver.internal.summary.InternalSummaryCounters;
import org.neo4j.driver.internal.util.MetadataExtractor;
import org.neo4j.driver.internal.util.QueryKeys;
import org.neo4j.driver.internal.util.ServerVersion;
import org.neo4j.driver.summary.DatabaseInfo;
import org.neo4j.driver.summary.Notification;
import org.neo4j.driver.summary.Plan;
import org.neo4j.driver.summary.ProfiledPlan;
import org.neo4j.driver.summary.QueryType;
import org.neo4j.driver.summary.ResultSummary;
import org.neo4j.driver.util.TestUtil;

class MetadataExtractorTest {
    private static final String RESULT_AVAILABLE_AFTER_KEY = "available_after";
    private static final String RESULT_CONSUMED_AFTER_KEY = "consumed_after";
    private final MetadataExtractor extractor = new MetadataExtractor("available_after", "consumed_after");

    MetadataExtractorTest() {
    }

    @Test
    void shouldExtractQueryKeys() {
        List<String> keys = Arrays.asList("hello", " ", "world", "!");
        HashMap<String, Integer> keyIndex = new HashMap<String, Integer>();
        keyIndex.put("hello", 0);
        keyIndex.put(" ", 1);
        keyIndex.put("world", 2);
        keyIndex.put("!", 3);
        QueryKeys extracted = this.extractor.extractQueryKeys(Collections.singletonMap("fields", Values.value(keys)));
        Assertions.assertEquals(keys, (Object)extracted.keys());
        Assertions.assertEquals(keyIndex, (Object)extracted.keyIndex());
    }

    @Test
    void shouldExtractEmptyQueryKeysWhenNoneInMetadata() {
        QueryKeys extracted = this.extractor.extractQueryKeys(Collections.emptyMap());
        Assertions.assertEquals(Collections.emptyList(), (Object)extracted.keys());
        Assertions.assertEquals(Collections.emptyMap(), (Object)extracted.keyIndex());
    }

    @Test
    void shouldExtractResultAvailableAfter() {
        Map<String, Value> metadata = Collections.singletonMap(RESULT_AVAILABLE_AFTER_KEY, Values.value((int)424242));
        long extractedResultAvailableAfter = this.extractor.extractResultAvailableAfter(metadata);
        Assertions.assertEquals((long)424242L, (long)extractedResultAvailableAfter);
    }

    @Test
    void shouldExtractNoResultAvailableAfterWhenNoneInMetadata() {
        long extractedResultAvailableAfter = this.extractor.extractResultAvailableAfter(Collections.emptyMap());
        Assertions.assertEquals((long)-1L, (long)extractedResultAvailableAfter);
    }

    @Test
    void shouldBuildResultSummaryWithQuery() {
        Query query = new Query("UNWIND range(10, 100) AS x CREATE (:Node {name: $name, x: x})", Collections.singletonMap("name", "Apa"));
        ResultSummary summary = this.extractor.extractSummary(query, MetadataExtractorTest.connectionMock(), 42L, Collections.emptyMap());
        Assertions.assertEquals((Object)query, (Object)summary.query());
    }

    @Test
    void shouldBuildResultSummaryWithServerInfo() {
        Connection connection = MetadataExtractorTest.connectionMock(new BoltServerAddress("server:42"), ServerVersion.v4_0_0);
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), connection, 42L, Collections.emptyMap());
        Assertions.assertEquals((Object)"server:42", (Object)summary.server().address());
        Assertions.assertEquals((Object)"Neo4j/4.0.0", (Object)summary.server().version());
    }

    @Test
    void shouldBuildResultSummaryWithQueryType() {
        Assertions.assertEquals((Object)QueryType.READ_ONLY, (Object)this.createWithQueryType(Values.value((String)"r")).queryType());
        Assertions.assertEquals((Object)QueryType.READ_WRITE, (Object)this.createWithQueryType(Values.value((String)"rw")).queryType());
        Assertions.assertEquals((Object)QueryType.WRITE_ONLY, (Object)this.createWithQueryType(Values.value((String)"w")).queryType());
        Assertions.assertEquals((Object)QueryType.SCHEMA_WRITE, (Object)this.createWithQueryType(Values.value((String)"s")).queryType());
        Assertions.assertNull((Object)this.createWithQueryType(null).queryType());
    }

    @Test
    void shouldBuildResultSummaryWithCounters() {
        Value stats = Values.parameters((Object[])new Object[]{"nodes-created", Values.value((int)42), "nodes-deleted", Values.value((int)4242), "relationships-created", Values.value((int)24), "relationships-deleted", Values.value((int)24), "properties-set", null, "labels-added", Values.value((int)5), "labels-removed", Values.value((int)10), "indexes-added", null, "indexes-removed", Values.value((int)0), "constraints-added", null, "constraints-removed", Values.value((int)2)});
        Map<String, Value> metadata = Collections.singletonMap("stats", stats);
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, metadata);
        Assertions.assertEquals((int)42, (int)summary.counters().nodesCreated());
        Assertions.assertEquals((int)4242, (int)summary.counters().nodesDeleted());
        Assertions.assertEquals((int)24, (int)summary.counters().relationshipsCreated());
        Assertions.assertEquals((int)24, (int)summary.counters().relationshipsDeleted());
        Assertions.assertEquals((int)0, (int)summary.counters().propertiesSet());
        Assertions.assertEquals((int)5, (int)summary.counters().labelsAdded());
        Assertions.assertEquals((int)10, (int)summary.counters().labelsRemoved());
        Assertions.assertEquals((int)0, (int)summary.counters().indexesAdded());
        Assertions.assertEquals((int)0, (int)summary.counters().indexesRemoved());
        Assertions.assertEquals((int)0, (int)summary.counters().constraintsAdded());
        Assertions.assertEquals((int)2, (int)summary.counters().constraintsRemoved());
    }

    @Test
    void shouldBuildResultSummaryWithoutCounters() {
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, Collections.emptyMap());
        Assertions.assertEquals((Object)InternalSummaryCounters.EMPTY_STATS, (Object)summary.counters());
    }

    @Test
    void shouldBuildResultSummaryWithPlan() {
        Value plan = Values.value((Object)Values.parameters((Object[])new Object[]{"operatorType", "Projection", "args", Values.parameters((Object[])new Object[]{"n", 42}), "identifiers", Values.values((Object[])new Object[]{"a", "b"}), "children", Values.values((Object[])new Object[]{Values.parameters((Object[])new Object[]{"operatorType", "AllNodeScan", "args", Values.parameters((Object[])new Object[]{"x", 4242}), "identifiers", Values.values((Object[])new Object[]{"n", "t", "f"})})})}));
        Map<String, Value> metadata = Collections.singletonMap("plan", plan);
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, metadata);
        Assertions.assertTrue((boolean)summary.hasPlan());
        Assertions.assertEquals((Object)"Projection", (Object)summary.plan().operatorType());
        Assertions.assertEquals(Collections.singletonMap("n", Values.value((int)42)), (Object)summary.plan().arguments());
        Assertions.assertEquals(Arrays.asList("a", "b"), (Object)summary.plan().identifiers());
        List children = summary.plan().children();
        Assertions.assertEquals((int)1, (int)children.size());
        Plan child = (Plan)children.get(0);
        Assertions.assertEquals((Object)"AllNodeScan", (Object)child.operatorType());
        Assertions.assertEquals(Collections.singletonMap("x", Values.value((int)4242)), (Object)child.arguments());
        Assertions.assertEquals(Arrays.asList("n", "t", "f"), (Object)child.identifiers());
        Assertions.assertEquals((int)0, (int)child.children().size());
    }

    @Test
    void shouldBuildResultSummaryWithoutPlan() {
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, Collections.emptyMap());
        Assertions.assertFalse((boolean)summary.hasPlan());
        Assertions.assertNull((Object)summary.plan());
    }

    @Test
    void shouldBuildResultSummaryWithProfiledPlan() {
        Value profile = Values.value((Object)Values.parameters((Object[])new Object[]{"operatorType", "ProduceResult", "args", Values.parameters((Object[])new Object[]{"a", 42}), "identifiers", Values.values((Object[])new Object[]{"a", "b"}), "rows", Values.value((int)424242), "dbHits", Values.value((int)242424), "time", Values.value((int)999), "children", Values.values((Object[])new Object[]{Values.parameters((Object[])new Object[]{"operatorType", "LabelScan", "args", Values.parameters((Object[])new Object[]{"x", 1}), "identifiers", Values.values((Object[])new Object[]{"y", "z"}), "rows", Values.value((int)2), "dbHits", Values.value((int)4)})})}));
        Map<String, Value> metadata = Collections.singletonMap("profile", profile);
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, metadata);
        Assertions.assertTrue((boolean)summary.hasPlan());
        Assertions.assertTrue((boolean)summary.hasProfile());
        Assertions.assertEquals((Object)"ProduceResult", (Object)summary.profile().operatorType());
        Assertions.assertEquals(Collections.singletonMap("a", Values.value((int)42)), (Object)summary.profile().arguments());
        Assertions.assertEquals(Arrays.asList("a", "b"), (Object)summary.profile().identifiers());
        Assertions.assertEquals((long)424242L, (long)summary.profile().records());
        Assertions.assertEquals((long)242424L, (long)summary.profile().dbHits());
        Assertions.assertEquals((long)999L, (long)summary.profile().time());
        Assertions.assertFalse((boolean)summary.profile().hasPageCacheStats());
        Assertions.assertEquals((double)0.0, (double)summary.profile().pageCacheHitRatio());
        Assertions.assertEquals((long)0L, (long)summary.profile().pageCacheMisses());
        Assertions.assertEquals((long)0L, (long)summary.profile().pageCacheHits());
        List children = summary.profile().children();
        Assertions.assertEquals((int)1, (int)children.size());
        ProfiledPlan child = (ProfiledPlan)children.get(0);
        Assertions.assertEquals((Object)"LabelScan", (Object)child.operatorType());
        Assertions.assertEquals(Collections.singletonMap("x", Values.value((int)1)), (Object)child.arguments());
        Assertions.assertEquals(Arrays.asList("y", "z"), (Object)child.identifiers());
        Assertions.assertEquals((long)2L, (long)child.records());
        Assertions.assertEquals((long)4L, (long)child.dbHits());
    }

    @Test
    void shouldBuildResultSummaryWithoutProfiledPlan() {
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, Collections.emptyMap());
        Assertions.assertFalse((boolean)summary.hasProfile());
        Assertions.assertNull((Object)summary.profile());
    }

    @Test
    void shouldBuildResultSummaryWithNotifications() {
        Value notification1 = Values.parameters((Object[])new Object[]{"description", "Almost bad thing", "code", "Neo.DummyNotification", "title", "A title", "severity", "WARNING", "position", Values.parameters((Object[])new Object[]{"offset", 42, "line", 4242, "column", 424242})});
        Value notification2 = Values.parameters((Object[])new Object[]{"description", "Almost good thing", "code", "Neo.GoodNotification", "title", "Good", "severity", "INFO", "position", Values.parameters((Object[])new Object[]{"offset", 1, "line", 2, "column", 3})});
        Value notifications = Values.value((Value[])new Value[]{notification1, notification2});
        Map<String, Value> metadata = Collections.singletonMap("notifications", notifications);
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, metadata);
        Assertions.assertEquals((int)2, (int)summary.notifications().size());
        Notification firstNotification = (Notification)summary.notifications().get(0);
        Notification secondNotification = (Notification)summary.notifications().get(1);
        Assertions.assertEquals((Object)"Almost bad thing", (Object)firstNotification.description());
        Assertions.assertEquals((Object)"Neo.DummyNotification", (Object)firstNotification.code());
        Assertions.assertEquals((Object)"A title", (Object)firstNotification.title());
        Assertions.assertEquals((Object)"WARNING", (Object)firstNotification.severity());
        Assertions.assertEquals((Object)new InternalInputPosition(42, 4242, 424242), (Object)firstNotification.position());
        Assertions.assertEquals((Object)"Almost good thing", (Object)secondNotification.description());
        Assertions.assertEquals((Object)"Neo.GoodNotification", (Object)secondNotification.code());
        Assertions.assertEquals((Object)"Good", (Object)secondNotification.title());
        Assertions.assertEquals((Object)"INFO", (Object)secondNotification.severity());
        Assertions.assertEquals((Object)new InternalInputPosition(1, 2, 3), (Object)secondNotification.position());
    }

    @Test
    void shouldBuildResultSummaryWithoutNotifications() {
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, Collections.emptyMap());
        Assertions.assertEquals((int)0, (int)summary.notifications().size());
    }

    @Test
    void shouldBuildResultSummaryWithResultAvailableAfter() {
        int value = 42000;
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), (long)value, Collections.emptyMap());
        Assertions.assertEquals((long)42L, (long)summary.resultAvailableAfter(TimeUnit.SECONDS));
        Assertions.assertEquals((long)value, (long)summary.resultAvailableAfter(TimeUnit.MILLISECONDS));
    }

    @Test
    void shouldBuildResultSummaryWithResultConsumedAfter() {
        int value = 42000;
        Map<String, Value> metadata = Collections.singletonMap(RESULT_CONSUMED_AFTER_KEY, Values.value((int)value));
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, metadata);
        Assertions.assertEquals((long)42L, (long)summary.resultConsumedAfter(TimeUnit.SECONDS));
        Assertions.assertEquals((long)value, (long)summary.resultConsumedAfter(TimeUnit.MILLISECONDS));
    }

    @Test
    void shouldBuildResultSummaryWithoutResultConsumedAfter() {
        ResultSummary summary = this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, Collections.emptyMap());
        Assertions.assertEquals((long)-1L, (long)summary.resultConsumedAfter(TimeUnit.SECONDS));
        Assertions.assertEquals((long)-1L, (long)summary.resultConsumedAfter(TimeUnit.MILLISECONDS));
    }

    @Test
    void shouldExtractBookmark() {
        String bookmarkValue = "neo4j:bookmark:v1:tx123456";
        Bookmark bookmark = MetadataExtractor.extractBookmarks(Collections.singletonMap("bookmark", Values.value((String)bookmarkValue)));
        Assertions.assertEquals((Object)InternalBookmark.parse((String)bookmarkValue), (Object)bookmark);
    }

    @Test
    void shouldExtractNoBookmarkWhenMetadataContainsNull() {
        Bookmark bookmark = MetadataExtractor.extractBookmarks(Collections.singletonMap("bookmark", null));
        Assertions.assertEquals((Object)InternalBookmark.empty(), (Object)bookmark);
    }

    @Test
    void shouldExtractNoBookmarkWhenMetadataContainsNullValue() {
        Bookmark bookmark = MetadataExtractor.extractBookmarks(Collections.singletonMap("bookmark", Values.NULL));
        Assertions.assertEquals((Object)InternalBookmark.empty(), (Object)bookmark);
    }

    @Test
    void shouldExtractNoBookmarkWhenMetadataContainsValueOfIncorrectType() {
        Bookmark bookmark = MetadataExtractor.extractBookmarks(Collections.singletonMap("bookmark", Values.value((int)42)));
        Assertions.assertEquals((Object)InternalBookmark.empty(), (Object)bookmark);
    }

    @Test
    void shouldExtractServerVersion() {
        Map<String, Value> metadata = Collections.singletonMap("server", Values.value((String)"Neo4j/3.5.0"));
        ServerVersion version = MetadataExtractor.extractNeo4jServerVersion(metadata);
        Assertions.assertEquals((Object)ServerVersion.v3_5_0, (Object)version);
    }

    @Test
    void shouldExtractServer() {
        String agent = "Neo4j/3.5.0";
        Map<String, Value> metadata = Collections.singletonMap("server", Values.value((String)agent));
        Value serverValue = MetadataExtractor.extractServer(metadata);
        Assertions.assertEquals((Object)agent, (Object)serverValue.asString());
    }

    @Test
    void shouldExtractDatabase() {
        Map<String, Value> metadata = Collections.singletonMap("db", Values.value((String)"MyAwesomeDatabase"));
        DatabaseInfo db = MetadataExtractor.extractDatabaseInfo(metadata);
        Assertions.assertEquals((Object)"MyAwesomeDatabase", (Object)db.name());
    }

    @Test
    void shouldDefaultToNullDatabaseName() {
        Map<String, Value> metadata = Collections.singletonMap("no_db", Values.value((String)"no_db"));
        DatabaseInfo db = MetadataExtractor.extractDatabaseInfo(metadata);
        Assertions.assertNull((Object)db.name());
    }

    @Test
    void shouldErrorWhenTypeIsWrong() {
        Map<String, Value> metadata = Collections.singletonMap("db", Values.value((long)10L));
        Uncoercible error = (Uncoercible)Assertions.assertThrows(Uncoercible.class, () -> MetadataExtractor.extractDatabaseInfo((Map)metadata));
        MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)CoreMatchers.startsWith((String)"Cannot coerce INTEGER to Java String"));
    }

    @Test
    void shouldFailToExtractServerVersionWhenMetadataDoesNotContainIt() {
        Assertions.assertThrows(UntrustedServerException.class, () -> MetadataExtractor.extractNeo4jServerVersion(Collections.singletonMap("server", Values.NULL)));
        Assertions.assertThrows(UntrustedServerException.class, () -> MetadataExtractor.extractNeo4jServerVersion(Collections.singletonMap("server", null)));
    }

    @Test
    void shouldFailToExtractServerVersionFromNonNeo4jProduct() {
        Assertions.assertThrows(UntrustedServerException.class, () -> MetadataExtractor.extractNeo4jServerVersion(Collections.singletonMap("server", Values.value((String)"NotNeo4j/1.2.3"))));
    }

    private ResultSummary createWithQueryType(Value typeValue) {
        Map<String, Value> metadata = Collections.singletonMap("type", typeValue);
        return this.extractor.extractSummary(MetadataExtractorTest.query(), MetadataExtractorTest.connectionMock(), 42L, metadata);
    }

    private static Query query() {
        return new Query("RETURN 1");
    }

    private static Connection connectionMock() {
        return MetadataExtractorTest.connectionMock(BoltServerAddress.LOCAL_DEFAULT, TestUtil.anyServerVersion());
    }

    private static Connection connectionMock(BoltServerAddress address, ServerVersion version) {
        Connection connection = (Connection)Mockito.mock(Connection.class);
        Mockito.when((Object)connection.serverAddress()).thenReturn((Object)address);
        Mockito.when((Object)connection.serverVersion()).thenReturn((Object)version);
        Mockito.when((Object)connection.protocol()).thenReturn((Object)BoltProtocolV43.INSTANCE);
        Mockito.when((Object)connection.serverAgent()).thenReturn((Object)"Neo4j/4.2.5");
        return connection;
    }
}

