/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.gds.similarity.nodesim;

import com.carrotsearch.hppc.BitSet;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.stream.BaseStream;
import java.util.stream.LongStream;
import java.util.stream.Stream;
import org.neo4j.gds.Algorithm;
import org.neo4j.gds.api.Graph;
import org.neo4j.gds.api.IdMap;
import org.neo4j.gds.api.RelationshipConsumer;
import org.neo4j.gds.core.concurrency.ParallelUtil;
import org.neo4j.gds.core.utils.BatchingProgressLogger;
import org.neo4j.gds.core.utils.SetBitsIterable;
import org.neo4j.gds.core.utils.paged.HugeObjectArray;
import org.neo4j.gds.core.utils.progress.tasks.ProgressTracker;
import org.neo4j.gds.similarity.SimilarityGraphBuilder;
import org.neo4j.gds.similarity.SimilarityGraphResult;
import org.neo4j.gds.similarity.SimilarityResult;
import org.neo4j.gds.similarity.nodesim.ImmutableNodeSimilarityResult;
import org.neo4j.gds.similarity.nodesim.MetricSimilarityComputer;
import org.neo4j.gds.similarity.nodesim.NodeSimilarityBaseConfig;
import org.neo4j.gds.similarity.nodesim.NodeSimilarityMetric;
import org.neo4j.gds.similarity.nodesim.NodeSimilarityResult;
import org.neo4j.gds.similarity.nodesim.TopKGraph;
import org.neo4j.gds.similarity.nodesim.TopKMap;
import org.neo4j.gds.similarity.nodesim.TopNList;
import org.neo4j.gds.similarity.nodesim.VectorComputer;

public class NodeSimilarity
extends Algorithm<NodeSimilarityResult> {
    private final Graph graph;
    private final boolean sortVectors;
    private final NodeSimilarityBaseConfig config;
    private final ExecutorService executorService;
    private final BitSet nodeFilter;
    private final MetricSimilarityComputer similarityComputer;
    private HugeObjectArray<long[]> vectors;
    private HugeObjectArray<double[]> weights;
    private long nodesToCompare;
    private final boolean weighted;

    public NodeSimilarity(Graph graph, NodeSimilarityBaseConfig config, ExecutorService executorService, ProgressTracker progressTracker) {
        super(progressTracker);
        this.graph = graph;
        this.sortVectors = graph.schema().relationshipSchema().availableTypes().size() > 1;
        this.config = config;
        this.executorService = executorService;
        this.nodeFilter = new BitSet(graph.nodeCount());
        this.weighted = config.hasRelationshipWeightProperty();
        String metricUsed = config.similarityMetric().toUpperCase(Locale.ROOT);
        NodeSimilarityMetric metric = MetricSimilarityComputer.valueOf(config.similarityMetric());
        this.similarityComputer = MetricSimilarityComputer.create(metric, config.similarityCutoff());
    }

    public void release() {
        this.graph.release();
    }

    public NodeSimilarityResult compute() {
        this.progressTracker.beginSubTask();
        if (this.config.computeToStream()) {
            Stream<SimilarityResult> computeToStream = this.computeToStream();
            this.progressTracker.endSubTask();
            return ImmutableNodeSimilarityResult.of(Optional.of(computeToStream), Optional.empty());
        }
        SimilarityGraphResult computeToGraph = this.computeToGraph();
        this.progressTracker.endSubTask();
        return ImmutableNodeSimilarityResult.of(Optional.empty(), Optional.of(computeToGraph));
    }

    public Stream<SimilarityResult> computeToStream() {
        this.prepare();
        this.assertRunning();
        if (this.config.hasTopN() && !this.config.hasTopK()) {
            return this.computeTopN();
        }
        return this.config.isParallel() ? this.computeParallel() : this.computeSimilarityResultStream();
    }

    public SimilarityGraphResult computeToGraph() {
        Object similarityGraph;
        boolean isTopKGraph = false;
        if (this.config.hasTopK() && !this.config.hasTopN()) {
            this.prepare();
            this.assertRunning();
            TopKMap topKMap = this.config.isParallel() ? this.computeTopKMapParallel() : this.computeTopKMap();
            isTopKGraph = true;
            similarityGraph = new TopKGraph(this.graph, topKMap);
        } else {
            Stream<SimilarityResult> similarities = this.computeToStream();
            similarityGraph = new SimilarityGraphBuilder((IdMap)this.graph, this.config.concurrency(), this.executorService).build(similarities);
        }
        return new SimilarityGraphResult((Graph)similarityGraph, this.nodesToCompare, isTopKGraph);
    }

    private void prepare() {
        this.progressTracker.beginSubTask();
        this.vectors = HugeObjectArray.newArray(long[].class, (long)this.graph.nodeCount());
        if (this.weighted) {
            this.weights = HugeObjectArray.newArray(double[].class, (long)this.graph.nodeCount());
        }
        DegreeComputer degreeComputer = new DegreeComputer();
        VectorComputer vectorComputer = VectorComputer.of(this.graph, this.weighted);
        this.vectors.setAll(node -> {
            this.graph.forEachRelationship(node, (RelationshipConsumer)degreeComputer);
            int degree = degreeComputer.degree;
            degreeComputer.reset();
            vectorComputer.reset(degree);
            if (degree >= this.config.degreeCutoff()) {
                ++this.nodesToCompare;
                this.nodeFilter.set(node);
                this.progressTracker.logProgress((long)this.graph.degree(node));
                vectorComputer.forEachRelationship(node);
                if (this.weighted) {
                    this.weights.set(node, (Object)vectorComputer.getWeights());
                }
                if (this.sortVectors) {
                    Arrays.sort(vectorComputer.targetIds.buffer);
                }
                return vectorComputer.targetIds.buffer;
            }
            this.progressTracker.logProgress((long)this.graph.degree(node));
            return null;
        });
        this.progressTracker.endSubTask();
    }

    private Stream<SimilarityResult> computeSimilarityResultStream() {
        return this.config.hasTopK() && this.config.hasTopN() ? this.computeTopN(this.computeTopKMap()) : (this.config.hasTopK() ? this.computeTopKMap().stream() : this.computeAll());
    }

    private Stream<SimilarityResult> computeParallel() {
        return this.config.hasTopK() && this.config.hasTopN() ? this.computeTopN(this.computeTopKMapParallel()) : (this.config.hasTopK() ? this.computeTopKMapParallel().stream() : this.computeAllParallel());
    }

    private Stream<SimilarityResult> computeAll() {
        this.progressTracker.beginSubTask(this.calculateWorkload());
        Stream<SimilarityResult> similarityResultStream = this.loggableAndTerminatableNodeStream().boxed().flatMap(this::computeSimilaritiesForNode);
        this.progressTracker.endSubTask();
        return similarityResultStream;
    }

    private Stream<SimilarityResult> computeAllParallel() {
        return (Stream)ParallelUtil.parallelStream((BaseStream)this.loggableAndTerminatableNodeStream(), (int)this.config.concurrency(), stream -> stream.boxed().flatMap(this::computeSimilaritiesForNode));
    }

    private TopKMap computeTopKMap() {
        this.progressTracker.beginSubTask(this.calculateWorkload());
        Comparator<SimilarityResult> comparator = this.config.normalizedK() > 0 ? SimilarityResult.DESCENDING : SimilarityResult.ASCENDING;
        TopKMap topKMap = new TopKMap(this.vectors.size(), this.nodeFilter, Math.abs(this.config.normalizedK()), comparator);
        this.loggableAndTerminatableNodeStream().forEach(node1 -> {
            long[] vector1 = (long[])this.vectors.get(node1);
            this.nodeStream(node1 + 1L).forEach(node2 -> {
                double similarity;
                double d = similarity = this.weighted ? this.computeWeightedSimilarity(vector1, (long[])this.vectors.get(node2), (double[])this.weights.get(node1), (double[])this.weights.get(node2)) : this.computeSimilarity(vector1, (long[])this.vectors.get(node2));
                if (!Double.isNaN(similarity)) {
                    topKMap.put(node1, node2, similarity);
                    topKMap.put(node2, node1, similarity);
                }
            });
        });
        this.progressTracker.endSubTask();
        return topKMap;
    }

    private TopKMap computeTopKMapParallel() {
        this.progressTracker.beginSubTask(this.calculateWorkload());
        Comparator<SimilarityResult> comparator = this.config.normalizedK() > 0 ? SimilarityResult.DESCENDING : SimilarityResult.ASCENDING;
        TopKMap topKMap = new TopKMap(this.vectors.size(), this.nodeFilter, Math.abs(this.config.normalizedK()), comparator);
        ParallelUtil.parallelStreamConsume((BaseStream)this.loggableAndTerminatableNodeStream(), (int)this.config.concurrency(), stream -> stream.forEach(node1 -> {
            long[] vector1 = (long[])this.vectors.get(node1);
            this.nodeStream().filter(node2 -> node1 != node2).forEach(node2 -> {
                double similarity;
                double d = similarity = this.weighted ? this.computeWeightedSimilarity(vector1, (long[])this.vectors.get(node2), (double[])this.weights.get(node1), (double[])this.weights.get(node2)) : this.computeSimilarity(vector1, (long[])this.vectors.get(node2));
                if (!Double.isNaN(similarity)) {
                    topKMap.put(node1, node2, similarity);
                }
            });
        }));
        this.progressTracker.endSubTask();
        return topKMap;
    }

    private Stream<SimilarityResult> computeTopN() {
        this.progressTracker.beginSubTask(this.calculateWorkload());
        TopNList topNList = new TopNList(this.config.normalizedN());
        this.loggableAndTerminatableNodeStream().forEach(node1 -> {
            long[] vector1 = (long[])this.vectors.get(node1);
            this.nodeStream(node1 + 1L).forEach(node2 -> {
                double similarity;
                double d = similarity = this.weighted ? this.computeWeightedSimilarity(vector1, (long[])this.vectors.get(node2), (double[])this.weights.get(node1), (double[])this.weights.get(node2)) : this.computeSimilarity(vector1, (long[])this.vectors.get(node2));
                if (!Double.isNaN(similarity)) {
                    topNList.add(node1, node2, similarity);
                }
            });
        });
        this.progressTracker.endSubTask();
        return topNList.stream();
    }

    private Stream<SimilarityResult> computeTopN(TopKMap topKMap) {
        TopNList topNList = new TopNList(this.config.normalizedN());
        topKMap.forEach(topNList::add);
        return topNList.stream();
    }

    private LongStream nodeStream() {
        return this.nodeStream(0L);
    }

    private LongStream loggableAndTerminatableNodeStream() {
        return this.checkProgress(this.nodeStream());
    }

    private double computeWeightedSimilarity(long[] vector1, long[] vector2, double[] weights1, double[] weights2) {
        double similarity = this.similarityComputer.computeWeightedSimilarity(vector1, vector2, weights1, weights2);
        this.progressTracker.logProgress();
        return similarity;
    }

    private double computeSimilarity(long[] vector1, long[] vector2) {
        double similarity = this.similarityComputer.computeSimilarity(vector1, vector2);
        this.progressTracker.logProgress();
        return similarity;
    }

    private LongStream checkProgress(LongStream stream) {
        return stream.peek(node -> {
            if ((node & BatchingProgressLogger.MAXIMUM_LOG_INTERVAL) == 0L) {
                this.assertRunning();
            }
        });
    }

    private LongStream nodeStream(long offset) {
        return new SetBitsIterable(this.nodeFilter, offset).stream();
    }

    private long calculateWorkload() {
        long workload = this.nodesToCompare * this.nodesToCompare;
        if (this.config.concurrency() == 1) {
            workload /= 2L;
        }
        return workload;
    }

    private Stream<SimilarityResult> computeSimilaritiesForNode(long node1) {
        long[] vector1 = (long[])this.vectors.get(node1);
        return this.nodeStream(node1 + 1L).mapToObj(node2 -> {
            double similarity = this.weighted ? this.computeWeightedSimilarity(vector1, (long[])this.vectors.get(node2), (double[])this.weights.get(node1), (double[])this.weights.get(node2)) : this.computeSimilarity(vector1, (long[])this.vectors.get(node2));
            return Double.isNaN(similarity) ? null : new SimilarityResult(node1, node2, similarity);
        }).filter(Objects::nonNull);
    }

    private static final class DegreeComputer
    implements RelationshipConsumer {
        long lastTarget = -1L;
        int degree = 0;

        private DegreeComputer() {
        }

        public boolean accept(long source, long target) {
            if (source != target && this.lastTarget != target) {
                ++this.degree;
            }
            this.lastTarget = target;
            return true;
        }

        void reset() {
            this.lastTarget = -1L;
            this.degree = 0;
        }
    }
}

