/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.gds.leiden;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.DoubleAdder;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.neo4j.gds.Algorithm;
import org.neo4j.gds.api.Graph;
import org.neo4j.gds.api.properties.nodes.NodePropertyValues;
import org.neo4j.gds.api.schema.Direction;
import org.neo4j.gds.core.concurrency.Pools;
import org.neo4j.gds.core.concurrency.RunWithConcurrency;
import org.neo4j.gds.core.utils.paged.HugeDoubleArray;
import org.neo4j.gds.core.utils.paged.HugeLongArray;
import org.neo4j.gds.core.utils.partition.Partition;
import org.neo4j.gds.core.utils.partition.PartitionUtils;
import org.neo4j.gds.core.utils.progress.tasks.ProgressTracker;
import org.neo4j.gds.leiden.CommunityData;
import org.neo4j.gds.leiden.GraphAggregationPhase;
import org.neo4j.gds.leiden.InitVolumeTask;
import org.neo4j.gds.leiden.LeidenDendrogramManager;
import org.neo4j.gds.leiden.LeidenResult;
import org.neo4j.gds.leiden.LeidenUtils;
import org.neo4j.gds.leiden.LocalMovePhase;
import org.neo4j.gds.leiden.ModularityComputer;
import org.neo4j.gds.leiden.RefinementPhase;
import org.neo4j.gds.leiden.SeedCommunityManager;

public class Leiden
extends Algorithm<LeidenResult> {
    private final Graph rootGraph;
    private final Direction direction;
    private final int maxIterations;
    private final double initialGamma;
    private final double theta;
    private double[] modularities;
    private double modularity;
    private final LeidenDendrogramManager dendrogramManager;
    private final Optional<NodePropertyValues> seedValues;
    private final ExecutorService executorService;
    private final int concurrency;
    private final long randomSeed;
    private SeedCommunityManager seedCommunityManager;
    private final double tolerance;

    public Leiden(Graph graph, int maxIterations, double initialGamma, double theta, boolean includeIntermediateCommunities, long randomSeed, @Nullable NodePropertyValues seedValues, double tolerance, int concurrency, ProgressTracker progressTracker) {
        super(progressTracker);
        this.rootGraph = graph;
        this.direction = this.rootGraph.schema().direction();
        this.maxIterations = maxIterations;
        this.initialGamma = initialGamma;
        this.theta = theta;
        this.randomSeed = randomSeed;
        this.executorService = Pools.DEFAULT;
        this.concurrency = concurrency;
        this.dendrogramManager = new LeidenDendrogramManager(this.rootGraph, maxIterations, concurrency, includeIntermediateCommunities);
        this.seedValues = Optional.ofNullable(seedValues);
        this.modularities = new double[maxIterations];
        this.modularity = 0.0;
        this.tolerance = tolerance;
    }

    public LeidenResult compute() {
        int iteration;
        this.progressTracker.beginSubTask("Leiden");
        Graph workingGraph = this.rootGraph;
        long nodeCount = workingGraph.nodeCount();
        HugeLongArray localMoveCommunities = LeidenUtils.createStartingCommunities(nodeCount, this.seedValues.orElse(null));
        this.seedCommunityManager = SeedCommunityManager.create(this.seedValues.isPresent(), localMoveCommunities);
        HugeDoubleArray localMoveNodeVolumes = HugeDoubleArray.newArray((long)nodeCount);
        HugeDoubleArray localMoveCommunityVolumes = HugeDoubleArray.newArray((long)nodeCount);
        double modularityScaleCoefficient = this.initVolumes(localMoveNodeVolumes, localMoveCommunityVolumes, localMoveCommunities);
        double gamma = this.initialGamma * modularityScaleCoefficient;
        HugeLongArray currentActualCommunities = HugeLongArray.newArray((long)this.rootGraph.nodeCount());
        boolean didConverge = false;
        this.progressTracker.beginSubTask("Iteration");
        for (iteration = 0; iteration < this.maxIterations; ++iteration) {
            this.progressTracker.beginSubTask("Local Move");
            LocalMovePhase localMovePhase = LocalMovePhase.create(workingGraph, localMoveCommunities, localMoveNodeVolumes, localMoveCommunityVolumes, gamma, this.concurrency);
            localMovePhase.run();
            boolean localPhaseConverged = localMovePhase.swaps == 0L;
            this.progressTracker.endSubTask("Local Move");
            this.progressTracker.beginSubTask("Modularity Computation");
            this.updateModularity(workingGraph, localMoveCommunities, localMoveCommunityVolumes, modularityScaleCoefficient, gamma, localPhaseConverged, iteration);
            this.progressTracker.endSubTask("Modularity Computation");
            if (localPhaseConverged) {
                didConverge = true;
                break;
            }
            ToleranceStatus toleranceStatus = this.getToleranceStatus(iteration);
            if (toleranceStatus == ToleranceStatus.DECREASE) break;
            this.dendrogramManager.updateOutputDendrogram(workingGraph, currentActualCommunities, localMoveCommunities, this.seedCommunityManager, iteration);
            if (toleranceStatus == ToleranceStatus.CONVERGED) {
                didConverge = true;
                this.modularity = this.modularities[iteration];
                ++iteration;
                break;
            }
            if (iteration < this.maxIterations - 1) {
                this.progressTracker.beginSubTask("Refinement");
                RefinementPhase refinementPhase = RefinementPhase.create(workingGraph, localMoveCommunities, localMoveNodeVolumes, localMoveCommunityVolumes, gamma, this.theta, this.randomSeed, this.concurrency, this.executorService, this.progressTracker);
                RefinementPhase.RefinementPhaseResult refinementPhaseResult = refinementPhase.run();
                HugeLongArray refinedCommunities = refinementPhaseResult.communities();
                HugeDoubleArray refinedCommunityVolumes = refinementPhaseResult.communityVolumes();
                long maximumRefinedCommunityId = refinementPhaseResult.maximumRefinedCommunityId();
                this.progressTracker.endSubTask("Refinement");
                this.progressTracker.beginSubTask("Aggregation");
                this.dendrogramManager.updateAlgorithmDendrogram(workingGraph, currentActualCommunities, refinedCommunities, iteration);
                GraphAggregationPhase graphAggregationPhase = new GraphAggregationPhase(workingGraph, this.direction, refinedCommunities, maximumRefinedCommunityId, this.executorService, this.concurrency, this.terminationFlag, this.progressTracker);
                long previousNodeCount = workingGraph.nodeCount();
                workingGraph = graphAggregationPhase.run();
                CommunityData communityData = Leiden.maintainPartition(workingGraph, localMoveCommunities, refinedCommunityVolumes, previousNodeCount);
                localMoveCommunities = communityData.seededCommunitiesForNextIteration;
                localMoveCommunityVolumes = communityData.communityVolumes;
                localMoveNodeVolumes = communityData.aggregatedNodeSeedVolume;
                this.progressTracker.endSubTask("Aggregation");
            }
            this.modularity = this.modularities[iteration];
        }
        this.progressTracker.endSubTask("Iteration");
        this.progressTracker.endSubTask("Leiden");
        return this.getLeidenResult(didConverge, iteration);
    }

    @NotNull
    private LeidenResult getLeidenResult(boolean didConverge, int iteration) {
        boolean stoppedAtFirstIteration;
        boolean bl = stoppedAtFirstIteration = didConverge && iteration == 0;
        if (stoppedAtFirstIteration) {
            double modularity = this.modularities[0];
            return LeidenResult.of(LeidenUtils.createStartingCommunities(this.rootGraph.nodeCount(), this.seedValues.orElse(null)), 1, didConverge, null, new double[]{modularity}, modularity);
        }
        return LeidenResult.of(this.dendrogramManager.getCurrent(), iteration, didConverge, this.dendrogramManager, this.resizeModularitiesArray(iteration), this.modularity);
    }

    private void updateModularity(Graph workingGraph, HugeLongArray localMoveCommunities, HugeDoubleArray localMoveCommunityVolumes, double modularityScaleCoefficient, double gamma, boolean localPhaseConverged, int iteration) {
        boolean shouldCalculateModularity;
        boolean bl = shouldCalculateModularity = !localPhaseConverged || iteration == 0;
        if (shouldCalculateModularity) {
            this.modularities[iteration] = ModularityComputer.compute(workingGraph, localMoveCommunities, localMoveCommunityVolumes, gamma, modularityScaleCoefficient, this.concurrency, this.executorService, this.progressTracker);
        }
    }

    private double initVolumes(HugeDoubleArray nodeVolumes, HugeDoubleArray communityVolumes, HugeLongArray initialCommunities) {
        double totalVolume;
        this.progressTracker.beginSubTask("Initialization");
        DoubleAdder volumeAdder = new DoubleAdder();
        if (this.rootGraph.hasRelationshipProperty()) {
            List tasks = PartitionUtils.rangePartition((int)this.concurrency, (long)this.rootGraph.nodeCount(), partition -> new InitVolumeTask(this.rootGraph.concurrentCopy(), nodeVolumes, (Partition)partition, volumeAdder), Optional.empty());
            RunWithConcurrency.builder().concurrency(this.concurrency).tasks((Iterable)tasks).executor(this.executorService).run();
            totalVolume = volumeAdder.sum();
        } else {
            nodeVolumes.setAll(arg_0 -> ((Graph)this.rootGraph).degree(arg_0));
            totalVolume = this.rootGraph.relationshipCount();
        }
        this.rootGraph.forEachNode(nodeId -> {
            long communityId = initialCommunities.get(nodeId);
            this.progressTracker.logProgress();
            communityVolumes.addTo(communityId, nodeVolumes.get(nodeId));
            return true;
        });
        this.progressTracker.endSubTask("Initialization");
        return 1.0 / totalVolume;
    }

    @NotNull
    static CommunityData maintainPartition(Graph workingGraph, @NotNull HugeLongArray localPhaseCommunities, HugeDoubleArray refinedCommunityVolumes, long previousNodeCount) {
        HugeLongArray inputCommunities = HugeLongArray.newArray((long)workingGraph.nodeCount());
        HugeLongArray localPhaseCommunityToAggregatedNewId = HugeLongArray.newArray((long)previousNodeCount);
        localPhaseCommunityToAggregatedNewId.setAll(l -> -1L);
        HugeDoubleArray aggregatedCommunitySeedVolume = HugeDoubleArray.newArray((long)workingGraph.nodeCount());
        HugeDoubleArray aggregatedNodeSeedVolume = HugeDoubleArray.newArray((long)workingGraph.nodeCount());
        workingGraph.forEachNode(aggregatedCommunityId -> {
            long aggregatedSeedCommunityId;
            long refinedCommunityId = workingGraph.toOriginalNodeId(aggregatedCommunityId);
            long localPhaseCommunityId = localPhaseCommunities.get(refinedCommunityId);
            if (localPhaseCommunityToAggregatedNewId.get(localPhaseCommunityId) != -1L) {
                aggregatedSeedCommunityId = localPhaseCommunityToAggregatedNewId.get(localPhaseCommunityId);
            } else {
                aggregatedSeedCommunityId = aggregatedCommunityId;
                localPhaseCommunityToAggregatedNewId.set(localPhaseCommunityId, aggregatedSeedCommunityId);
            }
            double volumeOfTheAggregatedCommunity = refinedCommunityVolumes.get(refinedCommunityId);
            aggregatedCommunitySeedVolume.addTo(aggregatedSeedCommunityId, volumeOfTheAggregatedCommunity);
            inputCommunities.set(aggregatedCommunityId, aggregatedSeedCommunityId);
            aggregatedNodeSeedVolume.set(aggregatedCommunityId, volumeOfTheAggregatedCommunity);
            return true;
        });
        return new CommunityData(inputCommunities, aggregatedCommunitySeedVolume, aggregatedNodeSeedVolume);
    }

    public void release() {
    }

    private double[] resizeModularitiesArray(int iteration) {
        double[] resizedModularities = new double[iteration];
        if (iteration >= this.maxIterations) {
            return this.modularities;
        }
        System.arraycopy(this.modularities, 0, resizedModularities, 0, iteration);
        return resizedModularities;
    }

    private ToleranceStatus getToleranceStatus(int iteration) {
        if (iteration == 0) {
            return ToleranceStatus.CONTINUE;
        }
        double difference = this.modularities[iteration] - this.modularities[iteration - 1];
        if (difference < 0.0) {
            return ToleranceStatus.DECREASE;
        }
        return Double.compare(difference, this.tolerance) < 0 ? ToleranceStatus.CONVERGED : ToleranceStatus.CONTINUE;
    }

    private static enum ToleranceStatus {
        CONVERGED,
        DECREASE,
        CONTINUE;

    }
}

