/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.gds.paths.delta;

import com.carrotsearch.hppc.DoubleArrayDeque;
import com.carrotsearch.hppc.LongArrayDeque;
import com.carrotsearch.hppc.LongArrayList;
import com.carrotsearch.hppc.cursors.LongCursor;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.LongStream;
import java.util.stream.Stream;
import org.apache.commons.lang3.mutable.MutableLong;
import org.neo4j.gds.Algorithm;
import org.neo4j.gds.api.Graph;
import org.neo4j.gds.core.concurrency.ParallelUtil;
import org.neo4j.gds.core.utils.mem.MemoryEstimation;
import org.neo4j.gds.core.utils.mem.MemoryEstimations;
import org.neo4j.gds.core.utils.mem.MemoryRange;
import org.neo4j.gds.core.utils.paged.HugeAtomicDoubleArray;
import org.neo4j.gds.core.utils.paged.HugeAtomicLongArray;
import org.neo4j.gds.core.utils.paged.HugeLongArray;
import org.neo4j.gds.core.utils.partition.PartitionUtils;
import org.neo4j.gds.core.utils.progress.tasks.ProgressTracker;
import org.neo4j.gds.paths.ImmutablePathResult;
import org.neo4j.gds.paths.PathResult;
import org.neo4j.gds.paths.delta.TentativeDistances;
import org.neo4j.gds.paths.delta.config.AllShortestPathsDeltaBaseConfig;
import org.neo4j.gds.paths.dijkstra.DijkstraResult;

public final class DeltaStepping
extends Algorithm<DijkstraResult> {
    public static final String DESCRIPTION = "The Delta Stepping shortest path algorithm computes the shortest (weighted) path between one node and any other node in the graph. The computation is run multi-threaded";
    private static final int NO_BIN = Integer.MAX_VALUE;
    private static final int BIN_SIZE_THRESHOLD = 1000;
    private static final int BATCH_SIZE = 64;
    private final Graph graph;
    private final long startNode;
    private final double delta;
    private final int concurrency;
    private final HugeLongArray frontier;
    private final TentativeDistances distances;
    private final ExecutorService executorService;
    private static final long[] EMPTY_ARRAY = new long[0];

    public static DeltaStepping of(Graph graph, AllShortestPathsDeltaBaseConfig config, ExecutorService executorService, ProgressTracker progressTracker) {
        return new DeltaStepping(graph, graph.toMappedNodeId(config.sourceNode()), config.delta(), config.concurrency(), true, executorService, progressTracker);
    }

    public static MemoryEstimation memoryEstimation(boolean storePredecessors) {
        MemoryEstimations.Builder builder = MemoryEstimations.builder(DeltaStepping.class).perNode("distance array", HugeAtomicDoubleArray::memoryEstimation).rangePerGraphDimension("shared bin", (dimensions, concurrency) -> {
            long lowerBound = HugeLongArray.memoryEstimation((long)dimensions.nodeCount());
            long upperBound = HugeLongArray.memoryEstimation((long)dimensions.relCountUpperBound());
            return MemoryRange.of((long)lowerBound, (long)Math.max(lowerBound, upperBound));
        }).rangePerGraphDimension("local bins", (dimensions, concurrency) -> {
            long lowerBound = HugeLongArray.memoryEstimation((long)(dimensions.nodeCount() / (long)concurrency.intValue()));
            long upperBound = HugeLongArray.memoryEstimation((long)((long)concurrency.intValue() * dimensions.nodeCount()));
            return MemoryRange.of((long)lowerBound, (long)Math.max(lowerBound, upperBound));
        });
        if (storePredecessors) {
            builder.perNode("predecessor array", HugeAtomicLongArray::memoryEstimation);
        }
        return builder.build();
    }

    private DeltaStepping(Graph graph, long startNode, double delta, int concurrency, boolean storePredecessors, ExecutorService executorService, ProgressTracker progressTracker) {
        super(progressTracker);
        this.graph = graph;
        this.startNode = startNode;
        this.delta = delta;
        this.concurrency = concurrency;
        this.executorService = executorService;
        this.frontier = HugeLongArray.newArray((long)graph.relationshipCount());
        this.distances = storePredecessors ? TentativeDistances.distanceAndPredecessors(graph.nodeCount(), concurrency) : TentativeDistances.distanceOnly(graph.nodeCount(), concurrency);
    }

    public DijkstraResult compute() {
        this.progressTracker.beginSubTask();
        int iteration = 0;
        int currentBin = 0;
        AtomicLong frontierIndex = new AtomicLong(0L);
        AtomicLong frontierSize = new AtomicLong(1L);
        this.frontier.set((long)currentBin, this.startNode);
        this.distances.set(this.startNode, -1L, 0.0);
        List<DeltaSteppingTask> relaxTasks = IntStream.range(0, this.concurrency).mapToObj(i -> new DeltaSteppingTask(this.graph, this.frontier, this.distances, this.delta, frontierIndex)).collect(Collectors.toList());
        while (currentBin != Integer.MAX_VALUE) {
            this.progressTracker.beginSubTask();
            for (DeltaSteppingTask task2 : relaxTasks) {
                task2.setPhase(Phase.RELAX);
                task2.setBinIndex(currentBin);
                task2.setFrontierLength(frontierSize.longValue());
            }
            ParallelUtil.run(relaxTasks, (ExecutorService)this.executorService);
            this.progressTracker.endSubTask();
            currentBin = relaxTasks.stream().mapToInt(DeltaSteppingTask::minNonEmptyBin).min().orElseThrow();
            this.progressTracker.beginSubTask();
            frontierIndex.set(0L);
            relaxTasks.forEach(task -> task.setPhase(Phase.SYNC));
            for (DeltaSteppingTask task2 : relaxTasks) {
                task2.setPhase(Phase.SYNC);
                task2.setBinIndex(currentBin);
            }
            ParallelUtil.run(relaxTasks, (ExecutorService)this.executorService);
            this.progressTracker.endSubTask();
            ++iteration;
            frontierSize.set(frontierIndex.longValue());
            frontierIndex.set(0L);
        }
        return new DijkstraResult(DeltaStepping.pathResults(this.distances, this.startNode, this.concurrency), () -> ((ProgressTracker)this.progressTracker).endSubTask());
    }

    public void release() {
    }

    private static Stream<PathResult> pathResults(TentativeDistances tentativeDistances, long sourceNode, int concurrency) {
        HugeAtomicDoubleArray distances = tentativeDistances.distances();
        HugeAtomicLongArray predecessors = tentativeDistances.predecessors().orElseThrow();
        AtomicLong pathIndex = new AtomicLong(0L);
        List partitions = PartitionUtils.rangePartition((int)concurrency, (long)predecessors.size(), partition -> partition, Optional.empty());
        return (Stream)ParallelUtil.parallelStream(partitions.stream(), (int)concurrency, parallelStream -> parallelStream.flatMap(partition -> {
            MutableLong localPathIndex = new MutableLong(pathIndex.getAndAdd(partition.nodeCount()));
            ImmutablePathResult.Builder pathResultBuilder = ImmutablePathResult.builder().sourceNode(sourceNode);
            return LongStream.range(partition.startNode(), partition.startNode() + partition.nodeCount()).filter(target -> predecessors.get(target) != Long.MAX_VALUE).mapToObj(targetNode -> DeltaStepping.pathResult(pathResultBuilder, localPathIndex.getAndIncrement(), sourceNode, targetNode, distances, predecessors));
        }));
    }

    private static PathResult pathResult(ImmutablePathResult.Builder pathResultBuilder, long pathIndex, long sourceNode, long targetNode, HugeAtomicDoubleArray distances, HugeAtomicLongArray predecessors) {
        LongArrayDeque pathNodeIds = new LongArrayDeque();
        DoubleArrayDeque costs = new DoubleArrayDeque();
        long lastNode = targetNode;
        while (true) {
            pathNodeIds.addFirst(lastNode);
            costs.addFirst(distances.get(lastNode));
            if (lastNode == sourceNode) break;
            lastNode = predecessors.get(lastNode);
        }
        return pathResultBuilder.index(pathIndex).targetNode(targetNode).nodeIds(pathNodeIds.toArray()).relationshipIds(EMPTY_ARRAY).costs(costs.toArray()).build();
    }

    private static class DeltaSteppingTask
    implements Runnable {
        private final Graph graph;
        private final HugeLongArray frontier;
        private final TentativeDistances distances;
        private final double delta;
        private int binIndex;
        private final AtomicLong frontierIndex;
        private long frontierLength;
        private LongArrayList[] localBins;
        private Phase phase = Phase.RELAX;

        DeltaSteppingTask(Graph graph, HugeLongArray frontier, TentativeDistances distances, double delta, AtomicLong frontierIndex) {
            this.graph = graph.concurrentCopy();
            this.frontier = frontier;
            this.distances = distances;
            this.delta = delta;
            this.frontierIndex = frontierIndex;
            this.localBins = new LongArrayList[0];
        }

        @Override
        public void run() {
            if (this.phase == Phase.RELAX) {
                this.relaxGlobalBin();
                this.relaxLocalBin();
            } else if (this.phase == Phase.SYNC) {
                this.updateFrontier();
            }
        }

        void setPhase(Phase phase) {
            this.phase = phase;
        }

        void setBinIndex(int binIndex) {
            this.binIndex = binIndex;
        }

        void setFrontierLength(long frontierLength) {
            this.frontierLength = frontierLength;
        }

        int minNonEmptyBin() {
            for (int i = this.binIndex; i < this.localBins.length; ++i) {
                if (this.localBins[i] == null || this.localBins[i].isEmpty()) continue;
                return i;
            }
            return Integer.MAX_VALUE;
        }

        private void relaxGlobalBin() {
            long offset;
            while ((offset = this.frontierIndex.getAndAdd(64L)) < this.frontierLength) {
                long limit = Math.min(offset + 64L, this.frontierLength);
                for (long idx = offset; idx < limit; ++idx) {
                    long nodeId = this.frontier.get(idx);
                    if (!(this.distances.distance(nodeId) >= this.delta * (double)this.binIndex)) continue;
                    this.relaxNode(nodeId);
                }
            }
        }

        private void relaxLocalBin() {
            while (this.binIndex < this.localBins.length && this.localBins[this.binIndex] != null && !this.localBins[this.binIndex].isEmpty() && this.localBins[this.binIndex].size() < 1000) {
                LongArrayList binCopy = this.localBins[this.binIndex].clone();
                this.localBins[this.binIndex].elementsCount = 0;
                binCopy.forEach(this::relaxNode);
            }
        }

        private void relaxNode(long nodeId) {
            this.graph.forEachRelationship(nodeId, 1.0, (sourceNodeId, targetNodeId, weight) -> {
                double oldDist = this.distances.distance(targetNodeId);
                double newDist = this.distances.distance(sourceNodeId) + weight;
                while (Double.compare(newDist, oldDist) < 0) {
                    double witness = this.distances.compareAndExchange(targetNodeId, oldDist, newDist, sourceNodeId);
                    if (Double.compare(witness, oldDist) == 0) {
                        int destBin = (int)(newDist / this.delta);
                        if (destBin >= this.localBins.length) {
                            this.localBins = Arrays.copyOf(this.localBins, destBin + 1);
                        }
                        if (this.localBins[destBin] == null) {
                            this.localBins[destBin] = new LongArrayList();
                        }
                        this.localBins[destBin].add(targetNodeId);
                        break;
                    }
                    oldDist = this.distances.distance(targetNodeId);
                }
                return true;
            });
        }

        private void updateFrontier() {
            if (this.binIndex < this.localBins.length && this.localBins[this.binIndex] != null && !this.localBins[this.binIndex].isEmpty()) {
                int size = this.localBins[this.binIndex].size();
                long offset = this.frontierIndex.getAndAdd(size);
                for (LongCursor longCursor : this.localBins[this.binIndex]) {
                    long index = offset + (long)longCursor.index;
                    this.frontier.set(index, longCursor.value);
                }
                this.localBins[this.binIndex].elementsCount = 0;
            }
        }
    }

    static enum Phase {
        RELAX,
        SYNC;

    }
}

