/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.gds.core.loading;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Locale;
import java.util.Random;
import java.util.function.Predicate;
import java.util.stream.DoubleStream;
import java.util.stream.LongStream;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.function.Executable;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.provider.ValueSource;
import org.neo4j.gds.annotation.SuppressForbidden;
import org.neo4j.gds.core.loading.DoubleCodec;
import org.neo4j.util.FeatureToggles;

public abstract class DoubleCodecTestBase {
    private static final boolean DEBUG_PRINT = FeatureToggles.flag(DoubleCodecTestBase.class, (String)"debugPrint", (boolean)false);
    private static final int NUMBER_OF_DOUBLES_TO_TEST = 100000;
    private final DoubleCodec compressor;

    public DoubleCodecTestBase(DoubleCodec compressor) {
        this.compressor = compressor;
    }

    static double longAsDouble(long value) {
        long signBit = value & Long.MIN_VALUE;
        if (value < 0L) {
            value = -(value + 1L);
        }
        assert (value < 0x8000000000000L) : "Can only encode longs with an absolute value of less than 2251799813685248";
        long dataBits = value & 0x7FFFFFFFFFFFFL;
        long doubleBits = Double.doubleToLongBits(Double.NaN) | signBit | dataBits;
        return Double.longBitsToDouble(doubleBits);
    }

    static long longFromDouble(double value) {
        long doubleBits = Double.doubleToRawLongBits(value);
        long longValue = doubleBits & 0x7FFFFFFFFFFFFL;
        long signBit = doubleBits & Long.MIN_VALUE;
        if (signBit != 0L) {
            longValue = -longValue - 1L;
        }
        return longValue;
    }

    @ValueSource(longs={0L, 1L, 42L, 1337L, 0L, -1L, -42L, -1337L, 0x7FFFFFFFL, -2147483648L, 0x4000000000000L, -1125899906842624L})
    @ParameterizedTest
    void testLongEmbedding(long value) {
        double encoded = DoubleCodecTestBase.longAsDouble(value);
        Assertions.assertTrue((boolean)Double.isNaN(encoded));
        long decoded = DoubleCodecTestBase.longFromDouble(encoded);
        Assertions.assertEquals((long)value, (long)decoded);
    }

    @ParameterizedTest
    @MethodSource(value={"testDoubles"})
    void testSingleValueDoubleCompression(double input) {
        byte[] compressed = this.compressor.compressDouble(input);
        DoubleCodec.CompressionInfo info = this.compressor.describeCompressedValue(compressed, 0, input);
        new Check(info).execute();
    }

    @Test
    void testDoubleCompression() {
        this.testDoubleCompression(DoubleCodecTestBase.testDoubles(), true);
    }

    @Test
    void testDoubleCompressionForConsecutiveIdValues() {
        double[] data = LongStream.range(0L, 100000L).mapToDouble(v -> v).toArray();
        this.testDoubleCompression(data, false);
    }

    @Test
    void testDoubleCompressionForRandomValues() {
        long seed = new Random().nextLong();
        Random random = new Random(seed);
        int maxSignificantWidth = this.compressor.supportedSignificandWith();
        double[] data = DoubleStream.generate(() -> {
            int significantWidth = random.nextInt(maxSignificantWidth);
            long significant = Long.reverse(random.nextLong() & (1L << significantWidth) - 1L);
            int exponent = random.nextInt(2047);
            return Double.longBitsToDouble((long)exponent << 52 | significant);
        }).limit(100000L).toArray();
        try {
            this.testDoubleCompression(data, false);
        }
        catch (AssertionError e) {
            throw new RuntimeException("Seed for random values: " + seed, (Throwable)((Object)e));
        }
    }

    static double[] testDoubles() {
        double specialNaN = Double.longBitsToDouble(Double.doubleToLongBits(Double.NaN) + 42L);
        return new double[]{0.0, 0.0, -0.0, 0.15, -0.15, 0.30000000000000004, -0.30000000000000004, 1.0, 2.0, 3.0, -1.0, -2.0, -3.0, 23.0, -23.0, 23.142857, -23.142857, 42.0, -42.0, 127.0, 128.0, 129.0, 1024.0, -2048.0, 1.6777216E7, -1.337421337E9, 1337.42, Math.pow(2.0, 52.0) - 1.0, Math.pow(2.0, 52.0), Math.pow(2.0, 52.0) + 1.0, Math.pow(2.0, 53.0) - 1.0, Math.pow(2.0, 53.0), Math.pow(2.0, 53.0) + 1.0, Math.pow(2.0, 53.0) + 2.0, Math.pow(2.0, 54.0), Math.pow(2.0, 54.0) + 1.0, Math.pow(2.0, 54.0) + 2.0, Math.pow(2.0, 55.0), Math.pow(2.0, 55.0) + 1.0, Math.pow(2.0, 55.0) + 2.0, Math.pow(2.0, 56.0), Math.pow(2.0, 56.0) + 1.0, Math.pow(2.0, 56.0) + 2.0, Math.pow(2.0, 255.0), Math.pow(2.0, 256.0), Math.pow(2.0, 512.0), Math.pow(2.0, 512.0) + 42.0, Double.POSITIVE_INFINITY, Double.NEGATIVE_INFINITY, Double.NaN, specialNaN};
    }

    void testDoubleCompression(double[] original, boolean detailPrint) {
        byte[] compressed = new byte[10 * original.length];
        int outLength = this.compressor.compressDoubles(original, original.length, compressed);
        compressed = Arrays.copyOf(compressed, outLength);
        RuntimeException caughtException = null;
        double[] decompressed = new double[original.length];
        try {
            this.compressor.decompressDoubles(compressed, original.length, decompressed, 0);
        }
        catch (RuntimeException e) {
            caughtException = e;
        }
        ByteBuffer uncompressedBuffer = ByteBuffer.allocate(8 * original.length).order(ByteOrder.BIG_ENDIAN);
        for (double v : original) {
            uncompressedBuffer.putLong(Double.doubleToRawLongBits(v));
        }
        byte[] uncompressed = Arrays.copyOfRange(uncompressedBuffer.array(), uncompressedBuffer.arrayOffset(), uncompressedBuffer.arrayOffset() + uncompressedBuffer.position());
        int uncompressedSize = uncompressed.length;
        int compressedSize = compressed.length;
        double savings = 1.0 - (double)compressedSize / (double)uncompressedSize;
        double bytesPerValue = (double)compressedSize / (double)original.length;
        ArrayList<DoubleCodec.CompressionInfo> compressionInfos = new ArrayList<DoubleCodec.CompressionInfo>(original.length);
        int[] sizeDistribution = new int[10];
        int[] typeDistribution = new int[9];
        int[][] sizePerType = new int[9][10];
        int pos = 0;
        for (double input : original) {
            DoubleCodec.CompressionInfo info = this.compressor.describeCompressedValue(compressed, pos, input);
            compressionInfos.add(info);
            int size = info.compressedSize();
            int type = info.compressedType();
            int n = size;
            sizeDistribution[n] = sizeDistribution[n] + 1;
            int n2 = type;
            typeDistribution[n2] = typeDistribution[n2] + 1;
            int[] nArray = sizePerType[type];
            int n3 = size;
            nArray[n3] = nArray[n3] + 1;
            pos += size;
        }
        this.debugPrint(original, detailPrint, compressed, decompressed, uncompressed, uncompressedSize, compressedSize, savings, bytesPerValue, sizeDistribution, typeDistribution, sizePerType);
        Assertions.assertAll((String)"Check compression<>decompression cycle", compressionInfos.stream().map(x$0 -> new Check((DoubleCodec.CompressionInfo)x$0)).filter(Predicate.not(Check::valid)).limit(5L).map(Check::asExecutable));
        if (caughtException != null) {
            throw caughtException;
        }
    }

    @SuppressForbidden(reason="this is supposed to print helpful stuff")
    private void debugPrint(double[] original, boolean detailPrint, byte[] compressed, double[] decompressed, byte[] uncompressed, int uncompressedSize, int compressedSize, double savings, double bytesPerValue, int[] sizeDistribution, int[] typeDistribution, int[][] sizePerType) {
        if (DEBUG_PRINT) {
            if (detailPrint) {
                System.out.printf(Locale.ENGLISH, "original = %s%n", Arrays.toString(original));
                System.out.printf(Locale.ENGLISH, "decompressed = %s%n", Arrays.toString(decompressed));
                System.out.printf(Locale.ENGLISH, "compressed [%d] = %s%n", compressedSize, Arrays.toString(compressed));
                System.out.printf(Locale.ENGLISH, "uncompressed [%d] = %s%n", uncompressedSize, Arrays.toString(uncompressed));
                System.out.printf(Locale.ENGLISH, "space savings %.2f%%%n", 100.0 * savings);
                System.out.printf(Locale.ENGLISH, "bytes per value %.4f%n", bytesPerValue);
            } else {
                System.out.printf(Locale.ENGLISH, "uncompressed size = [%d] | compressed size = [%d] | space savings %.2f%% | bytes per value = %.4f%n", uncompressedSize, compressedSize, 100.0 * savings, bytesPerValue);
            }
            System.out.println("   Compression size  |  Number of Values  |  Percentile  |  Total Percentile");
            int cumulativeSizes = 0;
            for (int compressionSize = 0; compressionSize < sizeDistribution.length; ++compressionSize) {
                int numberOfValuesAtCompression = sizeDistribution[compressionSize];
                System.out.printf(Locale.ENGLISH, "  %17s  |  %16s  |  %9.2f%%  |  %15.2f%%%n", compressionSize, numberOfValuesAtCompression, 100.0 * (double)numberOfValuesAtCompression / (double)original.length, 100.0 * (double)(cumulativeSizes += numberOfValuesAtCompression) / (double)original.length);
            }
            System.out.println();
            System.out.println();
            System.out.println("   Compression type  |  Compression size  |  Number of Values  |  Percentile  |  Total Percentile");
            int cumulativeTypes = 0;
            for (int compressionType = 0; compressionType < typeDistribution.length; ++compressionType) {
                int numberOfValuesWithCompressionType = typeDistribution[compressionType];
                cumulativeTypes += numberOfValuesWithCompressionType;
                if (numberOfValuesWithCompressionType <= 0) continue;
                System.out.printf(Locale.ENGLISH, "  %17s  |  %16s  |  %16s  |  %9.2f%%  |  %15.2f%%%n", this.compressor.describeCompression(compressionType), "", numberOfValuesWithCompressionType, 100.0 * (double)numberOfValuesWithCompressionType / (double)original.length, 100.0 * (double)cumulativeTypes / (double)original.length);
                int[] sizes = sizePerType[compressionType];
                int allSizes = 0;
                for (int compressionSize = 0; compressionSize < sizes.length; ++compressionSize) {
                    int numberOfValuesAtCompression = sizes[compressionSize];
                    allSizes += numberOfValuesAtCompression;
                    if (numberOfValuesAtCompression <= 0) continue;
                    System.out.printf(Locale.ENGLISH, "  %17s  |  %16s  |  %16s  |  %7.2f%% \u00a6  |  %13.2f%% \u00a6%n", "", compressionSize, numberOfValuesAtCompression, 100.0 * (double)numberOfValuesAtCompression / (double)numberOfValuesWithCompressionType, 100.0 * (double)allSizes / (double)numberOfValuesWithCompressionType);
                }
            }
        }
    }

    private static final class Check
    implements Executable {
        private final DoubleCodec.CompressionInfo info;

        private Check(DoubleCodec.CompressionInfo info) {
            this.info = info;
        }

        boolean valid() {
            return Double.compare(this.info.input(), this.info.decompressed()) == 0;
        }

        Executable asExecutable() {
            return this;
        }

        public void execute() {
            double expected = this.info.input();
            double actual = this.info.decompressed();
            Assertions.assertEquals((double)expected, (double)actual, (double)1.0E-6, () -> String.format(Locale.ENGLISH, "Expected value compressed as [%3$s]%4$s to be equal to input [%2$f] (%2$A) but it actually was decompressed as [%1$f] (%1$A)", actual, expected, this.info.compressionDescription(), Arrays.toString(this.info.compressed())));
        }
    }
}

