/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.graphalgo.core.write;

import java.util.Collection;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.IntFunction;
import java.util.function.LongUnaryOperator;
import org.neo4j.graphalgo.api.Graph;
import org.neo4j.graphalgo.api.HugeGraph;
import org.neo4j.graphalgo.api.HugeIdMapping;
import org.neo4j.graphalgo.api.IdMapping;
import org.neo4j.graphalgo.core.utils.LazyBatchCollection;
import org.neo4j.graphalgo.core.utils.ParallelUtil;
import org.neo4j.graphalgo.core.utils.Pools;
import org.neo4j.graphalgo.core.utils.ProgressLogger;
import org.neo4j.graphalgo.core.utils.ProgressLoggerAdapter;
import org.neo4j.graphalgo.core.utils.StatementApi;
import org.neo4j.graphalgo.core.utils.TerminationFlag;
import org.neo4j.graphalgo.core.write.PropertyTranslator;
import org.neo4j.helpers.Exceptions;
import org.neo4j.kernel.api.DataWriteOperations;
import org.neo4j.kernel.api.exceptions.KernelException;
import org.neo4j.kernel.api.properties.DefinedProperty;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.logging.Log;

public final class Exporter
extends StatementApi {
    private static final long MIN_BATCH_SIZE = 10000L;
    private static final long MAX_BATCH_SIZE = 100000L;
    public static final String TASK_EXPORT = "EXPORT";
    private final TerminationFlag terminationFlag;
    private final ExecutorService executorService;
    private final ProgressLogger progressLogger;
    private final int concurrency;
    private final long nodeCount;
    private final LongUnaryOperator toOriginalId;

    public static Builder of(GraphDatabaseAPI db, Graph graph) {
        if (graph instanceof HugeGraph) {
            return new Builder(db, (HugeIdMapping)((Object)graph));
        }
        return new Builder(db, graph);
    }

    public static Builder of(IdMapping mapping, GraphDatabaseAPI db) {
        return new Builder(db, mapping);
    }

    private Exporter(GraphDatabaseAPI db, long nodeCount, LongUnaryOperator toOriginalId, TerminationFlag terminationFlag, ProgressLogger log, int concurrency, ExecutorService executorService) {
        super(db);
        this.nodeCount = nodeCount;
        this.toOriginalId = toOriginalId;
        this.terminationFlag = terminationFlag;
        this.progressLogger = log;
        this.concurrency = concurrency;
        this.executorService = executorService;
    }

    public <T> void write(String property, T data, PropertyTranslator<T> translator) {
        int propertyId = this.getOrCreatePropertyId(property);
        if (propertyId == -1) {
            throw new IllegalStateException("no write property id is set");
        }
        if (ParallelUtil.canRunInParallel(this.executorService)) {
            this.writeParallel(propertyId, data, translator);
        } else {
            this.writeSequential(propertyId, data, translator);
        }
    }

    public <T, U> void write(String property1, T data1, PropertyTranslator<T> translator1, String property2, U data2, PropertyTranslator<U> translator2) {
        int propertyId1 = this.getOrCreatePropertyId(property1);
        if (propertyId1 == -1) {
            throw new IllegalStateException("no write property id is set");
        }
        int propertyId2 = this.getOrCreatePropertyId(property2);
        if (propertyId2 == -1) {
            throw new IllegalStateException("no write property id is set");
        }
        if (ParallelUtil.canRunInParallel(this.executorService)) {
            this.writeParallel(propertyId1, data1, translator1, propertyId2, data2, translator2);
        } else {
            this.writeSequential(propertyId1, data1, translator1, propertyId2, data2, translator2);
        }
    }

    public void write(String property, IntFunction<WriteConsumer> createWriter) {
        int propertyId = this.getOrCreatePropertyId(property);
        if (propertyId == -1) {
            throw new IllegalStateException("no write property id is set");
        }
        WriteConsumer writer = createWriter.apply(propertyId);
        if (ParallelUtil.canRunInParallel(this.executorService)) {
            this.writeParallel(writer);
        } else {
            this.writeSequential(writer);
        }
    }

    public void writeRelationships(String property, WriteConsumer writer) {
        int propertyId = this.getOrCreateRelationshipId(property);
        if (propertyId == -1) {
            throw new IllegalStateException("no write property id is set");
        }
        try {
            this.acceptInTransaction(stmt -> {
                DataWriteOperations ops = stmt.dataWriteOperations();
                writer.accept(ops, propertyId);
            });
        }
        catch (KernelException e) {
            throw Exceptions.launderedException((Throwable)e);
        }
    }

    private <T> void writeSequential(int propertyId, T data, PropertyTranslator<T> translator) {
        this.writeSequential((ops, offset) -> this.doWrite(propertyId, data, translator, ops, offset));
    }

    private <T, U> void writeSequential(int propertyId1, T data1, PropertyTranslator<T> translator1, int propertyId2, U data2, PropertyTranslator<U> translator2) {
        this.writeSequential((ops, offset) -> this.doWrite(propertyId1, data1, translator1, propertyId2, data2, translator2, ops, offset));
    }

    private <T> void writeParallel(int propertyId, T data, PropertyTranslator<T> translator) {
        this.writeParallel((ops, offset) -> this.doWrite(propertyId, data, translator, ops, offset));
    }

    private <T, U> void writeParallel(int propertyId1, T data1, PropertyTranslator<T> translator1, int propertyId2, U data2, PropertyTranslator<U> translator2) {
        this.writeParallel((ops, offset) -> this.doWrite(propertyId1, data1, translator1, propertyId2, data2, translator2, ops, offset));
    }

    private void writeSequential(WriteConsumer writer) {
        try {
            this.acceptInTransaction(stmt -> {
                long progress = 0L;
                DataWriteOperations ops = stmt.dataWriteOperations();
                for (long i = 0L; i < this.nodeCount; ++i) {
                    writer.accept(ops, i);
                    this.progressLogger.logProgress((double)(++progress), this.nodeCount);
                }
            });
        }
        catch (KernelException e) {
            throw Exceptions.launderedException((Throwable)e);
        }
    }

    private void writeParallel(WriteConsumer writer) {
        long batchSize = Math.min(100000L, ParallelUtil.adjustBatchSize(this.nodeCount, this.concurrency, 10000L));
        AtomicLong progress = new AtomicLong(0L);
        Collection<Runnable> runnables = LazyBatchCollection.of(this.nodeCount, batchSize, (start, len) -> () -> {
            try {
                this.acceptInTransaction(stmt -> {
                    long end = start + len;
                    DataWriteOperations ops = stmt.dataWriteOperations();
                    for (long j = start; j < end; ++j) {
                        writer.accept(ops, j);
                        this.progressLogger.logProgress((double)progress.incrementAndGet(), this.nodeCount);
                    }
                });
            }
            catch (KernelException e) {
                throw Exceptions.launderedException((Throwable)e);
            }
        });
        ParallelUtil.runWithConcurrency(this.concurrency, runnables, Integer.MAX_VALUE, 10L, TimeUnit.MICROSECONDS, this.terminationFlag, this.executorService);
    }

    private <T> void doWrite(int propertyId, T data, PropertyTranslator<T> trans, DataWriteOperations ops, long nodeId) throws KernelException {
        DefinedProperty prop = trans.toProperty(propertyId, data, nodeId);
        if (prop != null) {
            ops.nodeSetProperty(this.toOriginalId.applyAsLong(nodeId), prop);
        }
    }

    private <T, U> void doWrite(int propertyId1, T data1, PropertyTranslator<T> translator1, int propertyId2, U data2, PropertyTranslator<U> translator2, DataWriteOperations ops, long nodeId) throws KernelException {
        DefinedProperty prop2;
        long originalNodeId = this.toOriginalId.applyAsLong(nodeId);
        DefinedProperty prop1 = translator1.toProperty(propertyId1, data1, nodeId);
        if (prop1 != null) {
            ops.nodeSetProperty(originalNodeId, prop1);
        }
        if ((prop2 = translator2.toProperty(propertyId2, data2, nodeId)) != null) {
            ops.nodeSetProperty(originalNodeId, prop2);
        }
    }

    private int getOrCreatePropertyId(String propertyName) {
        try {
            return this.applyInTransaction(stmt -> stmt.tokenWriteOperations().propertyKeyGetOrCreateForName(propertyName));
        }
        catch (KernelException e) {
            throw new RuntimeException(e);
        }
    }

    private int getOrCreateRelationshipId(String propertyName) {
        try {
            return this.applyInTransaction(stmt -> stmt.tokenWriteOperations().relationshipTypeGetOrCreateForName(propertyName));
        }
        catch (KernelException e) {
            throw new RuntimeException(e);
        }
    }

    public static interface WriteConsumer {
        public void accept(DataWriteOperations var1, long var2) throws KernelException;
    }

    public static final class Builder {
        private final GraphDatabaseAPI db;
        private final LongUnaryOperator toOriginalId;
        private final long nodeCount;
        private TerminationFlag terminationFlag;
        private ExecutorService executorService;
        private ProgressLoggerAdapter loggerAdapter;
        private int concurrency = Pools.DEFAULT_CONCURRENCY;

        private Builder(GraphDatabaseAPI db, IdMapping idMapping) {
            Objects.requireNonNull(idMapping);
            this.db = Objects.requireNonNull(db);
            this.nodeCount = idMapping.nodeCount();
            this.toOriginalId = n -> idMapping.toOriginalNodeId((int)n);
        }

        private Builder(GraphDatabaseAPI db, HugeIdMapping idMapping) {
            Objects.requireNonNull(idMapping);
            this.db = Objects.requireNonNull(db);
            this.nodeCount = idMapping.nodeCount();
            this.toOriginalId = idMapping::toOriginalNodeId;
        }

        public Builder withLog(Log log) {
            this.loggerAdapter = new ProgressLoggerAdapter(Objects.requireNonNull(log), Exporter.TASK_EXPORT);
            return this;
        }

        public Builder withLogInterval(long time, TimeUnit unit) {
            if (this.loggerAdapter == null) {
                throw new IllegalStateException("no logger set");
            }
            long logTime = unit.toMillis(time);
            if ((long)((int)logTime) != logTime) {
                throw new IllegalArgumentException("timespan too large");
            }
            this.loggerAdapter.withLogIntervalMillis((int)logTime);
            return this;
        }

        public Builder parallel(ExecutorService es, int concurrency, TerminationFlag flag) {
            this.executorService = es;
            this.concurrency = concurrency;
            this.terminationFlag = flag;
            return this;
        }

        public Exporter build() {
            ProgressLogger progressLogger = this.loggerAdapter == null ? ProgressLogger.NULL_LOGGER : this.loggerAdapter;
            TerminationFlag flag = this.terminationFlag == null ? TerminationFlag.RUNNING_TRUE : this.terminationFlag;
            return new Exporter(this.db, this.nodeCount, this.toOriginalId, flag, progressLogger, this.concurrency, this.executorService);
        }
    }
}

