/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [https://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.neo4j.kernel.lifecycle;

/**
 * Lifecycle interface for kernel components. Init is called first,
 * followed by start,
 * and then any number of stop-start sequences,
 * and finally stop and shutdown.
 *
 * As a stop-start cycle could be due to change of configuration, please perform anything that depends on config
 * in start().
 *
 * Implementations can throw any exception. Caller must handle this properly.
 *
 * The primary purpose of init in a component is to set up structure: instantiate dependent objects,
 * register handlers/listeners, etc.
 * Only in start should the component actually do anything with this structure.
 * Stop reverses whatever was done in start, and shutdown finally clears any set-up structure, if necessary.
 */
public interface Lifecycle {

    Lifecycle NO_OP = new Lifecycle() {

        @Override
        public void init() throws Exception {}

        @Override
        public void start() throws Exception {}

        @Override
        public void stop() throws Exception {}

        @Override
        public void shutdown() throws Exception {}
    };

    void init() throws Exception;

    void start() throws Exception;

    void stop() throws Exception;

    void shutdown() throws Exception;
}
