/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.configuration;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.neo4j.configuration.BootloaderSettings;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.SettingMigrator;
import org.neo4j.configuration.connectors.BoltConnector;
import org.neo4j.configuration.connectors.BoltConnectorInternalSettings;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.logging.InternalLog;

public final class SettingMigrators {
    private SettingMigrators() {
    }

    public static void migrateGroupSettingPrefixChange(Map<String, String> values, InternalLog log, String oldGroupSettingPrefix, String newGroupSettingPrefix) {
        List<String> toUpdate = values.keySet().stream().filter(s -> s.startsWith(oldGroupSettingPrefix) && !s.equals(oldGroupSettingPrefix)).toList();
        for (String oldSetting : toUpdate) {
            String newSettingName = oldSetting.replace(oldGroupSettingPrefix, newGroupSettingPrefix);
            log.warn("Use of deprecated setting '%s'. It is replaced by '%s'.", new Object[]{oldSetting, newSettingName});
            String oldValue = values.remove(oldSetting);
            values.put(newSettingName, oldValue);
        }
    }

    public static void migrateSettingNameChange(Map<String, String> values, InternalLog log, Collection<String> prefixes, String oldSettingSuffix, Setting<?> newSetting) {
        prefixes.stream().map(p -> p + oldSettingSuffix).forEach(key -> SettingMigrators.migrateSettingNameChange(values, log, key, newSetting));
    }

    public static void migrateSettingNameChange(Map<String, String> values, InternalLog log, String oldSetting, Setting<?> newSetting) {
        SettingMigrators.migrateSettingNameChange(values, log, oldSetting, newSetting.name());
    }

    public static void migrateSettingNameChange(Map<String, String> values, InternalLog log, String oldSettingName, String newSettingName) {
        String value = values.remove(oldSettingName);
        if (StringUtils.isNotBlank((CharSequence)value)) {
            log.warn("Use of deprecated setting '%s'. It is replaced by '%s'.", new Object[]{oldSettingName, newSettingName});
            values.putIfAbsent(newSettingName, value);
        }
    }

    public static void migrateSettingRemoval(Map<String, String> values, InternalLog log, String name, String additionalDescription) {
        if (values.containsKey(name)) {
            log.warn("Setting '%s' is removed. %s.", new Object[]{name, additionalDescription});
            values.remove(name);
        }
    }

    public static void migrateSettingRemoval(Map<String, String> values, InternalLog log, Collection<String> prefixes, String oldSettingSuffix, String explanation) {
        prefixes.stream().map(p -> p + oldSettingSuffix).forEach(key -> SettingMigrators.migrateSettingRemoval(values, log, key, explanation));
    }

    public static class Neo4j5_0SettingMigrator
    implements SettingMigrator {
        private static final String OLD_PREFIX = "dbms.connector";
        private static final Pattern SUPPORTED_CONNECTOR_PATTERN = Pattern.compile("(.+)\\.(bolt|http|https)\\.(.+)");
        private static final List<String> REMOVED_SETTINGS = List.of("dbms.allow_single_automatic_upgrade", "dbms.allow_upgrade", "dbms.clustering.enable", "dbms.record_format", "dbms.backup.incremental.strategy", "dbms.directories.tx_log", "dbms.index.default_schema_provider", "dbms.index_searcher_cache_size", "dbms.logs.debug.rotation.delay", "dbms.logs.user.rotation.delay", "dbms.memory.pagecache.swapper", "dbms.routing.driver.api", "dbms.security.ldap.authentication.use_samaccountname", "dbms.security.procedures.default_allowed", "dbms.security.procedures.roles", "dbms.security.property_level.blacklist", "dbms.security.property_level.enabled", "dbms.logs.debug.format", "dbms.logs.debug.level", "dbms.logs.debug.path", "dbms.logs.debug.rotation.keep_number", "dbms.logs.debug.rotation.size", "dbms.logs.default_format", "dbms.logs.http.format", "dbms.logs.http.path", "dbms.logs.http.rotation.keep_number", "dbms.logs.http.rotation.size", "dbms.logs.query.format", "dbms.logs.query.path", "dbms.logs.query.rotation.keep_number", "dbms.logs.query.rotation.size", "dbms.logs.security.format", "dbms.logs.security.level", "dbms.logs.security.path", "dbms.logs.security.rotation.delay", "dbms.logs.security.rotation.keep_number", "dbms.logs.security.rotation.size", "dbms.logs.user.format", "dbms.logs.user.path", "dbms.logs.user.rotation.keep_number", "dbms.logs.user.rotation.size", "dbms.logs.user.stdout_enabled", "unsupported.cypher.parser", "unsupported.dbms.block_remote_alias", "unsupported.dbms.large_cluster.enable", "unsupported.dbms.memory.managed_network_buffers", "unsupported.dbms.memory.pagecache.warmup.legacy_profile_loader", "unsupported.dbms.lucene.ephemeral", "unsupported.dbms.recovery.ignore_store_id_validation", "unsupported.dbms.storage_engine", "unsupported.dbms.tokenscan.log.enabled", "unsupported.dbms.tokenscan.log.prune_threshold", "unsupported.dbms.tokenscan.log.rotation_threshold", "unsupported.dbms.topology_graph.enable", "unsupported.dbms.topology_graph_updater.enable", "unsupported.dbms.uris.rest", "unsupported.dbms.reserved.page.header.bytes", "unsupported.dbms.index.default_fulltext_provider", "unsupported.dbms.lock_manager", "internal.dbms.lock_manager", "cypher.default_language_version", "dbms.logs.query.allocation_logging_enabled", "dbms.track_query_allocation", "dbms.logs.query.page_logging_enabled", "dbms.logs.query.runtime_logging_enabled", "dbms.logs.query.time_logging_enabled", "dbms.logs.query.transaction_id.enabled", "dbms.logs.query.parameter_full_entities", "fabric.database.name", "fabric.driver.api", "fabric.driver.connection.connect_timeout", "fabric.driver.connection.max_lifetime", "fabric.driver.connection.pool.acquisition_timeout", "fabric.driver.connection.pool.idle_test", "fabric.driver.connection.pool.max_size", "fabric.driver.event_loop_count", "fabric.driver.idle_check_interval", "fabric.driver.logging.leaked_sessions", "fabric.driver.logging.level", "fabric.driver.timeout", "fabric.enabled_by_default", "fabric.graph.mega.database", "fabric.graph.mega.driver.api", "fabric.graph.mega.driver.connection.connect_timeout", "fabric.graph.mega.driver.connection.max_lifetime", "fabric.graph.mega.driver.connection.pool.acquisition_timeout", "fabric.graph.mega.driver.connection.pool.idle_test", "fabric.graph.mega.driver.connection.pool.max_size", "fabric.graph.mega.driver.logging.leaked_sessions", "fabric.graph.mega.driver.logging.level", "fabric.graph.mega.driver.ssl_enabled", "fabric.graph.mega.name", "fabric.graph.mega.uri", "fabric.routing.servers", "fabric.routing.ttl", "fabric.stream.batch_size", "fabric.stream.buffer.low_watermark", "fabric.stream.buffer.size", "fabric.stream.concurrency", "internal.cypher.planning_point_indexes_enabled", "internal.cypher.planning_text_indexes_enabled", "internal.cypher.planning_range_indexes_enabled", "unsupported.cypher.planning_point_indexes_enabled", "unsupported.cypher.planning_range_indexes_enabled", "unsupported.cypher.planning_text_indexes_enabled", "unsupported.cypher.splitting_top_behavior");
        private static final Collection<Mapping> LEGACY_UNSUPPORTED_SETTINGS_MAPPING = List.of(new Mapping("dbms.capabilities.blocked", "internal.dbms.capabilities.blocked"), new Mapping("dbms.connector.bolt.tcp_keep_alive", "internal.server.bolt.tcp_keep_alive"), new Mapping("dbms.init_file", "internal.dbms.init_file"), new Mapping("dbms.log_inconsistent_data_deletion", "internal.dbms.log_inconsistent_data_deletion"), new Mapping("dbms.routing.driver.event_loop_count", "internal.dbms.routing.driver.event_loop_count"), new Mapping("dbms.routing.driver.idle_check_interval", "internal.dbms.routing.driver.idle_check_interval"), new Mapping("dbms.routing.driver.logging.leaked_sessions", "internal.dbms.routing.driver.logging.leaked_sessions"), new Mapping("dbms.routing.driver.timeout", "internal.dbms.routing.driver.timeout"), new Mapping("unsupported.bootloader.auto.configuration", "internal.bootloader.auto.configuration"), new Mapping("unsupported.bootloader.auto.configuration.codecache.max", "internal.bootloader.auto.configuration.codecache.max"), new Mapping("unsupported.bootloader.auto.configuration.codecache.min", "internal.bootloader.auto.configuration.codecache.min"), new Mapping("unsupported.bootloader.auto.configuration.dataset.file", "internal.bootloader.auto.configuration.dataset.file"), new Mapping("unsupported.bootloader.auto.configuration.execution_plan_cache.max", "internal.bootloader.auto.configuration.execution_plan_cache.max"), new Mapping("unsupported.bootloader.auto.configuration.execution_plan_cache.min", "internal.bootloader.auto.configuration.execution_plan_cache.min"), new Mapping("unsupported.bootloader.auto.configuration.heap.max", "internal.bootloader.auto.configuration.heap.max"), new Mapping("unsupported.bootloader.auto.configuration.heap.min", "internal.bootloader.auto.configuration.heap.min"), new Mapping("unsupported.bootloader.auto.configuration.memory.limit", "internal.bootloader.auto.configuration.memory.limit"), new Mapping("unsupported.bootloader.auto.configuration.pagecache.max", "internal.bootloader.auto.configuration.pagecache.max"), new Mapping("unsupported.bootloader.auto.configuration.pagecache.min", "internal.bootloader.auto.configuration.pagecache.min"), new Mapping("unsupported.bootloader.auto.configuration.query_cache.max", "internal.bootloader.auto.configuration.query_cache.max"), new Mapping("unsupported.bootloader.auto.configuration.query_cache.min", "internal.bootloader.auto.configuration.query_cache.min"), new Mapping("unsupported.bootloader.auto.configuration.search.radius", "internal.bootloader.auto.configuration.search.radius"), new Mapping("unsupported.bootloader.auto.configuration.tx_limit.max", "internal.bootloader.auto.configuration.tx_limit.max"), new Mapping("unsupported.bootloader.auto.configuration.tx_limit.min", "internal.bootloader.auto.configuration.tx_limit.min"), new Mapping("unsupported.consistency_checker.fail_fast_threshold", "internal.consistency_checker.fail_fast_threshold"), new Mapping("unsupported.consistency_checker.memory_limit_factor", "internal.consistency_checker.memory_limit_factor"), new Mapping("unsupported.cypher.compiler_tracing", "internal.cypher.compiler_tracing"), new Mapping("unsupported.cypher.enable_extra_semantic_features", "internal.cypher.enable_extra_semantic_features"), new Mapping("unsupported.cypher.enable_runtime_monitors", "internal.cypher.enable_runtime_monitors"), new Mapping("unsupported.cypher.expression_engine", "internal.cypher.expression_engine"), new Mapping("unsupported.cypher.expression_recompilation_limit", "internal.cypher.expression_recompilation_limit"), new Mapping("unsupported.cypher.idp_solver_duration_threshold", "internal.cypher.idp_solver_duration_threshold"), new Mapping("unsupported.cypher.idp_solver_table_threshold", "internal.cypher.idp_solver_table_threshold"), new Mapping("unsupported.cypher.non_indexed_label_warning_threshold", "internal.cypher.non_indexed_label_warning_threshold"), new Mapping("unsupported.cypher.number_of_workers", "internal.cypher.number_of_workers"), new Mapping("unsupported.cypher.pipelined.batch_size_big", "internal.cypher.pipelined.batch_size_big"), new Mapping("unsupported.cypher.pipelined.batch_size_small", "internal.cypher.pipelined.batch_size_small"), new Mapping("unsupported.cypher.pipelined.enable_runtime_trace", "internal.cypher.pipelined.enable_runtime_trace"), new Mapping("unsupported.cypher.pipelined.operator_engine", "internal.cypher.pipelined.operator_engine"), new Mapping("unsupported.cypher.pipelined.operator_fusion_over_pipeline_limit", "internal.cypher.pipelined.operator_fusion_over_pipeline_limit"), new Mapping("unsupported.cypher.pipelined.runtime_trace_path", "internal.cypher.pipelined.runtime_trace_path"), new Mapping("unsupported.cypher.pipelined_interpreted_pipes_fallback", "internal.cypher.pipelined_interpreted_pipes_fallback"), new Mapping("unsupported.cypher.replan_algorithm", "internal.cypher.replan_algorithm"), new Mapping("unsupported.cypher.runtime", "internal.cypher.runtime"), new Mapping("unsupported.cypher.statistics_divergence_target", "internal.cypher.statistics_divergence_target"), new Mapping("unsupported.cypher.target_replan_interval", "internal.cypher.target_replan_interval"), new Mapping("unsupported.cypher.var_expand_relationship_id_set_threshold", "internal.cypher.var_expand_relationship_id_set_threshold"), new Mapping("unsupported.datacollector.max_query_text_size", "internal.datacollector.max_query_text_size"), new Mapping("unsupported.datacollector.max_recent_query_count", "internal.datacollector.max_recent_query_count"), new Mapping("unsupported.dbms.block_alter_database", "internal.dbms.block_alter_database"), new Mapping("unsupported.dbms.block_create_drop_database", "internal.dbms.block_create_drop_database"), new Mapping("unsupported.dbms.block_size.array_properties", "internal.dbms.block_size.array_properties"), new Mapping("unsupported.dbms.block_size.labels", "internal.dbms.block_size.labels"), new Mapping("unsupported.dbms.block_size.strings", "internal.dbms.block_size.strings"), new Mapping("unsupported.dbms.block_start_stop_database", "internal.dbms.block_start_stop_database"), new Mapping("unsupported.dbms.bolt.inbound_message_throttle.high_watermark", "internal.dbms.bolt.inbound_message_throttle.high_watermark"), new Mapping("unsupported.dbms.bolt.inbound_message_throttle.low_watermark", "internal.dbms.bolt.inbound_message_throttle.low_watermark"), new Mapping("unsupported.dbms.bolt.netty_message_merge_cumulator", "internal.dbms.bolt.netty_message_merge_cumulator"), new Mapping("unsupported.dbms.bolt.netty_server_shutdown_quiet_period", "internal.dbms.bolt.netty_server_shutdown_quiet_period"), new Mapping("unsupported.dbms.bolt.netty_server_shutdown_timeout", "internal.dbms.bolt.netty_server_shutdown_timeout"), new Mapping("unsupported.dbms.bolt.netty_server_use_epoll", "internal.dbms.bolt.netty_server_use_native_transport"), new Mapping("unsupported.dbms.bolt.outbound_buffer_throttle", "internal.dbms.bolt.outbound_buffer_throttle"), new Mapping("unsupported.dbms.bolt.outbound_buffer_throttle.high_watermark", "internal.dbms.bolt.outbound_buffer_throttle.high_watermark"), new Mapping("unsupported.dbms.bolt.outbound_buffer_throttle.low_watermark", "internal.dbms.bolt.outbound_buffer_throttle.low_watermark"), new Mapping("unsupported.dbms.bolt.outbound_buffer_throttle.max_duration", "internal.dbms.bolt.outbound_buffer_throttle.max_duration"), new Mapping("unsupported.dbms.checkpoint_log.rotation.keep.files", "internal.db.checkpoint_log.rotation.keep.files"), new Mapping("unsupported.dbms.checkpoint_log.rotation.size", "internal.db.checkpoint_log.rotation.size"), new Mapping("unsupported.dbms.config.command_evaluation_timeout", "internal.dbms.config.command_evaluation_timeout"), new Mapping("unsupported.dbms.counts_store_rotation_timeout", "internal.dbms.counts_store_rotation_timeout"), new Mapping("unsupported.dbms.cypher_ip_blocklist", "internal.dbms.cypher_ip_blocklist"), new Mapping("unsupported.dbms.debug.page_cache_tracer_speed_reporting_threshold", "internal.dbms.debug.page_cache_tracer_speed_reporting_threshold"), new Mapping("unsupported.dbms.debug.print_page_buffer_allocation_trace", "internal.dbms.debug.print_page_buffer_allocation_trace"), new Mapping("unsupported.dbms.debug.trace_cursors", "internal.dbms.debug.trace_cursors"), new Mapping("unsupported.dbms.debug.trace_tx_statement", "internal.dbms.debug.trace_tx_statement"), new Mapping("unsupported.dbms.debug.track_cursor_close", "internal.dbms.debug.track_cursor_close"), new Mapping("unsupported.dbms.debug.track_tx_statement_close", "internal.dbms.debug.track_tx_statement_close"), new Mapping("unsupported.dbms.directories.auth", "internal.server.directories.auth"), new Mapping("unsupported.dbms.directories.databases.root", "internal.server.directories.databases.root"), new Mapping("unsupported.dbms.directories.pid_file", "internal.server.directories.pid_file"), new Mapping("unsupported.dbms.directories.scripts", "internal.server.directories.scripts"), new Mapping("unsupported.dbms.directories.windows_tools", "internal.server.directories.windows_tools"), new Mapping("unsupported.dbms.discoverable_bolt_address", "internal.dbms.discoverable_bolt_address"), new Mapping("unsupported.dbms.discoverable_bolt_routing_address", "internal.dbms.discoverable_bolt_routing_address"), new Mapping("unsupported.dbms.dump_diagnostics", "internal.dbms.dump_diagnostics"), new Mapping("unsupported.dbms.enable_transaction_heap_allocation_tracking", "internal.dbms.enable_transaction_heap_allocation_tracking"), new Mapping("unsupported.dbms.executiontime_limit.time", "internal.dbms.executiontime_limit.time"), new Mapping("unsupported.dbms.extra_lock_verification", "internal.dbms.extra_lock_verification"), new Mapping("unsupported.dbms.force_small_id_cache", "internal.dbms.force_small_id_cache"), new Mapping("unsupported.dbms.http_paths_blacklist", "internal.dbms.http_paths_blacklist"), new Mapping("unsupported.dbms.id_buffering.offload_to_disk", "internal.dbms.id_buffering.offload_to_disk"), new Mapping("unsupported.dbms.idgenerator.log.enabled", "internal.dbms.idgenerator.log.enabled"), new Mapping("unsupported.dbms.idgenerator.log.prune_threshold", "internal.dbms.idgenerator.log.prune_threshold"), new Mapping("unsupported.dbms.idgenerator.log.rotation_threshold", "internal.dbms.idgenerator.log.rotation_threshold"), new Mapping("unsupported.dbms.include_dev_record_format_versions", "internal.dbms.include_dev_format_versions"), new Mapping("unsupported.dbms.index.archive_failed", "internal.dbms.index.archive_failed"), new Mapping("unsupported.dbms.index.default_fulltext_provider", "internal.dbms.index.default_fulltext_provider"), new Mapping("unsupported.dbms.index.lucene.merge_factor", "internal.dbms.index.lucene.merge_factor"), new Mapping("unsupported.dbms.index.lucene.min_merge", "internal.dbms.index.lucene.min_merge"), new Mapping("unsupported.dbms.index.lucene.nocfs.ratio", "internal.dbms.index.lucene.nocfs.ratio"), new Mapping("unsupported.dbms.index.lucene.population_max_buffered_docs", "internal.dbms.index.lucene.population_max_buffered_docs"), new Mapping("unsupported.dbms.index.lucene.population_ram_buffer_size", "internal.dbms.index.lucene.population_ram_buffer_size"), new Mapping("unsupported.dbms.index.lucene.population_serial_merge_scheduler", "internal.dbms.index.lucene.population_serial_merge_scheduler"), new Mapping("unsupported.dbms.index.lucene.standard_ram_buffer_size", "internal.dbms.index.lucene.standard_ram_buffer_size"), new Mapping("unsupported.dbms.index.lucene.writer_max_buffered_docs", "internal.dbms.index.lucene.writer_max_buffered_docs"), new Mapping("unsupported.dbms.index.population_batch_max_byte_size", "internal.dbms.index.population_batch_max_byte_size"), new Mapping("unsupported.dbms.index.population_print_debug", "internal.dbms.index.population_print_debug"), new Mapping("unsupported.dbms.index.population_queue_threshold", "internal.dbms.index.population_queue_threshold"), new Mapping("unsupported.dbms.index.populator_block_size", "internal.dbms.index.populator_block_size"), new Mapping("unsupported.dbms.index.populator_merge_factor", "internal.dbms.index.populator_merge_factor"), new Mapping("unsupported.dbms.index.sampling.async_recovery", "internal.dbms.index.sampling.async_recovery"), new Mapping("unsupported.dbms.index.sampling.async_recovery_wait", "internal.dbms.index.sampling.async_recovery_wait"), new Mapping("unsupported.dbms.index.sampling.log_recovered_samples", "internal.dbms.index.sampling.log_recovered_samples"), new Mapping("unsupported.dbms.index.skip_default_indexes_on_creation", "internal.dbms.index.skip_default_indexes_on_creation"), new Mapping("unsupported.dbms.index.spatial.curve.bottom_threshold", "internal.dbms.index.spatial.curve.bottom_threshold"), new Mapping("unsupported.dbms.index.spatial.curve.extra_levels", "internal.dbms.index.spatial.curve.extra_levels"), new Mapping("unsupported.dbms.index.spatial.curve.top_threshold", "internal.dbms.index.spatial.curve.top_threshold"), new Mapping("unsupported.dbms.index_population.parallelism", "internal.dbms.index_population.parallelism"), new Mapping("unsupported.dbms.index_population.workers", "internal.dbms.index_population.workers"), new Mapping("unsupported.dbms.index_sampling.parallelism", "internal.dbms.index_sampling.parallelism"), new Mapping("unsupported.dbms.initial_transaction_heap_grab_size", "internal.dbms.initial_transaction_heap_grab_size"), new Mapping("unsupported.dbms.io.controller.consider.external.enabled", "internal.dbms.io.controller.consider.external.enabled"), new Mapping("unsupported.dbms.kernel_id", "internal.dbms.kernel_id"), new Mapping("unsupported.dbms.lock_manager.verbose_deadlocks", "internal.dbms.lock_manager.verbose_deadlocks"), new Mapping("unsupported.dbms.logs.query.heap_dump_enabled", "internal.dbms.logs.query.heap_dump_enabled"), new Mapping("unsupported.dbms.loopback_delete", "internal.dbms.loopback_delete"), new Mapping("unsupported.dbms.loopback_enabled", "internal.dbms.loopback_enabled"), new Mapping("unsupported.dbms.loopback_file", "internal.dbms.loopback_file"), new Mapping("unsupported.dbms.lucene.max_partition_size", "internal.dbms.lucene.max_partition_size"), new Mapping("unsupported.dbms.max_http_request_header_size", "internal.dbms.max_http_request_header_size"), new Mapping("unsupported.dbms.max_http_response_header_size", "internal.dbms.max_http_response_header_size"), new Mapping("unsupported.dbms.memory.counts_store_max_cached_entries", "internal.dbms.memory.counts_store_max_cached_entries"), new Mapping("unsupported.dbms.memory.managed_network_buffers", "internal.dbms.memory.managed_network_buffers"), new Mapping("unsupported.dbms.multiversioned.store", "internal.dbms.multiversioned.store"), new Mapping("unsupported.dbms.page.file.tracer", "internal.dbms.page.file.tracer"), new Mapping("unsupported.dbms.parallel_index_updates_apply", "internal.dbms.parallel_index_updates_apply"), new Mapping("unsupported.dbms.query.snapshot", "internal.dbms.query.snapshot"), new Mapping("unsupported.dbms.query.snapshot.retries", "internal.dbms.query.snapshot.retries"), new Mapping("unsupported.dbms.query_execution_plan_cache_size", "internal.dbms.query_execution_plan_cache_size"), new Mapping("unsupported.dbms.readonly.failover", "internal.dbms.readonly.failover"), new Mapping("unsupported.dbms.recovery.enable_parallelism", "internal.dbms.recovery.enable_parallelism"), new Mapping("unsupported.dbms.report_configuration", "internal.dbms.report_configuration"), new Mapping("unsupported.dbms.security.ldap.authorization.connection_pooling", "internal.dbms.security.ldap.authorization.connection_pooling"), new Mapping("unsupported.dbms.select_specfic_record_format", "internal.dbms.select_specfic_record_format"), new Mapping("unsupported.dbms.ssl.system.ignore_dot_files", "internal.dbms.ssl.system.ignore_dot_files"), new Mapping("unsupported.dbms.storage.consistency_check_on_apply", "internal.dbms.storage.consistency_check_on_apply"), new Mapping("unsupported.dbms.strictly_prioritize_id_freelist", "internal.dbms.strictly_prioritize_id_freelist"), new Mapping("unsupported.dbms.tracer", "internal.dbms.tracer"), new Mapping("unsupported.dbms.transaction_start_timeout", "internal.dbms.transaction_start_timeout"), new Mapping("unsupported.dbms.tx.logs.dedicated.appender", "internal.dbms.tx.logs.dedicated.appender"), new Mapping("unsupported.dbms.tx_log.fail_on_corrupted_log_files", "internal.dbms.tx_log.fail_on_corrupted_log_files"), new Mapping("unsupported.dbms.tx_log.presketch", "internal.dbms.tx_log.presketch"), new Mapping("unsupported.dbms.upgrade_restriction_enabled", "internal.dbms.upgrade_restriction_enabled"), new Mapping("unsupported.dbms.uris.browser", "internal.dbms.uris.browser"), new Mapping("unsupported.dbms.uris.db", "internal.dbms.uris.db"), new Mapping("unsupported.dbms.uris.dbms", "internal.dbms.uris.dbms"), new Mapping("unsupported.dbms.uris.management", "internal.dbms.uris.management"), new Mapping("unsupported.dbms.use_old_token_index_location", "internal.dbms.use_old_token_index_location"), new Mapping("unsupported.dbms.wadl_generation_enabled", "internal.dbms.wadl_generation_enabled"), new Mapping("unsupported.metrics.cypher.cache.entries.enabled", "internal.metrics.cypher.cache.entries.enabled"), new Mapping("unsupported.tools.batch_inserter.batch_size", "internal.tools.batch_inserter.batch_size"), new Mapping("unsupported.vm_pause_monitor.measurement_duration", "internal.vm_pause_monitor.measurement_duration"), new Mapping("unsupported.vm_pause_monitor.stall_alert_threshold", "internal.vm_pause_monitor.stall_alert_threshold"), new Mapping("dbms.config.strict_validation", GraphDatabaseSettings.strict_config_validation.name()));

        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            Neo4j5_0SettingMigrator.cleanupRemovedSettings(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateUnsupportedSettingsToInternal(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateDirectoriesChanges(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateConnectors(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateDatabaseMemorySettings(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateWhitelistSettings(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateWindowsServiceName(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateGroupSpatialSettings(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateCheckpointSettings(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateKeepAliveSetting(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateReadOnlySettings(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateDefaultDatabaseSetting(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateDatabaseMaxSize(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateCypherNamespace(values, defaultValues, log);
            Neo4j5_0SettingMigrator.migrateCypherQueryCacheSize(values, defaultValues, log);
            this.migrateTxStateAndLogsSettings(values, defaultValues, log);
            this.migrateTransactionAndTrackingSettings(values, defaultValues, log);
            this.migrateGcLogsSettings(values, defaultValues, log);
            this.migrateMaxProcessorToInternal(values, defaultValues, log);
            this.migratePageCacheWarmerSettings(values, defaultValues, log);
            this.migrateShutdownTimeoutAndFilePreallocation(values, defaultValues, log);
            this.migrateQueryCacheSize(values, defaultValues, log);
            this.migrateTransactionMemorySettings(values, defaultValues, log);
            this.migrateGroupAndRecoverySettings(values, defaultValues, log);
            this.migrateWatcherSetting(values, defaultValues, log);
            this.migrateCsvImportSetting(values, defaultValues, log);
            this.migrateLockAcquisitionSetting(values, defaultValues, log);
            this.migrateDefaultAddress(values, defaultValues, log);
            this.migrateQueryLoggingSettings(values, defaultValues, log);
            this.migrateJvmAdditional(values, defaultValues, log);
            this.migrateSamplingSettings(values, defaultValues, log);
            this.migratePageCacheAndMemorySettings(values, defaultValues, log);
        }

        private void migratePageCacheAndMemorySettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.size", GraphDatabaseSettings.pagecache_memory);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.scan.prefetchers", GraphDatabaseSettings.pagecache_scan_prefetch);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.flush.buffer.size_in_pages", GraphDatabaseSettings.pagecache_flush_buffer_size_in_pages);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.flush.buffer.enabled", GraphDatabaseSettings.pagecache_buffered_flush_enabled);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.directio", GraphDatabaseSettings.pagecache_direct_io);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.off_heap.max_size", GraphDatabaseSettings.tx_state_max_off_heap_memory);
            SettingMigrators.migrateSettingNameChange(values, log, "server.memory.off_heap.max_size", GraphDatabaseSettings.tx_state_max_off_heap_memory);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.off_heap.max_cacheable_block_size", GraphDatabaseSettings.tx_state_off_heap_max_cacheable_block_size);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.off_heap.block_cache_size", GraphDatabaseSettings.tx_state_off_heap_block_cache_size);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.heap.max_size", BootloaderSettings.max_heap_size);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.heap.initial_size", BootloaderSettings.initial_heap_size);
        }

        private void migrateSamplingSettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.index_sampling.background_enabled", GraphDatabaseSettings.index_background_sampling_enabled);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.index_sampling.sample_size_limit", GraphDatabaseSettings.index_sample_size_limit);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.index_sampling.update_percentage", GraphDatabaseSettings.index_sampling_update_percentage);
        }

        private void migrateJvmAdditional(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            String deprecatedValue;
            String oldSettingName = "dbms.jvm.additional";
            String value = values.get(oldSettingName);
            if (StringUtils.isNotBlank((CharSequence)value) && value.contains(deprecatedValue = "-XX:-UseBiasedLocking")) {
                String newValue = value.replaceAll(System.lineSeparator() + deprecatedValue, "");
                newValue = newValue.replaceAll(deprecatedValue + System.lineSeparator(), "");
                newValue = newValue.replaceAll(deprecatedValue, "");
                values.put(oldSettingName, newValue);
            }
            SettingMigrators.migrateSettingNameChange(values, log, oldSettingName, BootloaderSettings.additional_jvm);
        }

        private void migrateQueryLoggingSettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.query.transaction.threshold", GraphDatabaseSettings.log_queries_transaction_threshold);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.query.transaction.enabled", GraphDatabaseSettings.log_queries_transactions_level);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.query.threshold", GraphDatabaseSettings.log_queries_threshold);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.query.plan_description_enabled", GraphDatabaseSettings.log_queries_query_plan);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.query.parameter_logging_enabled", GraphDatabaseSettings.log_queries_parameter_logging_enabled);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.query.obfuscate_literals", GraphDatabaseSettings.log_queries_obfuscate_literals);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.query.max_parameter_length", GraphDatabaseSettings.query_log_max_parameter_length);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.query.enabled", GraphDatabaseSettings.log_queries);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.query.early_raw_logging_enabled", GraphDatabaseSettings.log_queries_early_raw_logging_enabled);
        }

        private void migrateDefaultAddress(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.default_listen_address", GraphDatabaseSettings.default_listen_address);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.default_advertised_address", GraphDatabaseSettings.default_advertised_address);
        }

        private void migrateLockAcquisitionSetting(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.lock.acquisition.timeout", GraphDatabaseSettings.lock_acquisition_timeout);
        }

        private void migrateCsvImportSetting(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.import.csv.buffer_size", GraphDatabaseSettings.csv_buffer_size);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.import.csv.legacy_quote_escaping", GraphDatabaseSettings.csv_legacy_quote_escaping);
        }

        private void migrateWatcherSetting(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.filewatcher.enabled", GraphDatabaseSettings.filewatcher_enabled);
        }

        private void migrateGroupAndRecoverySettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.relationship_grouping_threshold", GraphDatabaseSettings.dense_node_threshold);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.recovery.fail_on_missing_files", GraphDatabaseSettings.fail_on_missing_files);
        }

        private void migrateTransactionMemorySettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.transaction.database_max_size", GraphDatabaseSettings.memory_transaction_database_max_size);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.transaction.global_max_size", GraphDatabaseSettings.memory_transaction_global_max_size);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.transaction.max_size", GraphDatabaseSettings.memory_transaction_max_size);
        }

        private void migrateQueryCacheSize(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.query_cache_size", GraphDatabaseSettings.query_cache_size);
        }

        private void migrateShutdownTimeoutAndFilePreallocation(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.shutdown_transaction_end_timeout", GraphDatabaseSettings.shutdown_transaction_end_timeout);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.store.files.preallocate", GraphDatabaseSettings.preallocate_store_files);
        }

        private void migratePageCacheWarmerSettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.warmup.enable", GraphDatabaseSettings.pagecache_warmup_enabled);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.warmup.preload", GraphDatabaseSettings.pagecache_warmup_prefetch);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.warmup.preload.allowlist", GraphDatabaseSettings.pagecache_warmup_prefetch_allowlist);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.warmup.profile.interval", GraphDatabaseSettings.pagecache_warmup_profiling_interval);
        }

        private void migrateMaxProcessorToInternal(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.upgrade_max_processors", GraphDatabaseInternalSettings.upgrade_processors);
        }

        private void migrateGcLogsSettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.gc.enabled", BootloaderSettings.gc_logging_enabled);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.gc.options", BootloaderSettings.gc_logging_options);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.gc.rotation.keep_number", BootloaderSettings.gc_logging_rotation_keep_number);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.logs.gc.rotation.size", BootloaderSettings.gc_logging_rotation_size);
        }

        private void migrateTxStateAndLogsSettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_log.buffer.size", GraphDatabaseSettings.transaction_log_buffer_size);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_log.preallocate", GraphDatabaseSettings.preallocate_logical_logs);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_log.rotation.retention_policy", GraphDatabaseSettings.keep_logical_logs);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_log.rotation.size", GraphDatabaseSettings.logical_log_rotation_threshold);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_state.memory_allocation", GraphDatabaseSettings.tx_state_memory_allocation);
        }

        private void migrateTransactionAndTrackingSettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.track_query_cpu_time", GraphDatabaseSettings.track_query_cpu_time);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.transaction.bookmark_ready_timeout", GraphDatabaseSettings.bookmark_ready_timeout);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.transaction.concurrent.maximum", GraphDatabaseSettings.max_concurrent_transactions);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.transaction.monitor.check.interval", GraphDatabaseSettings.transaction_monitor_check_interval);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.transaction.sampling.percentage", GraphDatabaseSettings.transaction_sampling_percentage);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.transaction.timeout", GraphDatabaseSettings.transaction_timeout);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.transaction.tracing.level", GraphDatabaseSettings.transaction_tracing_level);
        }

        private static void migrateUnsupportedSettingsToInternal(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            LEGACY_UNSUPPORTED_SETTINGS_MAPPING.forEach(mapping -> SettingMigrators.migrateSettingNameChange(values, log, mapping.oldSettingName(), mapping.newSettingName()));
        }

        private static void cleanupRemovedSettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            REMOVED_SETTINGS.forEach(setting -> SettingMigrators.migrateSettingRemoval(values, log, setting, "It no longer has any effect"));
        }

        private static void migrateDirectoriesChanges(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.neo4j_home", GraphDatabaseSettings.neo4j_home);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.data", GraphDatabaseSettings.data_directory);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.transaction.logs.root", GraphDatabaseSettings.transaction_logs_root_path);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.script.root", GraphDatabaseSettings.script_root_path);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.dumps.root", GraphDatabaseSettings.database_dumps_root_path);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.import", GraphDatabaseSettings.load_csv_file_url_root);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.plugins", GraphDatabaseSettings.plugin_dir);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.logs", GraphDatabaseSettings.logs_directory);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.licenses", GraphDatabaseSettings.licenses_directory);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.run", BootloaderSettings.run_directory);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.directories.lib", BootloaderSettings.lib_directory);
        }

        private static void migrateConnectors(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            List<String> connectorSettings = values.keySet().stream().filter(key -> key.startsWith(OLD_PREFIX)).filter(key -> SUPPORTED_CONNECTOR_PATTERN.matcher((CharSequence)key).matches()).toList();
            for (String connectorSetting : connectorSettings) {
                if (connectorSetting.endsWith(".type")) {
                    values.remove(connectorSetting);
                    continue;
                }
                if (connectorSetting.equals("dbms.connector.bolt.unsupported_thread_pool_shutdown_wait_time")) {
                    SettingMigrators.migrateSettingNameChange(values, log, connectorSetting, BoltConnectorInternalSettings.thread_pool_shutdown_wait_time);
                    continue;
                }
                if (connectorSetting.equals("dbms.connector.bolt.unsupported_thread_pool_queue_size")) {
                    SettingMigrators.migrateSettingNameChange(values, log, connectorSetting, BoltConnectorInternalSettings.unsupported_thread_pool_queue_size);
                    continue;
                }
                if (connectorSetting.equals("dbms.connector.bolt.unsupported_unauth_connection_timeout")) {
                    SettingMigrators.migrateSettingNameChange(values, log, connectorSetting, BoltConnectorInternalSettings.unsupported_bolt_unauth_connection_timeout);
                    continue;
                }
                if (connectorSetting.equals("dbms.connector.bolt.unsupported_unauth_max_inbound_bytes")) {
                    SettingMigrators.migrateSettingNameChange(values, log, connectorSetting, BoltConnectorInternalSettings.unsupported_bolt_unauth_connection_max_inbound_bytes);
                    continue;
                }
                String newName = connectorSetting.replace(OLD_PREFIX, "server");
                SettingMigrators.migrateSettingNameChange(values, log, connectorSetting, newName);
            }
        }

        private static void migrateDatabaseMemorySettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_state.max_off_heap_memory", GraphDatabaseSettings.tx_state_max_off_heap_memory);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_state.off_heap.max_cacheable_block_size", GraphDatabaseSettings.tx_state_off_heap_max_cacheable_block_size);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_state.off_heap.block_cache_size", GraphDatabaseSettings.tx_state_off_heap_block_cache_size);
            String maxAllocations = values.remove("cypher.query_max_allocations");
            if (StringUtils.isNotBlank((CharSequence)maxAllocations)) {
                if (!values.containsKey(GraphDatabaseSettings.memory_transaction_max_size.name())) {
                    log.warn("The setting cypher.query_max_allocations is removed and replaced by %s.", new Object[]{GraphDatabaseSettings.memory_transaction_max_size.name()});
                    values.put(GraphDatabaseSettings.memory_transaction_max_size.name(), maxAllocations);
                } else {
                    log.warn("The setting cypher.query_max_allocations is removed and replaced by %s. Since both are set, %s will take precedence and the value of cypher.query_max_allocations, %s, will be ignored.", new Object[]{GraphDatabaseSettings.memory_transaction_max_size.name(), GraphDatabaseSettings.memory_transaction_max_size.name(), maxAllocations});
                }
            }
        }

        private static void migrateWhitelistSettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.warmup.preload.whitelist", GraphDatabaseSettings.pagecache_warmup_prefetch_allowlist);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.security.procedures.whitelist", GraphDatabaseSettings.procedure_allowlist);
        }

        private static void migrateWindowsServiceName(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.windows_service_name", BootloaderSettings.windows_service_name);
        }

        private static void migrateGroupSpatialSettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateGroupSettingPrefixChange(values, log, "unsupported.dbms.db.spatial.crs", "internal.dbms.db.spatial.crs");
        }

        private static void migrateCheckpointSettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.checkpoint", GraphDatabaseSettings.check_point_policy);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.checkpoint.interval.time", GraphDatabaseSettings.check_point_interval_time);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.checkpoint.interval.tx", GraphDatabaseSettings.check_point_interval_tx);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.checkpoint.interval.volume", GraphDatabaseSettings.check_point_interval_volume);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.checkpoint.iops.limit", GraphDatabaseSettings.check_point_iops_limit);
        }

        private static void migrateKeepAliveSetting(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.connector.bolt.connection_keep_alive_scheduling_interval", BoltConnector.connection_keep_alive_streaming_scheduling_interval);
        }

        private static void migrateReadOnlySettings(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.read_only", GraphDatabaseSettings.read_only_database_default);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.databases.default_to_read_only", GraphDatabaseSettings.read_only_database_default);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.databases.read_only", GraphDatabaseSettings.read_only_databases);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.databases.writable", GraphDatabaseSettings.writable_databases);
        }

        private static void migrateDefaultDatabaseSetting(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.default_database", GraphDatabaseSettings.initial_default_database);
        }

        private static void migrateDatabaseMaxSize(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.transaction.datababase_max_size", GraphDatabaseSettings.memory_transaction_database_max_size);
        }

        private static void migrateCypherNamespace(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "cypher.forbid_exhaustive_shortestpath", GraphDatabaseSettings.forbid_exhaustive_shortestpath);
            SettingMigrators.migrateSettingNameChange(values, log, "cypher.forbid_shortestpath_common_nodes", GraphDatabaseSettings.forbid_shortestpath_common_nodes);
            SettingMigrators.migrateSettingNameChange(values, log, "cypher.hints_error", GraphDatabaseSettings.cypher_hints_error);
            SettingMigrators.migrateSettingNameChange(values, log, "cypher.lenient_create_relationship", GraphDatabaseSettings.cypher_lenient_create_relationship);
            SettingMigrators.migrateSettingNameChange(values, log, "cypher.min_replan_interval", GraphDatabaseSettings.cypher_min_replan_interval);
            SettingMigrators.migrateSettingNameChange(values, log, "cypher.planner", GraphDatabaseSettings.cypher_planner);
            SettingMigrators.migrateSettingNameChange(values, log, "cypher.render_plan_description", GraphDatabaseSettings.cypher_render_plan_descriptions);
            SettingMigrators.migrateSettingNameChange(values, log, "cypher.statistics_divergence_threshold", GraphDatabaseSettings.query_statistics_divergence_threshold);
        }

        private static void migrateCypherQueryCacheSize(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            SettingMigrators.migrateSettingNameChange(values, log, "server.db.query_cache_size", GraphDatabaseSettings.query_cache_size);
        }

        private record Mapping(String oldSettingName, String newSettingName) {
        }
    }
}

