/**
 * Copyright (c) 2002-2014 "Neo Technology,"
 * Network Engine for Objects in Lund AB [http://neotechnology.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.cypher.internal.compiler.v2_1.planner.logical.plans

import org.neo4j.cypher.internal.compiler.v2_1.ast.{ShortestPathExpression, ShortestPaths, RelTypeName}
import org.neo4j.graphdb.Direction
import org.neo4j.cypher.internal.compiler.v2_1.docbuilders.internalDocBuilder
import org.neo4j.cypher.internal.compiler.v2_1.ast

/*
A LogicalPlan is an algebraic query, which is represented by a query tree whose leaves are database relations and
non-leaf nodes are algebraic operators like selections, projections, and joins. An intermediate node indicates the
application of the corresponding operator on the relations generated by its children, the result of which is then sent
further up. Thus, the edges of a tree represent data flow from bottom to top, i.e., from the leaves, which correspond
to data in the database, to the root, which is the final operator producing the query answer. */
abstract class LogicalPlan extends Product with internalDocBuilder.AsPrettyToString {
  def lhs: Option[LogicalPlan]
  def rhs: Option[LogicalPlan]

  def availableSymbols: Set[IdName]
}

abstract class LogicalLeafPlan extends LogicalPlan {
  final val lhs = None
  final val rhs = None
}

final case class IdName(name: String) extends AnyVal

// TODO: Remove ast representation
final case class ShortestPathPattern(name: Option[IdName], rel: PatternRelationship, single: Boolean)(val expr: ast.ShortestPaths) {
  def isFindableFrom(symbols: Set[IdName]) = symbols.contains(rel.left) && symbols.contains(rel.right)
  def availableSymbols: Set[IdName] = name.toSet ++ rel.coveredIds
}

final case class PatternRelationship(name: IdName, nodes: (IdName, IdName), dir: Direction, types: Seq[RelTypeName], length: PatternLength)
  extends internalDocBuilder.AsPrettyToString {

  def directionRelativeTo(node: IdName): Direction = if (node == left) dir else dir.reverse()

  def otherSide(node: IdName) = if (node == left) right else left

  def coveredIds: Set[IdName] = Set(name, left, right)

  def left = nodes._1

  def right = nodes._2
}

trait PatternLength extends internalDocBuilder.AsPrettyToString {
  def isSimple: Boolean
}

case object SimplePatternLength extends PatternLength {
  def isSimple = true
}

final case class VarPatternLength(min: Int, max: Option[Int]) extends PatternLength {
  def isSimple = false
}

object VarPatternLength {
  def unlimited = VarPatternLength(1, None)
  def fixed(length: Int) = VarPatternLength(length, Some(length))
}

