/*
 * Copyright (c) 2002-2017 "Neo Technology,"
 * Network Engine for Objects in Lund AB [http://neotechnology.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.cypher.internal.compatibility.v3_2

import java.util.function.BiConsumer

import org.neo4j.cypher.internal.compatibility._
import org.neo4j.cypher.internal.compiler.v3_2
import org.neo4j.cypher.internal.compiler.v3_2.executionplan.{LegacyNodeIndexUsage, LegacyRelationshipIndexUsage, SchemaIndexScanUsage, SchemaIndexSeekUsage, ExecutionPlan => ExecutionPlan_v3_2}
import org.neo4j.cypher.internal.compiler.v3_2.phases.CompilerContext
import org.neo4j.cypher.internal.compiler.v3_2.{InfoLogger, ExplainMode => ExplainModev3_2, NormalMode => NormalModev3_2, ProfileMode => ProfileModev3_2}
import org.neo4j.cypher.internal.frontend.v3_2.helpers.rewriting.RewriterStepSequencer
import org.neo4j.cypher.internal.frontend.v3_2.phases.{BaseState, CompilationPhaseTracer, RecordingNotificationLogger}
import org.neo4j.cypher.internal.frontend.v3_3.phases.CacheCheckResult
import org.neo4j.cypher.internal.javacompat.ExecutionResult
import org.neo4j.cypher.internal.spi.v3_2.TransactionBoundQueryContext.IndexSearchMonitor
import org.neo4j.cypher.internal.spi.v3_2.{ExceptionTranslatingPlanContext, TransactionBoundGraphStatistics, TransactionBoundPlanContext, TransactionBoundQueryContext, TransactionalContextWrapper => TransactionalContextWrapperV3_2}
import org.neo4j.cypher.internal.spi.v3_3.{TransactionalContextWrapper => TransactionalContextWrapperV3_3}
import org.neo4j.cypher.internal.{frontend, _}
import org.neo4j.graphdb.Result
import org.neo4j.kernel.api.KernelAPI
import org.neo4j.kernel.api.query.IndexUsage.{explicitIndexUsage, schemaIndexUsage}
import org.neo4j.kernel.api.query.PlannerInfo
import org.neo4j.kernel.impl.query.QueryExecutionMonitor
import org.neo4j.kernel.monitoring.{Monitors => KernelMonitors}
import org.neo4j.logging.Log
import org.neo4j.values.AnyValue
import org.neo4j.values.virtual.MapValue

import scala.collection.mutable
import scala.util.Try

trait Compatibility[C <: CompilerContext] {

  val queryCacheSize: Int
  val kernelMonitors: KernelMonitors
  val kernelAPI: KernelAPI

  protected val rewriterSequencer: (String) => RewriterStepSequencer = {
    import RewriterStepSequencer._
    import org.neo4j.helpers.Assertion._

    if (assertionsEnabled()) newValidating else newPlain
  }

  protected val compiler: v3_2.CypherCompiler[C]

  implicit val executionMonitor: QueryExecutionMonitor = kernelMonitors.newMonitor(classOf[QueryExecutionMonitor])

  def produceParsedQuery(preParsedQuery: PreParsedQuery, tracer: CompilationPhaseTracer,
                         preParsingNotifications: Set[org.neo4j.graphdb.Notification]): ParsedQuery = {
    val notificationLogger = new RecordingNotificationLogger
    val preparedSyntacticQueryForV_3_2 =
      Try(compiler.parseQuery(preParsedQuery.statement,
                              preParsedQuery.rawStatement,
                              notificationLogger,
                              preParsedQuery.planner.name,
                              preParsedQuery.debugOptions,
                              Some(helpers.as3_2(preParsedQuery.offset)), tracer))
    new ParsedQuery {
      override def plan(transactionalContext: TransactionalContextWrapperV3_3,
                        tracer: frontend.v3_3.phases.CompilationPhaseTracer):
      (ExecutionPlan, Map[String, Any]) = exceptionHandler.runSafely {
        val tc = TransactionalContextWrapperV3_2(transactionalContext.tc)
        val planContext = new ExceptionTranslatingPlanContext(new TransactionBoundPlanContext(tc, notificationLogger))
        val syntacticQuery = preparedSyntacticQueryForV_3_2.get
        val pos3_2 = helpers.as3_2(preParsedQuery.offset)
        val (planImpl, extractedParameters) = compiler.planPreparedQuery(syntacticQuery, notificationLogger, planContext,
                                                                         preParsedQuery.debugOptions,
                                                                         Some(pos3_2),
                                                                         helpers.as3_2(tracer))

        // Log notifications/warnings from planning
        planImpl.notifications(planContext).foreach(notificationLogger.log)

        val searchMonitor = kernelMonitors.newMonitor(classOf[IndexSearchMonitor])
        val executionPlanWrapper = new ExecutionPlanWrapper(
          planImpl,
          transactionalContext,
          preParsingNotifications,
          pos3_2,
          searchMonitor,
          executionMonitor)
        (executionPlanWrapper, extractedParameters)
      }

      override protected val trier: Try[BaseState] = preparedSyntacticQueryForV_3_2
    }
  }


}

class StringInfoLogger(log: Log) extends InfoLogger {

  def info(message: String) {
    log.info(message)
  }
}

class ExecutionPlanWrapper(inner: ExecutionPlan_v3_2,
                           transactionalContext: TransactionalContextWrapperV3_3,
                           preParsingNotifications: Set[org.neo4j.graphdb.Notification],
                           offSet: frontend.v3_2.InputPosition,
                           searchMonitor: IndexSearchMonitor,
                           executionMonitor: QueryExecutionMonitor)
  extends ExecutionPlan {


  private def queryContext(transactionalContext: TransactionalContextWrapperV3_3) = {
    val ctx = new TransactionBoundQueryContext(TransactionalContextWrapperV3_2(transactionalContext.tc))(
      searchMonitor)
    new ExceptionTranslatingQueryContext(ctx)
  }

  def run(transactionalContext: TransactionalContextWrapperV3_3, executionMode: CypherExecutionMode,
          params: Map[String, Any]): Result = {
    val innerExecutionMode = executionMode match {
      case CypherExecutionMode.explain => ExplainModev3_2
      case CypherExecutionMode.profile => ProfileModev3_2
      case CypherExecutionMode.normal => NormalModev3_2
    }
    exceptionHandler.runSafely {
      val innerParams = typeConversions.asPrivateMap(params)
      val innerResult = inner.run(queryContext(transactionalContext), innerExecutionMode, innerParams)

      new ExecutionResult(
        new ClosingExecutionResult(
          transactionalContext.tc.executingQuery(),
          new ExecutionResultWrapper(innerResult,
            inner.plannerUsed,
            inner.runtimeUsed,
            preParsingNotifications,
            Some(offSet)),
          exceptionHandler.runSafely)(executionMonitor)
      )
    }
  }

  def isPeriodicCommit: Boolean = inner.isPeriodicCommit

  def isStale(lastCommittedTxId: LastCommittedTxIdProvider, ctx: TransactionalContextWrapperV3_3) =
    //TODO: When 3.2.9 is released, remove CacheCheckResult call, because inner should return the correct type already
    CacheCheckResult(inner.isStale(lastCommittedTxId, TransactionBoundGraphStatistics(ctx.readOperations)), 0)

  override val plannerInfo: PlannerInfo = {
    import scala.collection.JavaConverters._
    new PlannerInfo(inner.plannerUsed.name, inner.runtimeUsed.name, inner.plannedIndexUsage.map {
      case SchemaIndexSeekUsage(identifier, label, propertyKeys) =>
        val labelId = transactionalContext.readOperations.labelGetForName(label)
        schemaIndexUsage(identifier, labelId, label, propertyKeys: _*)
      case SchemaIndexScanUsage(identifier, label, propertyKey) =>
        val labelId = transactionalContext.readOperations.labelGetForName(label)
        schemaIndexUsage(identifier, labelId, label, propertyKey)
      case LegacyNodeIndexUsage(identifier, index) => explicitIndexUsage(identifier, "NODE", index)
      case LegacyRelationshipIndexUsage(identifier, index) => explicitIndexUsage(identifier, "RELATIONSHIP", index)
    }.asJava)
  }

  override def run(transactionalContext: TransactionalContextWrapperV3_3,
                   executionMode: CypherExecutionMode,
                   params: MapValue): Result = {
    var map: mutable.Map[String, Any] = mutable.Map[String, Any]()
    params.foreach(new BiConsumer[String, AnyValue] {
      override def accept(t: String, u: AnyValue): Unit = map.put(t, valueHelper.fromValue(u))
    })

    run(transactionalContext, executionMode, map.toMap)
  }
}
