/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.dbms;

import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.StringJoiner;
import org.eclipse.collections.api.set.MutableSet;
import org.eclipse.collections.impl.set.mutable.MutableSetFactoryImpl;
import org.neo4j.cli.AbstractAdminCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.commandline.Util;
import org.neo4j.commandline.dbms.CannotWriteException;
import org.neo4j.commandline.dbms.LockChecker;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.helpers.DatabaseNamePattern;
import org.neo4j.dbms.archive.CompressionFormat;
import org.neo4j.dbms.archive.DumpFormatSelector;
import org.neo4j.dbms.archive.Dumper;
import org.neo4j.graphdb.config.Configuration;
import org.neo4j.internal.helpers.ArrayUtil;
import org.neo4j.internal.helpers.Exceptions;
import org.neo4j.internal.helpers.Strings;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.layout.Neo4jLayout;
import org.neo4j.io.locker.FileLockException;
import org.neo4j.kernel.impl.util.Validators;
import org.neo4j.kernel.recovery.Recovery;
import org.neo4j.logging.InternalLog;
import org.neo4j.logging.log4j.Log4jLogProvider;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import picocli.CommandLine;

@CommandLine.Command(name="dump", header={"Dump a database into a single-file archive."}, description={"Dump a database into a single-file archive. The archive can be used by the load command. <destination-path> should be a directory (in which case a file called <database>.dump will be created), or --to-stdout can be supplied to use standard output. If neither --to-path or --to-stdout is supplied `server.directories.dumps.root` setting will be used as destination. It is not possible to dump a database that is mounted in a running Neo4j server."})
public class DumpCommand
extends AbstractAdminCommand {
    @CommandLine.Parameters(arity="1", description={"Name of the database to dump. Can contain * and ? for globbing."}, converter={Converters.DatabaseNamePatternConverter.class})
    private DatabaseNamePattern database;
    @CommandLine.ArgGroup
    private TargetOption target = new TargetOption();
    @CommandLine.Option(names={"--overwrite-destination"}, arity="0..1", paramLabel="true|false", fallbackValue="true", showDefaultValue=CommandLine.Help.Visibility.ALWAYS, description={"Overwrite any existing dump file in the destination folder."})
    private boolean overwriteDestination;
    private final Dumper dumper;

    public DumpCommand(ExecutionContext ctx, Dumper dumper) {
        super(ctx);
        this.dumper = Objects.requireNonNull(dumper);
    }

    protected Optional<String> commandConfigName() {
        return Optional.of("database-dump");
    }

    public void execute() {
        block34: {
            if (this.target.toDir != null && !Files.isDirectory(Path.of(this.target.toDir, new String[0]), new LinkOption[0])) {
                throw new CommandFailedException(this.target.toDir + " is not an existing directory");
            }
            boolean toStdOut = this.target.toStdout;
            if (toStdOut && this.database.containsPattern()) {
                throw new CommandFailedException("Globbing in database name can not be used in combination with standard output. Specify a directory as destination or a single target database");
            }
            Config config = this.createConfig();
            if (this.target.toDir == null && !toStdOut) {
                this.target.toDir = this.createDefaultDumpsDir(config);
            }
            try (Log4jLogProvider logProvider = Util.configuredLogProvider((OutputStream)this.ctx.out(), (boolean)this.verbose);){
                InternalLog log = logProvider.getLog(((Object)((Object)this)).getClass());
                ArrayList<FailedDump> failedDumps = new ArrayList<FailedDump>();
                try (DefaultFileSystemAbstraction fs = new DefaultFileSystemAbstraction();){
                    Set<String> dbNames = this.getDbNames(config, (FileSystemAbstraction)fs);
                    EmptyMemoryTracker memoryTracker = EmptyMemoryTracker.INSTANCE;
                    for (String databaseName : dbNames) {
                        try {
                            if (!toStdOut) {
                                log.info(String.format("Starting dump of database '%s'", databaseName));
                            }
                            DatabaseLayout databaseLayout = Neo4jLayout.of((Configuration)config).databaseLayout(databaseName);
                            try {
                                Validators.CONTAINS_EXISTING_DATABASE.validate((Object)databaseLayout.databaseDirectory());
                            }
                            catch (IllegalArgumentException e) {
                                throw new CommandFailedException("Database does not exist: " + databaseName, (Throwable)e);
                            }
                            if (fs.fileExists(databaseLayout.file("migrate"))) {
                                throw new CommandFailedException("Store migration folder detected - A dump can not be taken during a store migration. Make sure store migration is completed before trying again.");
                            }
                            try {
                                Closeable ignored = LockChecker.checkDatabaseLock(databaseLayout);
                                try {
                                    this.checkDbState(this.ctx.fs(), databaseLayout, config, (MemoryTracker)memoryTracker, databaseName);
                                    this.dump(databaseLayout, databaseName);
                                }
                                finally {
                                    if (ignored == null) continue;
                                    ignored.close();
                                }
                            }
                            catch (FileLockException e) {
                                throw new CommandFailedException("The database is in use. Stop database '" + databaseName + "' and try again.", (Throwable)e);
                            }
                            catch (IOException e) {
                                DumpCommand.wrapIOException(e);
                            }
                            catch (CannotWriteException e) {
                                throw new CommandFailedException("You do not have permission to dump the database.", (Throwable)e);
                            }
                        }
                        catch (Exception e) {
                            log.error("Failed to dump database '" + databaseName + "': " + e.getMessage());
                            failedDumps.add(new FailedDump(databaseName, e));
                        }
                    }
                }
                if (failedDumps.isEmpty()) {
                    if (!toStdOut) {
                        log.info("Dump completed successfully");
                    }
                    break block34;
                }
                StringJoiner failedDbs = new StringJoiner("', '", "Dump failed for databases: '", "'");
                Exception exceptions = null;
                for (FailedDump failedDump : failedDumps) {
                    failedDbs.add(failedDump.dbName);
                    exceptions = (Exception)Exceptions.chain(exceptions, (Throwable)failedDump.e);
                }
                log.error(failedDbs.toString());
                throw new CommandFailedException(failedDbs.toString(), exceptions);
            }
        }
    }

    private String createDefaultDumpsDir(Config config) {
        Path defaultDumpPath = (Path)config.get(GraphDatabaseSettings.database_dumps_root_path);
        try {
            this.ctx.fs().mkdirs(defaultDumpPath);
        }
        catch (IOException e) {
            throw new CommandFailedException(String.format("Unable to create default dumps directory at '%s': %s: %s", defaultDumpPath, e.getClass().getSimpleName(), e.getMessage()), (Throwable)e);
        }
        return defaultDumpPath.toString();
    }

    private Config createConfig() {
        return this.createPrefilledConfigBuilder().set(GraphDatabaseSettings.read_only_database_default, (Object)true).build();
    }

    private Set<String> getDbNames(Config config, FileSystemAbstraction fs) {
        if (!this.database.containsPattern()) {
            return Set.of(this.database.getDatabaseName());
        }
        MutableSet dbNames = MutableSetFactoryImpl.INSTANCE.empty();
        Path databasesDir = Neo4jLayout.of((Configuration)config).databasesDirectory();
        try {
            for (Path path : fs.listFiles(databasesDir)) {
                String name;
                if (!fs.isDirectory(path) || !this.database.matches(name = path.getFileName().toString())) continue;
                dbNames.add(name);
            }
        }
        catch (IOException e) {
            throw new CommandFailedException("Failed to list databases", (Throwable)e);
        }
        return dbNames;
    }

    private static Path buildArchivePath(String database, Path to) {
        return to.resolve(database + ".dump");
    }

    private OutputStream openDumpStream(String databaseName, TargetOption destination) throws IOException {
        if (destination.toStdout) {
            return this.ctx.out();
        }
        Path archive = DumpCommand.buildArchivePath(databaseName, Path.of(destination.toDir, new String[0]).toAbsolutePath());
        return this.dumper.openForDump(archive, this.overwriteDestination);
    }

    private void dump(DatabaseLayout databaseLayout, String databaseName) {
        Path databasePath = databaseLayout.databaseDirectory();
        try {
            CompressionFormat format = DumpFormatSelector.selectFormat(this.ctx.err());
            String lockFile = databaseLayout.databaseLockFile().getFileName().toString();
            String quarantineMarkerFile = databaseLayout.quarantineMarkerFile().getFileName().toString();
            OutputStream out = this.openDumpStream(databaseName, this.target);
            this.dumper.dump(databasePath, databaseLayout.getTransactionLogsDirectory(), out, format, path -> DumpCommand.oneOf(path, lockFile, quarantineMarkerFile));
        }
        catch (FileAlreadyExistsException e) {
            throw new CommandFailedException("Archive already exists: " + e.getMessage(), (Throwable)e);
        }
        catch (NoSuchFileException e) {
            if (Paths.get(e.getMessage(), new String[0]).toAbsolutePath().equals(databasePath)) {
                throw new CommandFailedException("Database does not exist: " + databaseLayout.getDatabaseName(), (Throwable)e);
            }
            DumpCommand.wrapIOException(e);
        }
        catch (IOException e) {
            DumpCommand.wrapIOException(e);
        }
    }

    private static boolean oneOf(Path path, String ... names) {
        return ArrayUtil.contains((Object[])names, (Object)path.getFileName().toString());
    }

    protected void checkDbState(FileSystemAbstraction fs, DatabaseLayout databaseLayout, Config additionalConfiguration, MemoryTracker memoryTracker, String databaseName) {
        if (DumpCommand.checkRecoveryState(fs, databaseLayout, additionalConfiguration, memoryTracker)) {
            throw new CommandFailedException(Strings.joinAsLines((String[])new String[]{"Active logical log detected, this might be a source of inconsistencies.", "Please recover database before running the dump.", "To perform recovery please start database and perform clean shutdown."}));
        }
    }

    private static boolean checkRecoveryState(FileSystemAbstraction fs, DatabaseLayout databaseLayout, Config additionalConfiguration, MemoryTracker memoryTracker) {
        try {
            return Recovery.isRecoveryRequired((FileSystemAbstraction)fs, (DatabaseLayout)databaseLayout, (Config)additionalConfiguration, (MemoryTracker)memoryTracker);
        }
        catch (Exception e) {
            throw new CommandFailedException("Failure when checking for recovery state: '%s'." + e.getMessage(), (Throwable)e);
        }
    }

    private static void wrapIOException(IOException e) {
        throw new CommandFailedException(String.format("Unable to dump database: %s: %s", e.getClass().getSimpleName(), e.getMessage()), (Throwable)e);
    }

    private static class TargetOption {
        @CommandLine.Option(names={"--to-path"}, paramLabel="<path>", description={"Destination folder of database dump."})
        private String toDir;
        @CommandLine.Option(names={"--to-stdout"}, description={"Use standard output as destination for database dump."})
        private boolean toStdout;

        private TargetOption() {
        }
    }

    record FailedDump(String dbName, Exception e) {
    }
}

