/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.dbms;

import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Optional;
import java.util.StringJoiner;
import org.neo4j.cli.AbstractAdminCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.cloud.storage.SchemeFileSystemAbstraction;
import org.neo4j.commandline.Util;
import org.neo4j.commandline.dbms.CannotWriteException;
import org.neo4j.commandline.dbms.LockChecker;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.helpers.DatabaseNamePattern;
import org.neo4j.dbms.archive.CompressionFormat;
import org.neo4j.dbms.archive.DumpFormatSelector;
import org.neo4j.dbms.archive.Dumper;
import org.neo4j.graphdb.config.Configuration;
import org.neo4j.internal.helpers.ArrayUtil;
import org.neo4j.internal.helpers.Exceptions;
import org.neo4j.internal.helpers.Strings;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.layout.Neo4jLayout;
import org.neo4j.io.locker.FileLockException;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.io.pagecache.tracing.PageCacheTracer;
import org.neo4j.kernel.impl.pagecache.ConfiguringPageCacheFactory;
import org.neo4j.kernel.impl.scheduler.JobSchedulerFactory;
import org.neo4j.kernel.impl.util.Validators;
import org.neo4j.kernel.recovery.Recovery;
import org.neo4j.logging.InternalLog;
import org.neo4j.logging.InternalLogProvider;
import org.neo4j.logging.NullLog;
import org.neo4j.logging.log4j.Log4jLog;
import org.neo4j.logging.log4j.Log4jLogProvider;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryPools;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.scheduler.JobScheduler;
import org.neo4j.storageengine.api.DeprecatedFormatWarning;
import org.neo4j.storageengine.api.StorageEngineFactory;
import org.neo4j.time.Clocks;
import picocli.CommandLine;

@CommandLine.Command(name="dump", header={"Dump a database into a single-file archive."}, description={"Dump a database into a single-file archive. The archive can be used by the load command. <to-path> should be a directory (in which case a file called <database>.dump will be created), or --to-stdout can be supplied to use standard output. If neither --to-path or --to-stdout is supplied `server.directories.dumps.root` setting will be used as destination. It is not possible to dump a database that is mounted in a running Neo4j server."})
public class DumpCommand
extends AbstractAdminCommand {
    @CommandLine.Parameters(arity="1", description={"Name of the database to dump. Can contain * and ? for globbing. Note that * and ? have special meaning in some shells and might need to be escaped or used with quotes."}, converter={Converters.DatabaseNamePatternConverter.class})
    private DatabaseNamePattern database;
    @CommandLine.ArgGroup
    private TargetOption target = new TargetOption();
    @CommandLine.Option(names={"--overwrite-destination"}, arity="0..1", paramLabel="true|false", fallbackValue="true", showDefaultValue=CommandLine.Help.Visibility.ALWAYS, description={"Overwrite any existing dump file in the destination folder."})
    private boolean overwriteDestination;

    public DumpCommand(ExecutionContext ctx) {
        super(ctx);
    }

    protected Optional<String> commandConfigName() {
        return Optional.of("database-dump");
    }

    public void execute() {
        block32: {
            Config config = this.createConfig();
            boolean dumpToStdOut = this.target.toStdout;
            PrintStream logStream = dumpToStdOut ? this.ctx.err() : this.ctx.out();
            try (Log4jLogProvider logProvider = Util.configuredLogProvider((OutputStream)logStream, (boolean)this.verbose);
                 SchemeFileSystemAbstraction fs = new SchemeFileSystemAbstraction(this.ctx.fs(), config, (InternalLogProvider)logProvider);){
                Dumper dumper = this.createDumper((FileSystemAbstraction)fs, logStream);
                Path storagePath = null;
                if (this.target.toDir != null && !fs.isDirectory(storagePath = fs.resolve(this.target.toDir))) {
                    throw new CommandFailedException(this.target.toDir + " is not an existing directory");
                }
                if (dumpToStdOut && this.database.containsPattern()) {
                    throw new CommandFailedException("Globbing in database name can not be used in combination with standard output. Specify a directory as destination or a single target database");
                }
                if (storagePath == null && !dumpToStdOut) {
                    storagePath = this.createDefaultDumpsDir(fs, config);
                }
                Log4jLog log = logProvider.getLog(((Object)((Object)this)).getClass());
                ArrayList<FailedDump> failedDumps = new ArrayList<FailedDump>();
                EmptyMemoryTracker memoryTracker = EmptyMemoryTracker.INSTANCE;
                for (String databaseName : DumpCommand.getDbNames((Config)config, (FileSystemAbstraction)fs, (DatabaseNamePattern)this.database)) {
                    try {
                        log.info(String.format("Starting dump of database '%s'", databaseName));
                        DatabaseLayout databaseLayout = Neo4jLayout.of((Configuration)config).databaseLayout(databaseName);
                        try {
                            Validators.CONTAINS_EXISTING_DATABASE.validate((Object)databaseLayout.databaseDirectory());
                        }
                        catch (IllegalArgumentException e) {
                            throw new CommandFailedException("Database does not exist: " + databaseName, (Throwable)e);
                        }
                        if (fs.fileExists(databaseLayout.file("migrate"))) {
                            throw new CommandFailedException("Store migration folder detected - A dump can not be taken during a store migration. Make sure store migration is completed before trying again.");
                        }
                        try {
                            Closeable ignored = LockChecker.checkDatabaseLock((DatabaseLayout)databaseLayout);
                            try {
                                this.checkDbState((FileSystemAbstraction)fs, databaseLayout, config, (MemoryTracker)memoryTracker, databaseName, (InternalLog)log);
                                DumpCommand.logFormatDeprecationWarning((InternalLog)log, databaseLayout, config, (FileSystemAbstraction)fs);
                                this.dump(dumper, databaseLayout, databaseName, storagePath);
                            }
                            finally {
                                if (ignored == null) continue;
                                ignored.close();
                            }
                        }
                        catch (FileLockException e) {
                            throw new CommandFailedException("The database is in use. Stop database '" + databaseName + "' and try again.", (Throwable)e);
                        }
                        catch (CannotWriteException e) {
                            throw new CommandFailedException("You do not have permission to dump the database.", (Throwable)e);
                        }
                    }
                    catch (Exception e) {
                        log.error("Failed to dump database '" + databaseName + "': " + e.getMessage());
                        failedDumps.add(new FailedDump(databaseName, e));
                    }
                }
                if (failedDumps.isEmpty()) {
                    log.info("Dump completed successfully");
                    break block32;
                }
                StringJoiner failedDbs = new StringJoiner("', '", "Dump failed for databases: '", "'");
                Exception exceptions = null;
                for (FailedDump failedDump : failedDumps) {
                    failedDbs.add(failedDump.dbName);
                    exceptions = (Exception)Exceptions.chain(exceptions, (Throwable)failedDump.e);
                }
                log.error(failedDbs.toString());
                throw new CommandFailedException(failedDbs.toString(), exceptions);
            }
            catch (IOException e) {
                DumpCommand.wrapIOException(e);
            }
        }
    }

    protected Dumper createDumper(FileSystemAbstraction fs, PrintStream out) {
        return new Dumper(fs, out);
    }

    private Path createDefaultDumpsDir(SchemeFileSystemAbstraction fs, Config config) {
        Path defaultDumpPath = (Path)config.get(GraphDatabaseSettings.database_dumps_root_path);
        try {
            fs.mkdirs(defaultDumpPath);
        }
        catch (IOException e) {
            throw new CommandFailedException(String.format("Unable to create default dumps directory at '%s': %s: %s", defaultDumpPath, e.getClass().getSimpleName(), e.getMessage()), (Throwable)e);
        }
        return defaultDumpPath;
    }

    private Config createConfig() {
        return this.createPrefilledConfigBuilder().set(GraphDatabaseSettings.read_only_database_default, (Object)true).build();
    }

    private static void logFormatDeprecationWarning(InternalLog log, DatabaseLayout databaseLayout, Config config, FileSystemAbstraction fs) {
        try (JobScheduler jobScheduler = JobSchedulerFactory.createInitialisedScheduler();
             PageCache pageCache = DumpCommand.getPageCache(config, fs, jobScheduler);){
            StorageEngineFactory storageEngineFactory = StorageEngineFactory.selectStorageEngine((FileSystemAbstraction)fs, (DatabaseLayout)databaseLayout, (Configuration)config);
            String format = storageEngineFactory.retrieveStoreId(fs, databaseLayout, pageCache, CursorContext.NULL_CONTEXT).getFormatName();
            if (storageEngineFactory.isDeprecated(format)) {
                log.warn(DeprecatedFormatWarning.getFormatWarning((String)databaseLayout.getDatabaseName(), (String)format));
            }
        }
        catch (IOException e) {
            throw new CommandFailedException("Unable to find store id");
        }
    }

    private static PageCache getPageCache(Config config, FileSystemAbstraction fs, JobScheduler jobScheduler) {
        ConfiguringPageCacheFactory pageCacheFactory = new ConfiguringPageCacheFactory(fs, config, PageCacheTracer.NULL, (InternalLog)NullLog.getInstance(), jobScheduler, Clocks.nanoClock(), new MemoryPools());
        return pageCacheFactory.getOrCreatePageCache();
    }

    private static Path buildArchivePath(String database, Path to) {
        return to.resolve(database + ".dump");
    }

    private OutputStream openDumpStream(Dumper dumper, String databaseName, Path storagePath) throws IOException {
        if (storagePath == null) {
            return this.ctx.out();
        }
        Path archive = DumpCommand.buildArchivePath(databaseName, storagePath);
        return dumper.openForDump(archive, this.overwriteDestination);
    }

    private void dump(Dumper dumper, DatabaseLayout databaseLayout, String databaseName, Path storagePath) {
        Path databasePath = databaseLayout.databaseDirectory();
        try {
            CompressionFormat format = DumpFormatSelector.selectFormat(this.ctx.err());
            String lockFile = databaseLayout.databaseLockFile().getFileName().toString();
            String quarantineMarkerFile = databaseLayout.quarantineFile().getFileName().toString();
            OutputStream out = this.openDumpStream(dumper, databaseName, storagePath);
            dumper.dump(databasePath, databaseLayout.getTransactionLogsDirectory(), out, format, path -> DumpCommand.oneOf(path, lockFile, quarantineMarkerFile));
        }
        catch (FileAlreadyExistsException e) {
            throw new CommandFailedException("Archive already exists: " + e.getMessage(), (Throwable)e);
        }
        catch (NoSuchFileException e) {
            if (Paths.get(e.getMessage(), new String[0]).toAbsolutePath().equals(databasePath)) {
                throw new CommandFailedException("Database does not exist: " + databaseLayout.getDatabaseName(), (Throwable)e);
            }
            DumpCommand.wrapIOException(e);
        }
        catch (IOException e) {
            DumpCommand.wrapIOException(e);
        }
    }

    private static boolean oneOf(Path path, String ... names) {
        return ArrayUtil.contains((Object[])names, (Object)path.getFileName().toString());
    }

    protected void checkDbState(FileSystemAbstraction fs, DatabaseLayout databaseLayout, Config additionalConfiguration, MemoryTracker memoryTracker, String databaseName, InternalLog log) {
        if (DumpCommand.checkRecoveryState(fs, databaseLayout, additionalConfiguration, memoryTracker)) {
            throw new CommandFailedException(Strings.joinAsLines((String[])new String[]{"Active logical log detected, this might be a source of inconsistencies.", "Please recover database before running the dump.", "To perform recovery please start database and perform clean shutdown."}));
        }
    }

    private static boolean checkRecoveryState(FileSystemAbstraction fs, DatabaseLayout databaseLayout, Config additionalConfiguration, MemoryTracker memoryTracker) {
        try {
            return Recovery.isRecoveryRequired((FileSystemAbstraction)fs, (DatabaseLayout)databaseLayout, (Config)additionalConfiguration, (MemoryTracker)memoryTracker);
        }
        catch (Exception e) {
            throw new CommandFailedException("Failure when checking for recovery state: '%s'." + e.getMessage(), (Throwable)e);
        }
    }

    private static void wrapIOException(IOException e) {
        throw new CommandFailedException(String.format("Unable to dump database: %s: %s", e.getClass().getSimpleName(), e.getMessage()), (Throwable)e);
    }

    private static class TargetOption {
        @CommandLine.Option(names={"--to-path"}, paramLabel="<path>", description={"Destination folder of a database dump.\nIt is possible to dump databases into AWS S3 buckets, Google Cloud storage buckets, and Azure buckets using the appropriate URI as the path."})
        private String toDir;
        @CommandLine.Option(names={"--to-stdout"}, description={"Use standard output as the destination for the database dump."})
        private boolean toStdout;

        private TargetOption() {
        }
    }

    record FailedDump(String dbName, Exception e) {
    }
}

