/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.fs;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.ClosedChannelException;
import java.util.Objects;
import java.util.zip.Checksum;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.ChecksumMismatchException;
import org.neo4j.io.fs.ChecksumWriter;
import org.neo4j.io.fs.ReadPastEndException;
import org.neo4j.io.fs.ReadableChannel;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.memory.NativeScopedBuffer;
import org.neo4j.io.memory.ScopedBuffer;
import org.neo4j.memory.MemoryTracker;

public class ReadAheadChannel<T extends StoreChannel>
implements ReadableChannel {
    public static final int DEFAULT_READ_AHEAD_SIZE = Math.toIntExact(ByteUnit.kibiBytes(4L));
    private final ScopedBuffer scopedBuffer;
    protected T channel;
    private final ByteBuffer aheadBuffer;
    private final int readAheadSize;
    private final Checksum checksum;
    private final ByteBuffer checksumView;

    private ReadAheadChannel(T channel, ByteBuffer byteBuffer, ScopedBuffer scopedBuffer) {
        Objects.requireNonNull(channel);
        Objects.requireNonNull(byteBuffer);
        this.aheadBuffer = byteBuffer;
        this.aheadBuffer.position(this.aheadBuffer.capacity());
        this.channel = channel;
        this.readAheadSize = this.aheadBuffer.capacity();
        this.checksumView = this.aheadBuffer.duplicate();
        this.checksum = ChecksumWriter.CHECKSUM_FACTORY.get();
        this.scopedBuffer = scopedBuffer;
    }

    public ReadAheadChannel(T channel, ByteBuffer byteBuffer) {
        this(channel, byteBuffer, null);
    }

    public ReadAheadChannel(T channel, ScopedBuffer scopedBuffer) {
        this(channel, scopedBuffer.getBuffer(), scopedBuffer);
    }

    public ReadAheadChannel(T channel, MemoryTracker memoryTracker) {
        this(channel, new NativeScopedBuffer(DEFAULT_READ_AHEAD_SIZE, ByteOrder.LITTLE_ENDIAN, memoryTracker));
    }

    public int readAheadBufferSize() {
        return this.aheadBuffer.capacity();
    }

    @Override
    public long position() throws IOException {
        return this.channel.position() - (long)this.aheadBuffer.remaining();
    }

    @Override
    public byte get() throws IOException {
        this.ensureDataExists(1);
        return this.aheadBuffer.get();
    }

    @Override
    public short getShort() throws IOException {
        this.ensureDataExists(2);
        return this.aheadBuffer.getShort();
    }

    @Override
    public int getInt() throws IOException {
        this.ensureDataExists(4);
        return this.aheadBuffer.getInt();
    }

    @Override
    public long getLong() throws IOException {
        this.ensureDataExists(8);
        return this.aheadBuffer.getLong();
    }

    @Override
    public float getFloat() throws IOException {
        this.ensureDataExists(4);
        return this.aheadBuffer.getFloat();
    }

    @Override
    public double getDouble() throws IOException {
        this.ensureDataExists(8);
        return this.aheadBuffer.getDouble();
    }

    @Override
    public void get(byte[] bytes, int length) throws IOException {
        int chunkSize;
        assert (length <= bytes.length);
        for (int bytesGotten = 0; bytesGotten < length; bytesGotten += chunkSize) {
            chunkSize = Math.min(this.readAheadSize >> 2, length - bytesGotten);
            this.ensureDataExists(chunkSize);
            this.aheadBuffer.get(bytes, bytesGotten, chunkSize);
        }
    }

    @Override
    public byte getVersion() throws IOException {
        return this.get();
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        int length = dst.remaining();
        if (this.aheadBuffer.remaining() >= length) {
            dst.put(dst.position(), this.aheadBuffer, this.aheadBuffer.position(), length);
            this.aheadBuffer.position(this.aheadBuffer.position() + length);
            dst.position(dst.position() + length);
            return length;
        }
        this.checksumView.limit(this.aheadBuffer.limit());
        this.checksum.update(this.checksumView);
        this.checksumView.clear();
        dst.put(this.aheadBuffer);
        this.aheadBuffer.limit(0);
        dst.mark();
        int remainingBytes = dst.remaining();
        while (remainingBytes > 0) {
            int read = this.channel.read(dst);
            if (read == -1) {
                T nextChannel = this.next(this.channel);
                if (nextChannel == this.channel) {
                    throw ReadPastEndException.INSTANCE;
                }
                this.channel = nextChannel;
                continue;
            }
            remainingBytes -= read;
        }
        dst.reset();
        this.checksum.update(dst);
        return length;
    }

    @Override
    public int endChecksumAndValidate() throws IOException {
        this.ensureDataExists(4);
        this.checksumView.limit(this.aheadBuffer.position());
        this.checksum.update(this.checksumView);
        int calculatedChecksum = (int)this.checksum.getValue();
        int storedChecksum = this.aheadBuffer.getInt();
        if (calculatedChecksum != storedChecksum) {
            throw new ChecksumMismatchException(storedChecksum, calculatedChecksum);
        }
        this.beginChecksum();
        return calculatedChecksum;
    }

    @Override
    public int getChecksum() {
        this.checksumView.limit(this.aheadBuffer.position());
        this.checksum.update(this.checksumView);
        return (int)this.checksum.getValue();
    }

    @Override
    public void beginChecksum() {
        this.checksum.reset();
        this.checksumView.limit(this.checksumView.capacity());
        this.checksumView.position(this.aheadBuffer.position());
    }

    @Override
    public boolean isOpen() {
        return this.channel != null && this.channel.isOpen();
    }

    @Override
    public void close() throws IOException {
        if (this.channel != null) {
            this.channel.close();
            this.channel = null;
        }
        if (this.scopedBuffer != null) {
            this.scopedBuffer.close();
        }
    }

    private void ensureDataExists(int requestedNumberOfBytes) throws IOException {
        int remaining = this.aheadBuffer.remaining();
        if (remaining >= requestedNumberOfBytes) {
            return;
        }
        if (this.channel == null || !this.channel.isOpen()) {
            throw new ClosedChannelException();
        }
        this.checksumView.limit(this.aheadBuffer.position());
        this.checksum.update(this.checksumView);
        this.checksumView.clear();
        this.aheadBuffer.compact();
        while (this.aheadBuffer.position() < this.aheadBuffer.capacity()) {
            int read = this.channel.read(this.aheadBuffer);
            if (read != -1) continue;
            if (this.aheadBuffer.position() >= requestedNumberOfBytes) break;
            T nextChannel = this.next(this.channel);
            assert (nextChannel != null);
            if (nextChannel == this.channel) {
                this.aheadBuffer.flip();
                throw ReadPastEndException.INSTANCE;
            }
            this.channel = nextChannel;
        }
        this.aheadBuffer.flip();
    }

    protected T next(T channel) throws IOException {
        return channel;
    }

    protected void resetAheadBuffer() {
        this.aheadBuffer.limit(0);
    }

    @Override
    public void position(long byteOffset) throws IOException {
        long positionRelativeToAheadBuffer = byteOffset - (this.channel.position() - (long)this.aheadBuffer.limit());
        if (positionRelativeToAheadBuffer >= (long)this.aheadBuffer.limit() || positionRelativeToAheadBuffer < 0L) {
            this.aheadBuffer.position(this.aheadBuffer.limit());
            this.channel.position(byteOffset);
        } else {
            this.aheadBuffer.position(Math.toIntExact(positionRelativeToAheadBuffer));
        }
        this.beginChecksum();
    }
}

