/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.pagecache;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.RuleChain;
import org.junit.rules.TestRule;
import org.neo4j.io.pagecache.FileHandle;
import org.neo4j.io.pagecache.Page;
import org.neo4j.io.pagecache.PageEvictionCallback;
import org.neo4j.io.pagecache.PageSwapper;
import org.neo4j.io.pagecache.PageSwapperFactory;
import org.neo4j.io.pagecache.impl.ByteBufferPage;
import org.neo4j.test.rule.TestDirectory;

public abstract class PageSwapperTest {
    public static final PageEvictionCallback NO_CALLBACK = (pageId, page) -> {};
    public static final long X = -3819410105021120785L;
    public static final long Y = 6846544296635974449L;
    public static final int Z = -16843010;
    protected static final int cachePageSize = 32;
    public final TestDirectory testDir = TestDirectory.testDirectory();
    public final ExpectedException expectedException = ExpectedException.none();
    @Rule
    public final RuleChain rules = RuleChain.outerRule((TestRule)this.testDir).around((TestRule)this.expectedException);
    private final ConcurrentLinkedQueue<PageSwapperFactory> openedFactories = new ConcurrentLinkedQueue();
    private final ConcurrentLinkedQueue<PageSwapper> openedSwappers = new ConcurrentLinkedQueue();

    protected abstract PageSwapperFactory swapperFactory() throws Exception;

    protected abstract void mkdirs(File var1) throws IOException;

    protected abstract File baseDirectory() throws IOException;

    protected final PageSwapperFactory createSwapperFactory() throws Exception {
        PageSwapperFactory factory = this.swapperFactory();
        this.openedFactories.add(factory);
        return factory;
    }

    protected int cachePageSize() {
        return 32;
    }

    protected ByteBufferPage createPage(int cachePageSize) {
        return new ByteBufferPage(ByteBuffer.allocateDirect(cachePageSize));
    }

    protected ByteBufferPage createPage() {
        return this.createPage(this.cachePageSize());
    }

    protected void clear(ByteBufferPage page) {
        byte b = 0;
        for (int i = 0; i < this.cachePageSize(); ++i) {
            page.putByte(b, i);
        }
    }

    protected PageSwapper createSwapper(PageSwapperFactory factory, File file, int filePageSize, PageEvictionCallback callback, boolean createIfNotExist) throws IOException {
        PageSwapper swapper = factory.createPageSwapper(file, filePageSize, callback, createIfNotExist);
        this.openedSwappers.add(swapper);
        return swapper;
    }

    protected final PageSwapper createSwapperAndFile(PageSwapperFactory factory, File file) throws IOException {
        return this.createSwapperAndFile(factory, file, this.cachePageSize());
    }

    protected final PageSwapper createSwapperAndFile(PageSwapperFactory factory, File file, int filePageSize) throws IOException {
        return this.createSwapper(factory, file, filePageSize, NO_CALLBACK, true);
    }

    private File file(String filename) throws IOException {
        File file = this.testDir.file(filename);
        this.mkdirs(file.getParentFile());
        return file;
    }

    private long sizeOf(ByteBufferPage page) {
        return page.size();
    }

    @Before
    @After
    public void clearStrayInterrupts() {
        Thread.interrupted();
    }

    @After
    public void closeOpenedPageSwappers() throws Exception {
        PageSwapperFactory factory;
        PageSwapper swapper;
        Exception exception = null;
        while ((swapper = this.openedSwappers.poll()) != null) {
            try {
                swapper.close();
            }
            catch (IOException e) {
                if (exception == null) {
                    exception = e;
                    continue;
                }
                exception.addSuppressed(e);
            }
        }
        while ((factory = this.openedFactories.poll()) != null) {
            try {
                factory.close();
            }
            catch (Exception e) {
                if (exception == null) {
                    exception = e;
                    continue;
                }
                exception.addSuppressed(e);
            }
        }
        if (exception != null) {
            throw exception;
        }
    }

    @Test
    public void readMustNotSwallowInterrupts() throws Exception {
        File file = this.file("a");
        ByteBufferPage page = this.createPage();
        page.putInt(1, 0);
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Assert.assertThat((Object)swapper.write(0L, (Page)page), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        page.putInt(0, 0);
        Thread.currentThread().interrupt();
        Assert.assertThat((Object)swapper.read(0L, (Page)page), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)1));
        Assert.assertThat((Object)swapper.read(0L, (Page)page), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)1));
    }

    @Test
    public void vectoredReadMustNotSwallowInterrupts() throws Exception {
        File file = this.file("a");
        ByteBufferPage page = this.createPage();
        page.putInt(1, 0);
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Assert.assertThat((Object)swapper.write(0L, (Page)page), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        page.putInt(0, 0);
        Thread.currentThread().interrupt();
        Assert.assertThat((Object)swapper.read(0L, new Page[]{page}, 0, 1), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)1));
        Assert.assertThat((Object)swapper.read(0L, new Page[]{page}, 0, 1), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)1));
    }

    @Test
    public void writeMustNotSwallowInterrupts() throws Exception {
        File file = this.file("a");
        ByteBufferPage page = this.createPage();
        page.putInt(1, 0);
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        Assert.assertThat((Object)swapper.write(0L, (Page)page), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertTrue((boolean)Thread.currentThread().isInterrupted());
        page.putInt(0, 0);
        Assert.assertThat((Object)swapper.read(0L, (Page)page), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)1));
        Assert.assertThat((Object)swapper.write(0L, (Page)page), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertTrue((boolean)Thread.currentThread().isInterrupted());
        page.putInt(0, 0);
        Assert.assertThat((Object)swapper.read(0L, (Page)page), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)1));
    }

    @Test
    public void vectoredWriteMustNotSwallowInterrupts() throws Exception {
        File file = this.file("a");
        ByteBufferPage page = this.createPage();
        page.putInt(1, 0);
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        Assert.assertThat((Object)swapper.write(0L, new Page[]{page}, 0, 1), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertTrue((boolean)Thread.currentThread().isInterrupted());
        page.putInt(0, 0);
        Assert.assertThat((Object)swapper.read(0L, (Page)page), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)1));
        Assert.assertThat((Object)swapper.write(0L, new Page[]{page}, 0, 1), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertTrue((boolean)Thread.currentThread().isInterrupted());
        page.putInt(0, 0);
        Assert.assertThat((Object)swapper.read(0L, (Page)page), (Matcher)Matchers.is((Object)this.sizeOf(page)));
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)1));
    }

    @Test
    public void forcingMustNotSwallowInterrupts() throws Exception {
        File file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        swapper.force();
        Assert.assertTrue((boolean)Thread.currentThread().isInterrupted());
    }

    @Test
    public void mustReopenChannelWhenReadFailsWithAsynchronousCloseException() throws Exception {
        File file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        ByteBufferPage page = this.createPage();
        page.putLong(-3819410105021120785L, 0);
        page.putLong(6846544296635974449L, 8);
        page.putInt(-16843010, 16);
        swapper.write(0L, (Page)page);
        Thread.currentThread().interrupt();
        swapper.read(0L, (Page)page);
        Assert.assertTrue((boolean)Thread.interrupted());
        Assert.assertThat((Object)page.getLong(0), (Matcher)Matchers.is((Object)-3819410105021120785L));
        Assert.assertThat((Object)page.getLong(8), (Matcher)Matchers.is((Object)6846544296635974449L));
        Assert.assertThat((Object)page.getInt(16), (Matcher)Matchers.is((Object)-16843010));
        swapper.force();
    }

    @Test
    public void mustReopenChannelWhenVectoredReadFailsWithAsynchronousCloseException() throws Exception {
        File file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        ByteBufferPage page = this.createPage();
        page.putLong(-3819410105021120785L, 0);
        page.putLong(6846544296635974449L, 8);
        page.putInt(-16843010, 16);
        swapper.write(0L, (Page)page);
        Thread.currentThread().interrupt();
        swapper.read(0L, new Page[]{page}, 0, 1);
        Assert.assertTrue((boolean)Thread.interrupted());
        Assert.assertThat((Object)page.getLong(0), (Matcher)Matchers.is((Object)-3819410105021120785L));
        Assert.assertThat((Object)page.getLong(8), (Matcher)Matchers.is((Object)6846544296635974449L));
        Assert.assertThat((Object)page.getInt(16), (Matcher)Matchers.is((Object)-16843010));
        swapper.force();
    }

    @Test
    public void mustReopenChannelWhenWriteFailsWithAsynchronousCloseException() throws Exception {
        ByteBufferPage page = this.createPage();
        page.putLong(-3819410105021120785L, 0);
        page.putLong(6846544296635974449L, 8);
        page.putInt(-16843010, 16);
        File file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        swapper.write(0L, (Page)page);
        Assert.assertTrue((boolean)Thread.interrupted());
        swapper.force();
        this.clear(page);
        swapper.read(0L, (Page)page);
        Assert.assertThat((Object)page.getLong(0), (Matcher)Matchers.is((Object)-3819410105021120785L));
        Assert.assertThat((Object)page.getLong(8), (Matcher)Matchers.is((Object)6846544296635974449L));
        Assert.assertThat((Object)page.getInt(16), (Matcher)Matchers.is((Object)-16843010));
    }

    @Test
    public void mustReopenChannelWhenVectoredWriteFailsWithAsynchronousCloseException() throws Exception {
        ByteBufferPage page = this.createPage();
        page.putLong(-3819410105021120785L, 0);
        page.putLong(6846544296635974449L, 8);
        page.putInt(-16843010, 16);
        File file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        swapper.write(0L, new Page[]{page}, 0, 1);
        Assert.assertTrue((boolean)Thread.interrupted());
        swapper.force();
        this.clear(page);
        swapper.read(0L, (Page)page);
        Assert.assertThat((Object)page.getLong(0), (Matcher)Matchers.is((Object)-3819410105021120785L));
        Assert.assertThat((Object)page.getLong(8), (Matcher)Matchers.is((Object)6846544296635974449L));
        Assert.assertThat((Object)page.getInt(16), (Matcher)Matchers.is((Object)-16843010));
    }

    @Test
    public void mustReopenChannelWhenForceFailsWithAsynchronousCloseException() throws Exception {
        File file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        for (int i = 0; i < 10; ++i) {
            Thread.currentThread().interrupt();
            swapper.force();
            Assert.assertTrue((boolean)Thread.interrupted());
        }
    }

    @Test
    public void readMustNotReopenExplicitlyClosedChannel() throws Exception {
        String filename = "a";
        File file = this.file(filename);
        ByteBufferPage page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.write(0L, (Page)page);
        swapper.close();
        try {
            swapper.read(0L, (Page)page);
            Assert.fail((String)"Should have thrown because the channel should be closed");
        }
        catch (ClosedChannelException closedChannelException) {
            // empty catch block
        }
    }

    @Test
    public void vectoredReadMustNotReopenExplicitlyClosedChannel() throws Exception {
        String filename = "a";
        File file = this.file(filename);
        ByteBufferPage page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.write(0L, (Page)page);
        swapper.close();
        try {
            swapper.read(0L, new Page[]{page}, 0, 1);
            Assert.fail((String)"Should have thrown because the channel should be closed");
        }
        catch (ClosedChannelException closedChannelException) {
            // empty catch block
        }
    }

    @Test
    public void writeMustNotReopenExplicitlyClosedChannel() throws Exception {
        File file = this.file("a");
        ByteBufferPage page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.close();
        try {
            swapper.write(0L, (Page)page);
            Assert.fail((String)"Should have thrown because the channel should be closed");
        }
        catch (ClosedChannelException closedChannelException) {
            // empty catch block
        }
    }

    @Test
    public void vectoredWriteMustNotReopenExplicitlyClosedChannel() throws Exception {
        File file = this.file("a");
        ByteBufferPage page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.close();
        try {
            swapper.write(0L, new Page[]{page}, 0, 1);
            Assert.fail((String)"Should have thrown because the channel should be closed");
        }
        catch (ClosedChannelException closedChannelException) {
            // empty catch block
        }
    }

    @Test
    public void forceMustNotReopenExplicitlyClosedChannel() throws Exception {
        File file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.close();
        try {
            swapper.force();
            Assert.fail((String)"Should have thrown because the channel should be closed");
        }
        catch (ClosedChannelException closedChannelException) {
            // empty catch block
        }
    }

    @Test
    public void mustNotOverwriteDataInOtherFiles() throws Exception {
        File fileA = this.file("a");
        File fileB = this.file("b");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapperA = this.createSwapperAndFile(factory, fileA);
        PageSwapper swapperB = this.createSwapperAndFile(factory, fileB);
        ByteBufferPage page = this.createPage();
        page.putLong(-3819410105021120785L, 0);
        swapperA.write(0L, (Page)page);
        page.putLong(6846544296635974449L, 8);
        swapperB.write(0L, (Page)page);
        this.clear(page);
        swapperA.read(0L, (Page)page);
        Assert.assertThat((Object)page.getLong(0), (Matcher)Matchers.is((Object)-3819410105021120785L));
        Assert.assertThat((Object)page.getLong(8), (Matcher)Matchers.is((Object)0L));
    }

    @Test
    public void mustRunEvictionCallbackOnEviction() throws Exception {
        AtomicLong callbackFilePageId = new AtomicLong();
        AtomicReference callbackPage = new AtomicReference();
        PageEvictionCallback callback = (filePageId, page) -> {
            callbackFilePageId.set(filePageId);
            callbackPage.set(page);
        };
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapper(factory, file, this.cachePageSize(), callback, true);
        ByteBufferPage page2 = this.createPage();
        swapper.evicted(42L, (Page)page2);
        Assert.assertThat((Object)callbackFilePageId.get(), (Matcher)Matchers.is((Object)42L));
        Assert.assertThat(callbackPage.get(), (Matcher)Matchers.sameInstance((Object)page2));
    }

    @Test
    public void mustNotIssueEvictionCallbacksAfterSwapperHasBeenClosed() throws Exception {
        AtomicBoolean gotCallback = new AtomicBoolean();
        PageEvictionCallback callback = (filePageId, page) -> gotCallback.set(true);
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapper(factory, file, this.cachePageSize(), callback, true);
        ByteBufferPage page2 = this.createPage();
        swapper.close();
        swapper.evicted(42L, (Page)page2);
        Assert.assertFalse((boolean)gotCallback.get());
    }

    @Test
    public void mustThrowExceptionIfFileDoesNotExist() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        this.expectedException.expect(NoSuchFileException.class);
        this.createSwapper(factory, this.file("does not exist"), this.cachePageSize(), NO_CALLBACK, false);
    }

    @Test
    public void mustCreateNonExistingFileWithCreateFlag() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper pageSwapper = this.createSwapperAndFile(factory, this.file("does not exist"));
        ByteBufferPage page = this.createPage();
        page.putLong(-3819410105021120785L, 0);
        pageSwapper.write(0L, (Page)page);
        this.clear(page);
        pageSwapper.read(0L, (Page)page);
        Assert.assertThat((Object)page.getLong(0), (Matcher)Matchers.is((Object)-3819410105021120785L));
    }

    @Test
    public void truncatedFilesMustBeEmpty() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file);
        Assert.assertThat((Object)swapper.getLastPageId(), (Matcher)Matchers.is((Object)-1L));
        ByteBufferPage page = this.createPage();
        page.putInt(-889275714, 0);
        swapper.write(10L, (Page)page);
        this.clear(page);
        swapper.read(10L, (Page)page);
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)-889275714));
        Assert.assertThat((Object)swapper.getLastPageId(), (Matcher)Matchers.is((Object)10L));
        swapper.close();
        swapper = this.createSwapper(factory, file, this.cachePageSize(), NO_CALLBACK, false);
        this.clear(page);
        swapper.read(10L, (Page)page);
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)-889275714));
        Assert.assertThat((Object)swapper.getLastPageId(), (Matcher)Matchers.is((Object)10L));
        swapper.truncate();
        this.clear(page);
        swapper.read(10L, (Page)page);
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)0));
        Assert.assertThat((Object)swapper.getLastPageId(), (Matcher)Matchers.is((Object)-1L));
        swapper.close();
        swapper = this.createSwapper(factory, file, this.cachePageSize(), NO_CALLBACK, false);
        this.clear(page);
        swapper.read(10L, (Page)page);
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)0));
        Assert.assertThat((Object)swapper.getLastPageId(), (Matcher)Matchers.is((Object)-1L));
        swapper.close();
    }

    @Test
    public void positionedVectoredWriteMustFlushAllBuffersInOrder() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage pageA = this.createPage(4);
        ByteBufferPage pageB = this.createPage(4);
        ByteBufferPage pageC = this.createPage(4);
        ByteBufferPage pageD = this.createPage(4);
        pageA.putInt(2, 0);
        pageB.putInt(3, 0);
        pageC.putInt(4, 0);
        pageD.putInt(5, 0);
        swapper.write(1L, new Page[]{pageA, pageB, pageC, pageD}, 0, 4);
        ByteBufferPage result = this.createPage(4);
        swapper.read(0L, (Page)result);
        Assert.assertThat((Object)result.getInt(0), (Matcher)Matchers.is((Object)0));
        result.putInt(0, 0);
        Assert.assertThat((Object)swapper.read(1L, (Page)result), (Matcher)Matchers.is((Object)4L));
        Assert.assertThat((Object)result.getInt(0), (Matcher)Matchers.is((Object)2));
        result.putInt(0, 0);
        Assert.assertThat((Object)swapper.read(2L, (Page)result), (Matcher)Matchers.is((Object)4L));
        Assert.assertThat((Object)result.getInt(0), (Matcher)Matchers.is((Object)3));
        result.putInt(0, 0);
        Assert.assertThat((Object)swapper.read(3L, (Page)result), (Matcher)Matchers.is((Object)4L));
        Assert.assertThat((Object)result.getInt(0), (Matcher)Matchers.is((Object)4));
        result.putInt(0, 0);
        Assert.assertThat((Object)swapper.read(4L, (Page)result), (Matcher)Matchers.is((Object)4L));
        Assert.assertThat((Object)result.getInt(0), (Matcher)Matchers.is((Object)5));
        result.putInt(0, 0);
        Assert.assertThat((Object)swapper.read(5L, (Page)result), (Matcher)Matchers.is((Object)0L));
        Assert.assertThat((Object)result.getInt(0), (Matcher)Matchers.is((Object)0));
    }

    @Test
    public void positionedVectoredReadMustFillAllBuffersInOrder() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage output = this.createPage();
        output.putInt(2, 0);
        swapper.write(1L, (Page)output);
        output.putInt(3, 0);
        swapper.write(2L, (Page)output);
        output.putInt(4, 0);
        swapper.write(3L, (Page)output);
        output.putInt(5, 0);
        swapper.write(4L, (Page)output);
        ByteBufferPage pageA = this.createPage(4);
        ByteBufferPage pageB = this.createPage(4);
        ByteBufferPage pageC = this.createPage(4);
        ByteBufferPage pageD = this.createPage(4);
        Assert.assertThat((Object)swapper.read(1L, new Page[]{pageA, pageB, pageC, pageD}, 0, 4), (Matcher)Matchers.is((Object)16L));
        Assert.assertThat((Object)pageA.getInt(0), (Matcher)Matchers.is((Object)2));
        Assert.assertThat((Object)pageB.getInt(0), (Matcher)Matchers.is((Object)3));
        Assert.assertThat((Object)pageC.getInt(0), (Matcher)Matchers.is((Object)4));
        Assert.assertThat((Object)pageD.getInt(0), (Matcher)Matchers.is((Object)5));
    }

    @Test
    public void positionedVectoredReadFromEmptyFileMustFillPagesWithZeros() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage page = this.createPage(4);
        page.putInt(1, 0);
        Assert.assertThat((Object)swapper.read(0L, new Page[]{page}, 0, 1), (Matcher)Matchers.is((Object)0L));
        Assert.assertThat((Object)page.getInt(0), (Matcher)Matchers.is((Object)0));
    }

    @Test
    public void positionedVectoredReadBeyondEndOfFileMustFillPagesWithZeros() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage output = this.createPage(4);
        output.putInt(-1, 0);
        swapper.write(0L, new Page[]{output, output, output}, 0, 3);
        ByteBufferPage pageA = this.createPage(4);
        ByteBufferPage pageB = this.createPage(4);
        pageA.putInt(-1, 0);
        pageB.putInt(-1, 0);
        Assert.assertThat((Object)swapper.read(3L, new Page[]{pageA, pageB}, 0, 2), (Matcher)Matchers.is((Object)0L));
        Assert.assertThat((Object)pageA.getInt(0), (Matcher)Matchers.is((Object)0));
        Assert.assertThat((Object)pageB.getInt(0), (Matcher)Matchers.is((Object)0));
    }

    @Test
    public void positionedVectoredReadWhereLastPageExtendBeyondEndOfFileMustHaveRemainderZeroFilled() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage output = this.createPage(4);
        output.putInt(-1, 0);
        swapper.write(0L, new Page[]{output, output, output, output, output}, 0, 5);
        swapper.close();
        swapper = this.createSwapper(factory, file, 8, NO_CALLBACK, false);
        ByteBufferPage pageA = this.createPage(8);
        ByteBufferPage pageB = this.createPage(8);
        pageA.putLong(-3819410105021120785L, 0);
        pageB.putLong(6846544296635974449L, 0);
        Assert.assertThat((Object)swapper.read(1L, new Page[]{pageA, pageB}, 0, 2), (Matcher)Matchers.isOneOf((Object[])new Long[]{12L, 16L}));
        Assert.assertThat((Object)pageA.getLong(0), (Matcher)Matchers.is((Object)-1L));
        Assert.assertThat((Object)pageB.getLong(0), (Matcher)Matchers.is((Object)-4294967296L));
    }

    @Test
    public void positionedVectoredReadWhereSecondLastPageExtendBeyondEndOfFileMustHaveRestZeroFilled() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage output = this.createPage(4);
        output.putInt(1, 0);
        swapper.write(0L, (Page)output);
        output.putInt(2, 0);
        swapper.write(1L, (Page)output);
        output.putInt(3, 0);
        swapper.write(2L, (Page)output);
        swapper.close();
        swapper = this.createSwapper(factory, file, 8, NO_CALLBACK, false);
        ByteBufferPage pageA = this.createPage(8);
        ByteBufferPage pageB = this.createPage(8);
        ByteBufferPage pageC = this.createPage(8);
        pageA.putInt(-1, 0);
        pageB.putInt(-1, 0);
        pageC.putInt(-1, 0);
        Assert.assertThat((Object)swapper.read(0L, new Page[]{pageA, pageB, pageC}, 0, 3), (Matcher)Matchers.isOneOf((Object[])new Long[]{12L, 16L}));
        Assert.assertThat((Object)pageA.getInt(0), (Matcher)Matchers.is((Object)1));
        Assert.assertThat((Object)pageA.getInt(4), (Matcher)Matchers.is((Object)2));
        Assert.assertThat((Object)pageB.getInt(0), (Matcher)Matchers.is((Object)3));
        Assert.assertThat((Object)pageB.getInt(4), (Matcher)Matchers.is((Object)0));
        Assert.assertThat((Object)pageC.getLong(0), (Matcher)Matchers.is((Object)0L));
    }

    @Test
    public void concurrentPositionedVectoredReadsAndWritesMustNotInterfere() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        int pageCount = 100;
        int iterations = 20000;
        CountDownLatch startLatch = new CountDownLatch(1);
        ByteBufferPage output = this.createPage(4);
        for (int i = 0; i < 100; ++i) {
            output.putInt(i + 1, 0);
            swapper.write((long)i, (Page)output);
        }
        Callable<Void> work = () -> {
            int i;
            ThreadLocalRandom rng = ThreadLocalRandom.current();
            Page[] pages = new ByteBufferPage[10];
            for (i = 0; i < pages.length; ++i) {
                pages[i] = this.createPage(4);
            }
            startLatch.await();
            for (i = 0; i < 20000; ++i) {
                long startFilePageId = rng.nextLong(0L, 100 - pages.length);
                if (rng.nextBoolean()) {
                    long bytesRead = swapper.read(startFilePageId, pages, 0, pages.length);
                    Assert.assertThat((Object)bytesRead, (Matcher)Matchers.is((Object)((long)pages.length * 4L)));
                    for (int j = 0; j < pages.length; ++j) {
                        int expectedValue = (int)((long)(1 + j) + startFilePageId);
                        int actualValue = pages[j].getInt(0);
                        Assert.assertThat((Object)actualValue, (Matcher)Matchers.is((Object)expectedValue));
                    }
                    continue;
                }
                for (int j = 0; j < pages.length; ++j) {
                    int value = (int)((long)(1 + j) + startFilePageId);
                    pages[j].putInt(value, 0);
                }
                Assert.assertThat((Object)swapper.write(startFilePageId, pages, 0, pages.length), (Matcher)Matchers.is((Object)((long)pages.length * 4L)));
            }
            return null;
        };
        int threads = 8;
        ExecutorService executor = Executors.newFixedThreadPool(threads, r -> {
            Thread thread = Executors.defaultThreadFactory().newThread(r);
            thread.setDaemon(true);
            return thread;
        });
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>(threads);
        for (int i = 0; i < threads; ++i) {
            futures.add(executor.submit(work));
        }
        startLatch.countDown();
        for (Future future : futures) {
            future.get();
        }
    }

    @Test
    public void positionedVectoredReadMustWorkOnSubsequenceOfGivenArray() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage pageA = this.createPage(4);
        ByteBufferPage pageB = this.createPage(4);
        ByteBufferPage pageC = this.createPage(4);
        ByteBufferPage pageD = this.createPage(4);
        pageA.putInt(1, 0);
        pageB.putInt(2, 0);
        pageC.putInt(3, 0);
        pageD.putInt(4, 0);
        Page[] pages = new Page[]{pageA, pageB, pageC, pageD};
        long bytesWritten = swapper.write(0L, pages, 0, 4);
        Assert.assertThat((Object)bytesWritten, (Matcher)Matchers.is((Object)16L));
        pageA.putInt(5, 0);
        pageB.putInt(6, 0);
        pageC.putInt(7, 0);
        pageD.putInt(8, 0);
        long bytesRead = swapper.read(1L, pages, 1, 2);
        Assert.assertThat((Object)bytesRead, (Matcher)Matchers.is((Object)8L));
        int[] actualValues = new int[]{pageA.getInt(0), pageB.getInt(0), pageC.getInt(0), pageD.getInt(0)};
        int[] expectedValues = new int[]{5, 2, 3, 8};
        Assert.assertThat((Object)actualValues, (Matcher)Matchers.is((Object)expectedValues));
    }

    @Test
    public void positionedVectoredWriteMustWorkOnSubsequenceOfGivenArray() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage pageA = this.createPage(4);
        ByteBufferPage pageB = this.createPage(4);
        ByteBufferPage pageC = this.createPage(4);
        ByteBufferPage pageD = this.createPage(4);
        pageA.putInt(1, 0);
        pageB.putInt(2, 0);
        pageC.putInt(3, 0);
        pageD.putInt(4, 0);
        Page[] pages = new Page[]{pageA, pageB, pageC, pageD};
        long bytesWritten = swapper.write(0L, pages, 0, 4);
        Assert.assertThat((Object)bytesWritten, (Matcher)Matchers.is((Object)16L));
        pageB.putInt(6, 0);
        pageC.putInt(7, 0);
        bytesWritten = swapper.write(1L, pages, 1, 2);
        Assert.assertThat((Object)bytesWritten, (Matcher)Matchers.is((Object)8L));
        pageA.putInt(0, 0);
        pageB.putInt(0, 0);
        pageC.putInt(0, 0);
        pageD.putInt(0, 0);
        long bytesRead = swapper.read(0L, pages, 0, 4);
        Assert.assertThat((Object)bytesRead, (Matcher)Matchers.is((Object)16L));
        int[] actualValues = new int[]{pageA.getInt(0), pageB.getInt(0), pageC.getInt(0), pageD.getInt(0)};
        int[] expectedValues = new int[]{1, 6, 7, 4};
        Assert.assertThat((Object)actualValues, (Matcher)Matchers.is((Object)expectedValues));
    }

    @Test
    public void mustThrowNullPointerExceptionFromReadWhenPageArrayElementsAreNull() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage page = this.createPage(4);
        swapper.write(0L, new Page[]{page, page, page, page}, 0, 4);
        try {
            swapper.read(0L, new Page[]{page, page, null, page}, 0, 4);
            Assert.fail((String)"vectored read with nulls in array should have thrown");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void mustThrowNullPointerExceptionFromWriteWhenPageArrayElementsAreNull() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage page = this.createPage(4);
        try {
            swapper.write(0L, new Page[]{page, page, null, page}, 0, 4);
            Assert.fail((String)"vectored read with nulls in array should have thrown");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void mustThrowNullPointerExceptionFromReadWhenPageArrayIsNull() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage page = this.createPage(4);
        swapper.write(0L, new Page[]{page, page, page, page}, 0, 4);
        try {
            swapper.read(0L, null, 0, 4);
            Assert.fail((String)"vectored read with null array should have thrown");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void mustThrowNullPointerExceptionFromWriteWhenPageArrayIsNull() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        try {
            swapper.write(0L, null, 0, 4);
            Assert.fail((String)"vectored write with null array should have thrown");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void readMustThrowForNegativeFilePageIds() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        this.expectedException.expect(IOException.class);
        swapper.read(-1L, (Page)this.createPage(4));
    }

    @Test
    public void writeMustThrowForNegativeFilePageIds() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        this.expectedException.expect(IOException.class);
        swapper.write(-1L, (Page)this.createPage(4));
    }

    @Test
    public void vectoredReadMustThrowForNegativeFilePageIds() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        this.expectedException.expect(IOException.class);
        swapper.read(-1L, new Page[]{this.createPage(4), this.createPage(4)}, 0, 2);
    }

    @Test
    public void vectoredWriteMustThrowForNegativeFilePageIds() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        this.expectedException.expect(IOException.class);
        swapper.write(-1L, new Page[]{this.createPage(4), this.createPage(4)}, 0, 2);
    }

    @Test
    public void vectoredReadMustThrowForNegativeArrayOffsets() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        Page[] pages = new Page[]{this.createPage(4), this.createPage(4)};
        swapper.write(0L, pages, 0, 2);
        this.expectedException.expect(ArrayIndexOutOfBoundsException.class);
        swapper.read(0L, pages, -1, 2);
    }

    @Test
    public void vectoredWriteMustThrowForNegativeArrayOffsets() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        Page[] pages = new Page[]{this.createPage(4), this.createPage(4)};
        this.expectedException.expect(ArrayIndexOutOfBoundsException.class);
        swapper.write(0L, pages, -1, 2);
    }

    @Test
    public void vectoredReadMustThrowWhenLengthGoesBeyondArraySize() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        Page[] pages = new Page[]{this.createPage(4), this.createPage(4)};
        swapper.write(0L, pages, 0, 2);
        this.expectedException.expect(ArrayIndexOutOfBoundsException.class);
        swapper.read(0L, pages, 1, 2);
    }

    @Test
    public void vectoredWriteMustThrowWhenLengthGoesBeyondArraySize() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        Page[] pages = new Page[]{this.createPage(4), this.createPage(4)};
        this.expectedException.expect(ArrayIndexOutOfBoundsException.class);
        swapper.write(0L, pages, 1, 2);
    }

    @Test
    public void vectoredReadMustThrowWhenArrayOffsetIsEqualToArrayLength() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        Page[] pages = new Page[]{this.createPage(4), this.createPage(4)};
        swapper.write(0L, pages, 0, 2);
        this.expectedException.expect(ArrayIndexOutOfBoundsException.class);
        swapper.read(0L, pages, 2, 1);
    }

    @Test
    public void vectoredWriteMustThrowWhenArrayOffsetIsEqualToArrayLength() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        Page[] pages = new Page[]{this.createPage(4), this.createPage(4)};
        this.expectedException.expect(ArrayIndexOutOfBoundsException.class);
        swapper.write(0L, pages, 2, 1);
    }

    @Test
    public void vectoredReadMustThrowWhenArrayOffsetIsGreaterThanArrayLength() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        Page[] pages = new Page[]{this.createPage(4), this.createPage(4)};
        swapper.write(0L, pages, 0, 2);
        this.expectedException.expect(ArrayIndexOutOfBoundsException.class);
        swapper.read(0L, pages, 3, 1);
    }

    @Test
    public void vectoredWriteMustThrowWhenArrayOffsetIsGreaterThanArrayLength() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        Page[] pages = new Page[]{this.createPage(4), this.createPage(4)};
        this.expectedException.expect(ArrayIndexOutOfBoundsException.class);
        swapper.write(0L, pages, 3, 1);
    }

    @Test
    public void vectoredReadMustReadNothingWhenLengthIsZero() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage pageA = this.createPage(4);
        ByteBufferPage pageB = this.createPage(4);
        pageA.putInt(1, 0);
        pageB.putInt(2, 0);
        Page[] pages = new Page[]{pageA, pageB};
        swapper.write(0L, pages, 0, 2);
        pageA.putInt(3, 0);
        pageB.putInt(4, 0);
        swapper.read(0L, pages, 0, 0);
        int[] expectedValues = new int[]{3, 4};
        int[] actualValues = new int[]{pageA.getInt(0), pageB.getInt(0)};
        Assert.assertThat((Object)actualValues, (Matcher)Matchers.is((Object)expectedValues));
    }

    @Test
    public void vectoredWriteMustReadNothingWhenLengthIsZero() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        ByteBufferPage pageA = this.createPage(4);
        ByteBufferPage pageB = this.createPage(4);
        pageA.putInt(1, 0);
        pageB.putInt(2, 0);
        Page[] pages = new Page[]{pageA, pageB};
        swapper.write(0L, pages, 0, 2);
        pageA.putInt(3, 0);
        pageB.putInt(4, 0);
        swapper.write(0L, pages, 0, 0);
        swapper.read(0L, pages, 0, 2);
        int[] expectedValues = new int[]{1, 2};
        int[] actualValues = new int[]{pageA.getInt(0), pageB.getInt(0)};
        Assert.assertThat((Object)actualValues, (Matcher)Matchers.is((Object)expectedValues));
    }

    @Test
    public void mustDeleteFileIfClosedWithCloseAndDelete() throws Exception {
        File file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory();
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        swapper.closeAndDelete();
        try {
            this.createSwapper(factory, file, 4, NO_CALLBACK, false);
            Assert.fail((String)"should not have been able to create a page swapper for non-existing file");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void streamFilesRecursiveMustBeEmptyForEmptyBaseDirectory() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        Assert.assertThat((Object)factory.streamFilesRecursive(this.baseDirectory()).count(), (Matcher)Matchers.is((Object)0L));
    }

    @Test
    public void streamFilesRecursiveMustListAllFilesInBaseDirectory() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File b = new File(base, "b");
        this.createSwapperAndFile(factory, a);
        this.createSwapperAndFile(factory, b);
        Set files = factory.streamFilesRecursive(base).map(FileHandle::getFile).collect(Collectors.toSet());
        Assert.assertThat(files, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a, b}));
    }

    @Test
    public void streamFilesRecursiveMustListAllFilesInSubDirectories() throws Exception {
        Object[] files;
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File sub1 = new File(base, "sub1");
        File sub1sub1 = new File(sub1, "sub1");
        File sub2 = new File(base, "sub2");
        File sub3 = new File(base, "sub3");
        this.mkdirs(sub1);
        this.mkdirs(sub1sub1);
        this.mkdirs(sub2);
        this.mkdirs(sub3);
        File a = new File(base, "a");
        File b = new File(sub1, "b");
        File c = new File(sub1sub1, "c");
        File d = new File(sub1sub1, "d");
        File e = new File(sub2, "e");
        for (File file : files = new File[]{a, b, c, d, e}) {
            this.createSwapperAndFile(factory, file);
        }
        Set set = factory.streamFilesRecursive(base).map(FileHandle::getFile).collect(Collectors.toSet());
        Assert.assertThat(set, (Matcher)Matchers.containsInAnyOrder((Object[])files));
    }

    @Test
    public void streamFilesRecursiveFilePathsMustBeCanonical() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File sub = new File(base, "sub");
        this.mkdirs(sub);
        File a = new File(new File(new File(sub, ".."), "sub"), "a");
        File canonicalFile = a.getCanonicalFile();
        this.createSwapperAndFile(factory, canonicalFile);
        String actualPath = factory.streamFilesRecursive(a).map(fh -> fh.getFile().getAbsolutePath()).findAny().get();
        Assert.assertThat((Object)actualPath, (Matcher)Matchers.is((Object)canonicalFile.getAbsolutePath()));
    }

    @Test
    public void streamFilesRecursiveMustListSingleFileGivenAsBase() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File b = new File(base, "b");
        this.createSwapperAndFile(factory, a);
        this.createSwapperAndFile(factory, b);
        Set files = factory.streamFilesRecursive(a).map(FileHandle::getFile).collect(Collectors.toSet());
        Assert.assertThat(files, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a}));
    }

    @Test
    public void streamFilesRecursiveMustThrowOnNonExistingBasePath() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File nonExisting = new File(base, "nonExisting");
        this.expectedException.expect(NoSuchFileException.class);
        factory.streamFilesRecursive(nonExisting);
    }

    @Test
    public void streamFilesRecursiveMustRenameFiles() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File b = new File(base, "b");
        this.createSwapperAndFile(factory, a).close();
        FileHandle handle = (FileHandle)factory.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[0]);
        this.createSwapper(factory, b, this.cachePageSize(), NO_CALLBACK, false);
    }

    @Test
    public void streamFilesRecursiveMustRenameDelete() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File b = new File(base, "b");
        this.createSwapperAndFile(factory, a).close();
        this.createSwapperAndFile(factory, b).close();
        FileHandle handle = (FileHandle)factory.streamFilesRecursive(a).findAny().get();
        handle.delete();
        Set files = factory.streamFilesRecursive(base).map(FileHandle::getFile).collect(Collectors.toSet());
        Assert.assertThat(files, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{b}));
    }

    @Test
    public void streamFilesRecursiveMustThrowWhenDeletingNonExistingFile() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        PageSwapper swapperA = this.createSwapperAndFile(factory, a);
        FileHandle handle = (FileHandle)factory.streamFilesRecursive(a).findAny().get();
        swapperA.closeAndDelete();
        this.expectedException.expect(NoSuchFileException.class);
        handle.delete();
    }

    @Test
    public void streamFilesRecursiveMustThrowWhenTargetFileOfRenameAlreadyExists() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File b = new File(base, "b");
        this.createSwapperAndFile(factory, a).close();
        this.createSwapperAndFile(factory, b).close();
        FileHandle handle = (FileHandle)factory.streamFilesRecursive(a).findAny().get();
        this.expectedException.expect(FileAlreadyExistsException.class);
        handle.rename(b, new CopyOption[0]);
    }

    @Test
    public void streamFilesRecursiveMustNotThrowWhenTargetFileOfRenameAlreadyExistsAndUsingReplaceExisting() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File b = new File(base, "b");
        this.createSwapperAndFile(factory, a).close();
        this.createSwapperAndFile(factory, b).close();
        FileHandle handle = (FileHandle)factory.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[]{StandardCopyOption.REPLACE_EXISTING});
    }

    @Test
    public void streamFilesRecursiveMustCreateMissingPathDirectoriesImpliedByFileRename() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File target = new File(new File(new File(base, "sub"), "sub"), "target");
        this.createSwapperAndFile(factory, a).close();
        FileHandle handle = (FileHandle)factory.streamFilesRecursive(a).findAny().get();
        handle.rename(target, new CopyOption[0]);
        this.createSwapper(factory, target, this.cachePageSize(), NO_CALLBACK, false);
    }

    @Test
    public void streamFilesRecursiveMustNotSeeFilesLaterCreatedBaseDirectory() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File b = new File(base, "b");
        this.createSwapperAndFile(factory, a).close();
        Stream stream = factory.streamFilesRecursive(base);
        this.createSwapperAndFile(factory, b).close();
        Assert.assertThat(stream.map(FileHandle::getFile).collect(Collectors.toSet()), (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a}));
    }

    @Test
    public void streamFilesRecursiveMustNotSeeFilesRenamedIntoBaseDirectory() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File sub = new File(base, "sub");
        this.mkdirs(sub);
        File x = new File(sub, "x");
        this.createSwapperAndFile(factory, a).close();
        this.createSwapperAndFile(factory, x).close();
        File target = new File(base, "target");
        Iterable handles = factory.streamFilesRecursive(base)::iterator;
        HashSet<File> observedFiles = new HashSet<File>();
        for (FileHandle handle : handles) {
            File file = handle.getFile();
            observedFiles.add(file);
            if (!file.equals(x)) continue;
            handle.rename(target, new CopyOption[0]);
        }
        Assert.assertThat(observedFiles, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a, x}));
    }

    @Test
    public void streamFilesRecursiveMustNotSeeFilesRenamedIntoSubDirectory() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File sub = new File(base, "sub");
        this.mkdirs(sub);
        File target = new File(sub, "target");
        this.createSwapperAndFile(factory, a).close();
        Iterable handles = factory.streamFilesRecursive(base)::iterator;
        HashSet<File> observedFiles = new HashSet<File>();
        for (FileHandle handle : handles) {
            File file = handle.getFile();
            observedFiles.add(file);
            if (!file.equals(a)) continue;
            handle.rename(target, new CopyOption[0]);
        }
        Assert.assertThat(observedFiles, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a}));
    }

    @Test
    public void streamFilesRecursiveSourceFileMustNotExistAfterRename() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File b = new File(base, "b");
        this.createSwapperAndFile(factory, a).close();
        FileHandle handle = (FileHandle)factory.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[0]);
        this.expectedException.expect(NoSuchFileException.class);
        this.createSwapper(factory, a, this.cachePageSize(), NO_CALLBACK, false);
    }

    @Test
    public void streamFilesRecursiveRenameMustNotChangeSourceFileContents() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File b = new File(base, "b");
        ByteBufferPage page = this.createPage();
        PageSwapper swapper = this.createSwapperAndFile(factory, a);
        long expectedValue = 62678480411623166L;
        page.putLong(expectedValue, 0);
        swapper.write(0L, (Page)page);
        this.clear(page);
        swapper.close();
        FileHandle handle = (FileHandle)factory.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[0]);
        swapper = this.createSwapper(factory, b, this.cachePageSize(), NO_CALLBACK, false);
        swapper.read(0L, (Page)page);
        long actualValue = page.getLong(0);
        Assert.assertThat((Object)actualValue, (Matcher)Matchers.is((Object)expectedValue));
    }

    @Test
    public void streamFilesRecursiveRenameMustNotChangeSourceFileContentsWithReplaceExisting() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory();
        File base = this.baseDirectory();
        File a = new File(base, "a");
        File b = new File(base, "b");
        ByteBufferPage page = this.createPage();
        PageSwapper swapper = this.createSwapperAndFile(factory, a);
        long expectedValue = 62678480411623166L;
        page.putLong(expectedValue, 0);
        swapper.write(0L, (Page)page);
        this.clear(page);
        swapper.close();
        swapper = this.createSwapperAndFile(factory, b);
        page.putLong(ThreadLocalRandom.current().nextLong(), 0);
        swapper.write(0L, (Page)page);
        swapper.close();
        this.clear(page);
        FileHandle handle = (FileHandle)factory.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[]{StandardCopyOption.REPLACE_EXISTING});
        swapper = this.createSwapper(factory, b, this.cachePageSize(), NO_CALLBACK, false);
        swapper.read(0L, (Page)page);
        long actualValue = page.getLong(0);
        Assert.assertThat((Object)actualValue, (Matcher)Matchers.is((Object)expectedValue));
    }
}

