/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.mem;

import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.neo4j.internal.unsafe.UnsafeUtil;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.mem.GrabAllocator;
import org.neo4j.io.mem.MemoryAllocator;
import org.neo4j.memory.LocalMemoryTracker;
import org.neo4j.memory.MemoryTracker;

class MemoryAllocatorTest {
    private static final long ONE_PAGE = 8192L;
    private static final long EIGHT_PAGES = 65536L;
    private MemoryAllocator allocator;

    MemoryAllocatorTest() {
    }

    @AfterEach
    void tearDown() {
        this.closeAllocator();
    }

    @Test
    void allocatedPointerMustNotBeNull() {
        MemoryAllocator mman = this.createAllocator(65536L);
        long address = mman.allocateAligned(8192L, 8L);
        Assertions.assertThat((long)address).isNotEqualTo(0L);
    }

    @Test
    void allocatedPointerMustBePageAligned() {
        MemoryAllocator mman = this.createAllocator(65536L);
        long address = mman.allocateAligned(8192L, (long)UnsafeUtil.pageSize());
        Assertions.assertThat((long)(address % (long)UnsafeUtil.pageSize())).isEqualTo(0L);
    }

    @Test
    void allocatedPointerMustBeAlignedToArbitraryByte() {
        int pageSize = UnsafeUtil.pageSize();
        for (int initialOffset = 0; initialOffset < 8; ++initialOffset) {
            for (int i = 0; i < pageSize - 1; ++i) {
                MemoryAllocator mman = this.createAllocator(8192L);
                mman.allocateAligned((long)initialOffset, 1L);
                long alignment = 1 + i;
                long address = mman.allocateAligned(8192L, alignment);
                ((AbstractLongAssert)Assertions.assertThat((long)(address % alignment)).as("With initial offset " + initialOffset + ", iteration " + i + ", aligning to " + alignment + " and got address " + address, new Object[0])).isEqualTo(0L);
            }
        }
    }

    @Test
    void mustBeAbleToAllocatePastMemoryLimit() {
        MemoryAllocator mman = this.createAllocator(8192L);
        for (int i = 0; i < 4100; ++i) {
            Assertions.assertThat((long)(mman.allocateAligned(1L, 2L) % 2L)).isEqualTo(0L);
        }
    }

    @Test
    void allocatedPointersMustBeAlignedPastMemoryLimit() {
        MemoryAllocator mman = this.createAllocator(8192L);
        for (int i = 0; i < 4100; ++i) {
            Assertions.assertThat((long)(mman.allocateAligned(1L, 2L) % 2L)).isEqualTo(0L);
        }
        int pageSize = UnsafeUtil.pageSize();
        for (int i = 0; i < pageSize - 1; ++i) {
            int alignment = pageSize - i;
            long address = mman.allocateAligned(8192L, (long)alignment);
            ((AbstractLongAssert)Assertions.assertThat((long)(address % (long)alignment)).as("iteration " + i + ", aligning to " + alignment, new Object[0])).isEqualTo(0L);
        }
    }

    @Test
    void alignmentCannotBeZero() {
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.createAllocator(8192L).allocateAligned(8L, 0L));
    }

    @Test
    void mustBeAbleToAllocateSlabsLargerThanGrabSize() {
        MemoryAllocator mman = this.createAllocator(ByteUnit.MebiByte.toBytes(2L));
        long page1 = mman.allocateAligned((long)UnsafeUtil.pageSize(), 1L);
        long largeBlock = mman.allocateAligned(0x100000L, 1L);
        long page2 = mman.allocateAligned((long)UnsafeUtil.pageSize(), 1L);
        Assertions.assertThat((long)page1).isNotEqualTo(0L);
        Assertions.assertThat((long)largeBlock).isNotEqualTo(0L);
        Assertions.assertThat((long)page2).isNotEqualTo(0L);
    }

    @Test
    void allocatingMustIncreaseMemoryUsedAndDecreaseAvailableMemory() {
        MemoryAllocator mman = this.createAllocator(8192L);
        Assertions.assertThat((long)mman.usedMemory()).isEqualTo(0L);
        Assertions.assertThat((long)mman.availableMemory()).isEqualTo(8192L);
        mman.allocateAligned(32L, 1L);
        Assertions.assertThat((long)mman.usedMemory()).isGreaterThanOrEqualTo(32L);
        Assertions.assertThat((long)mman.availableMemory()).isLessThanOrEqualTo(8160L);
        mman.allocateAligned(32L, 1L);
        Assertions.assertThat((long)mman.usedMemory()).isGreaterThanOrEqualTo(64L);
        Assertions.assertThat((long)mman.availableMemory()).isLessThanOrEqualTo(8128L);
        mman.allocateAligned(1L, 1L);
        mman.allocateAligned(32L, 16L);
        Assertions.assertThat((long)mman.usedMemory()).isGreaterThanOrEqualTo(97L);
        Assertions.assertThat((long)mman.availableMemory()).isLessThanOrEqualTo(8095L);
    }

    @Test
    void trackMemoryAllocations() {
        LocalMemoryTracker memoryTracker = new LocalMemoryTracker();
        GrabAllocator allocator = (GrabAllocator)MemoryAllocator.createAllocator((long)ByteUnit.MebiByte.toBytes(2L), (MemoryTracker)memoryTracker);
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)memoryTracker.usedNativeMemory());
        allocator.allocateAligned(ByteUnit.mebiBytes((long)1L), 1L);
        org.junit.jupiter.api.Assertions.assertEquals((long)ByteUnit.mebiBytes((long)1L), (long)memoryTracker.usedNativeMemory());
        allocator.close();
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)memoryTracker.usedNativeMemory());
    }

    @Test
    void allAllocatedMemoryMustBeAccessibleForAllAlignments() throws Exception {
        int k512 = (int)ByteUnit.kibiBytes((long)512L);
        int maxAlign = 2048;
        for (int align = 1; align <= maxAlign; align += 8) {
            for (int alloc = 8192; alloc <= k512; alloc += 8192) {
                this.createAllocator(ByteUnit.MebiByte.toBytes(2L));
                long addr = this.allocator.allocateAligned((long)alloc, (long)align);
                int i = 0;
                try {
                    UnsafeUtil.getLong((long)(addr + (long)i));
                    i = alloc - 8;
                    UnsafeUtil.getLong((long)(addr + (long)i));
                    continue;
                }
                catch (Throwable e) {
                    throw new Exception(String.format("Access failed at offset %s (%x) into allocated address %s (%x) of size %s (align %s).", i, i, addr, addr, alloc, align), e);
                }
            }
        }
    }

    @Test
    void bufferCannotBeAlignedOutsideAllocatedSlot() {
        MemoryAllocator mman = this.createAllocator(8192L);
        long address = mman.allocateAligned(8192L, 8191L);
        Assertions.assertThat((long)address).isNotEqualTo(0L);
        UnsafeUtil.getLong((long)address);
        UnsafeUtil.getLong((long)(address + 8192L - 8L));
    }

    private void closeAllocator() {
        if (this.allocator != null) {
            this.allocator.close();
            this.allocator = null;
        }
    }

    private MemoryAllocator createAllocator(long expectedMaxMemory) {
        this.closeAllocator();
        this.allocator = MemoryAllocator.createAllocator((long)expectedMaxMemory, (MemoryTracker)new LocalMemoryTracker());
        return this.allocator;
    }
}

